#include "Date.h"
#include <assert.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include "Exception.h"

Date::Date( unsigned int year, Month_e month, unsigned int day ) :
	_year( year ),
	_month( month ),
	_dayOfMonth( day )
{
	assert( _dayOfMonth <= daysInMonth() );
    if ( _dayOfMonth > daysInMonth() ) {
        throw Exception(_T("Bad date."));
    }
}


Date::~Date()
{

}

Date
Date::current()
{
	time_t tms = time( 0 );
	struct tm* ts = localtime( &tms ); 
	
	return Date( ts->tm_year + 1900, (Month_e)(ts->tm_mon + 1), ts->tm_mday );
}

Date::Date( const Date& other )
{
	*this = other;
}

Date& Date::operator=( const Date& other )
{
	_year = other._year;
	_month = other._month;
	_dayOfMonth = other._dayOfMonth;

	return *this;
}

unsigned int
Date::dayOfMonth() const
{
	return _dayOfMonth;
}

Date::Month_e
Date::month() const
{
	return _month;
}

unsigned int
Date::year() const
{
	return _year;
}

Date::Day_e Date::dayOfWeek()
{
	return dayOfWeek( _year, _month, _dayOfMonth );
}

Date::Day_e Date::dayOfWeek( int year, int month, int day )
{
	int a = ( 14 - month ) / 12;
	int y = year - a;
	int m = month + 12 * a - 2;
	int d = ( day + y + y / 4 - y / 100 + y / 400 + ( 31 * m ) / 12 ) % 7;

	return (Day_e)d;
}

bool
Date::isLeapYear( unsigned int year )
{
	if ( year % 4 == 0 && ( year % 100 != 0 || year % 400 == 0 ) ) {
		return true;
	} 

	return false;
}	

bool 
Date::isLeapYear()
{
	return isLeapYear( _year );
}

void
Date::increment()
{
	_dayOfMonth++;

	if ( _dayOfMonth > daysInMonth() ) {
		_dayOfMonth = 1;
		_month = (Month_e)((int)_month + 1);
		if ( _month > 12 ) {
			_year++;
			_month = January;
		}
	}
}

void
Date::decrement()
{
	_dayOfMonth--;

	if ( _dayOfMonth == 0 ) {
		_month = (Month_e)((int)_month - 1 );
		if ( (int)_month == 0 ) {
			_month = (Month_e)12;
			_year--;
		}

		_dayOfMonth = daysInMonth();
	}
}

bool
Date::operator==( const Date& other ) {
	return _year == other._year && _month == other._month && _dayOfMonth ==
		other._dayOfMonth;
}

bool
Date::operator<( const Date& other )
{
    return _year < other._year || 
        ( _year == other._year && ( _month < other._month || 
            ( _month == other._month && _dayOfMonth < other._dayOfMonth ) ) );
}

unsigned int
Date::daysInMonth( Month_e month, unsigned int year ) 
{
	static int array[] = { -1, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 }; 

	assert( month > 0 && month <= 12 );

	if ( month == 2 && isLeapYear( year ) ) {
		return array[month] + 1;
	} else {
		return array[month];
	}
}

unsigned int 
Date::daysInMonth()
{
	return daysInMonth( _month, _year );
}

unsigned int
Date::daysInMonth( Month_e month )
{
    return daysInMonth( month, _year );
}

const char*
Date::monthString()
{
	return monthString( _month );
}

const char*
Date::monthString( Month_e month )
{
	static const char* array[] = {
		"January", "February", "March", "April", "May", "June", "July", 
        "August", "September", "October", "November", "December" };

	assert( month >= 1 && month <= 12 );
	return array[month-1];
}

const char*
Date::monthStringAbb( Month_e month )
{
	static const char* array[] = {
		"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sept", 
        "Oct", "Nov", "Dec" };

	assert( month >= 1 && month <= 12 );
	return array[month-1];
}

const char*
Date::dayOfWeekString( Day_e day )
{
	static const char* array[] = {
		"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", 
        "Saturday" };

	assert( day >= 0 && day <= 6 );
	return array[day];
}

const char*
Date::dayOfWeekString()
{
    return dayOfWeekString( dayOfWeek() );
}


bool 
Date::format( char* buffer, int* bufferLen )
{
    int len = 0;
    // Tuesday, December 26, 1999
    const char* dayOfWeekStr = dayOfWeekString();
    const char* monthStr = monthString();
    
    len = strlen( dayOfWeekStr );
    len += 2;
    len += strlen( monthStr );
    len += 1;
    if ( _dayOfMonth <= 9 ) {
        len += 1;
    } else if ( _dayOfMonth <= 99 ) {
        len += 2;
    } else {
        assert( 0 );
        throw Exception(_T("Bad date."));
    }
    len += 2;
    if ( _year <= 999 ) {
        len += 3;
    } else if ( _year <= 9999 ) {
        len += 4;
    } else {
        assert( 0 );
        throw Exception( _T("Bad date.") );
    }

    len += 1; // null character.

    if ( *bufferLen < len ) {
        *bufferLen = len;
        return false;
    }

    *bufferLen = len;
    sprintf( buffer, "%s, %s %d, %d", dayOfWeekStr, monthStr, _dayOfMonth,
        _year );

    return true;
    

}

unsigned int
Date::dayOfYear()
{
    unsigned int days = 0;

    for( Month_e i = January; i < _month; i = (Month_e)((int)i + 1 )) {
        days += daysInMonth( _month );
    }

    days += _dayOfMonth;

    return days;
}

int
Date::numMonths()
{
    return 12;
}

Date::Month_e operator++( Date::Month_e& month )
{
    // needs to allow Date::December + 1 for clean for() loops
    assert( month >= Date::January && month <= Date::December );
    month = (Date::Month_e)((int)month + 1 );
    return month;
}

unsigned int 
Date::daysSinceEpoch() const
{
    // ??? is this right
	int a = ( 14 - _month ) / 12;
	int y = _year - a;
	int m = _month + 12 * a - 2;
	int d = ( _dayOfMonth + y + y / 4 - y / 100 + y / 400 + ( 31 * m ) / 12 );
    return d;
}

int
Date::operator-( const Date& other )  
{
    return (int)other.daysSinceEpoch() - (int)daysSinceEpoch();
}

Date&
Date::operator+=( int days )
{
    // slow but sure way for now

    if ( days >= 0 ) {
        for( int i = 0; i < days; ++i ) {
            increment();
        }
    } else {
        for ( int i = 0; i < -days; ++i ) {
            decrement();
        }
    }

    return *this;
}

Date& 
Date::operator-=( int days )
{
    return operator+=( -days );
}

Date
Date::operator+( int days )
{
    Date ret(*this);

    ret += days;

    return ret;
}

Date
Date::operator-( int days )
{
    return operator+( -days );
}

Date::Month_e
Date::nextMonth( Date::Month_e month )
{
    assert( month >= January && month <= December );
    if ( month == December ) {
        month = January;
    } else {
        ++month;
    }

    return month;
}

