// DibImage.h: interface for the DibImage class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_DIBIMAGE_H__5659EB81_0A90_11D4_8267_00E0291FB8C6__INCLUDED_)
#define AFX_DIBIMAGE_H__5659EB81_0A90_11D4_8267_00E0291FB8C6__INCLUDED_

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <windows.h>
#include <tchar.h>
#include "SmartRef.h"

/**
 * @ingroup Graphics
 * The Canvas class provides generic access to the dimensions and pixels of an image.
 */
class Canvas
{
public:
    Canvas();
    virtual ~Canvas();

    virtual void setPixel( int x, int y, unsigned colour );
    virtual unsigned getPixel( int x, int y );
    virtual int height()=0;
    virtual int width()=0;
};

struct DIBSTRUCT
{
	PBYTE * ppRow;
	HBITMAP hBitmap ;
	BYTE *pBits;
	DIBSECTION ds;
	int iRShift[3];
	int iLShift[3];
	BITMAPINFO* pbmi;
};

class Stream;

/**
 * @ingroup Graphics
 * The DibImage class encapsulates a 24 bit device independent bitmap (DIB). A
 * dib is much faster to work with than a windows Bitmap, because it provides
 * direct access to the pixel data. We also provide ways of loading and saving
 * DIBs to jpeg and png files. (This requires linking with libjpeg, libpng, and
 * zlib) We also provide a stretching function that produces nice results, and
 * a way to cache the stretched images so they do not have to be recalculated
 * each time they are drawn.
 */
class DibImage : public Canvas
{
public:
	BYTE* PixelPtr(int x, int y);
	virtual void setPixel(int x, int y, unsigned color);
	virtual unsigned getPixel(int x, int y);
    DibImage* createScaled( int x, int y );
	bool display(HDC dc, int x, int y, int cxClient, int cyClient, int xsrc,
            int ysrc);
    bool stretch(HDC dc, RECT* src, RECT* dest );  
	bool Delete();
    bool createFromBitmap( HBITMAP hBitmap );
	bool Create(int cx, int cy, int cBits, int cColors);
	bool Overlay(unsigned int x, unsigned int y, DibImage* pImage, RECT* source, double transparency);
	bool GetRect(RECT* rect);
	bool GetFromDC(HDC dc, RECT* source);
    void save( const char* fileName, const char* type);
	DibImage();
	virtual ~DibImage();

    static DibImage* Load( const TCHAR* fileName );
    static DibImage* LoadPng( Stream* stream );
    virtual int width();
    virtual int height();

    HRGN createRegion(COLORREF cTransparentColor, COLORREF cTolerance,
            XFORM* matrix);

    void cacheStretch(HDC dc, RECT* src, RECT* dest, int maxCached);
    void clearCache();
    BITMAPINFO* getBitmapInfo();
    unsigned char* getRawBits();

    HBITMAP createBitmap( HDC dc );

protected:
	bool SetPixel(int x, int y, DWORD dwPixel);
	DWORD GetPixel(int x, int y);
	int MaskToLShift(DWORD dwMask);
	int MaskToRShift(DWORD dwMask);
	bool CreateFromInfo(BITMAPINFO *pbmi);
	bool _bCreated;
	DIBSTRUCT _dib;

private:
    void scaleNearestNeighbour( DibImage* dest );
    void scaleBilinear( DibImage* dest );

    struct CacheEntry {
        RECT src;
        SIZE size;
        DibImage* dib;
    }** _cache;
    int _cacheSize;
    void freeCacheEntry( CacheEntry*& entry );
};

typedef SmartRef<DibImage> DibImageRef;

class Stream;

bool
SavePng( Canvas* canvas, const char* filename );

bool
SavePng( Canvas* canvas, Stream* stream );

bool
SaveJpeg( Canvas* canvas, const TCHAR* fileName );


#endif // !defined(AFX_DIBIMAGE_H__5659EB81_0A90_11D4_8267_00E0291FB8C6__INCLUDED_)
