// FileEnumerator.cpp: implementation of the FileEnumerator class.
//
//////////////////////////////////////////////////////////////////////

#include "FileEnumerator.h"
#include "debug.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


using namespace std;

namespace FileSystem
{

FILETIME
File::writeTime()
{
    return _data.ftLastWriteTime;
}

File::File(const WIN32_FIND_DATA* pData, const std::_tstring& path)
{
	_data = *pData;
	_path = path;
    _good = false;
}

File::File()
{
    _good = false;
    memset( &_data, 0, sizeof(_data));
}

File::File(const _TCHAR* fullPath)
{
	_good = false;
	HANDLE handle = ::FindFirstFile(fullPath, &_data);
	if ( handle != INVALID_HANDLE_VALUE ) {
		_good = true;
		FindClose(handle);
	}

	_tstring temppath = fullPath;
	_tstring::size_type pos = temppath.rfind('\\');
	if ( pos != _tstring::npos) {
		_path = temppath.substr(0, pos);
	}
}

File::File(const File& right)
{
	*this = right;
}

File::~File()
{

}

const File& File::operator=(const File& right)
{
    if ( &right == this ) {
        return *this;
    }
    _data = right._data;
    _path = right._path;
    _good = right._good;
    return *this;
}

std::_tstring File::name()
{
	return _data.cFileName;
}

std::_tstring File::path()
{
	return _path;
}

bool File::exists()
{
	return _good;
}

bool File::isFolder()
{
	return (_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
}

bool File::isReadOnly()
{
	return (_data.dwFileAttributes & FILE_ATTRIBUTE_READONLY) != 0;
}

void File::makeReadOnly(bool readOnly)
{
	if (isReadOnly() == readOnly ) {
		return;
	}

	int attr = readOnly ? FILE_ATTRIBUTE_READONLY : FILE_ATTRIBUTE_NORMAL;
	SetFileAttributes((path() + name()).c_str(), attr);

	if ( readOnly ) {
		_data.dwFileAttributes = _data.dwFileAttributes & FILE_ATTRIBUTE_READONLY;
	} else {
		_data.dwFileAttributes = _data.dwFileAttributes & ~FILE_ATTRIBUTE_READONLY;
	}
}

void File::erase()
{
	makeReadOnly(true);
	DeleteFile( (path() + name()).c_str() );
	_good = false;
}

std::_tstring File::extension()
{
	_tstring ext;
	_tstring strName = name();
	_tstring::size_type pos = strName.rfind('.');
	if ( pos != _tstring::npos ) {
		ext = strName.substr(pos + 1, strName.length() - (pos + 1));
	}

	return ext;
}

FileEnumerator::FileEnumerator(const _TCHAR* fullPath)
{
	memset(&_data, 0, sizeof(_data));
	_handle = ::FindFirstFile((_tstring(fullPath) + _T("*.*")).c_str(), &_data);
	_found = _handle != INVALID_HANDLE_VALUE;
	_path = fullPath;
}

FileEnumerator::~FileEnumerator()
{
	if(_handle != INVALID_HANDLE_VALUE) {
		FindClose(_handle);
	}
    _handle = 0;
}

bool FileEnumerator::more()
{
	return _found;
}

File FileEnumerator::next()
{
	File file;

	//if we got a file last time, 
	if(_found) {
		file = File(&_data, _path);
	} else {
		throw Exception();
	}
	
	_found = ::FindNextFile(_handle, &_data) == TRUE;
	return file;
}

//lint -e429


RecursiveFileEnumerator::RecursiveFileEnumerator(const _TCHAR* fullPath)
{
	// create the root enumerator
	// put it on the stack.
	_dirStack.push(new FileEnumerator(fullPath) );
}

RecursiveFileEnumerator::~RecursiveFileEnumerator()
{
	// while the stack is not empty,
	while ( ! _dirStack.empty() ) {
		// pop off an enumerator.
		FileEnumerator* dir = _dirStack.top();
		_dirStack.pop();

		// delete it.
		delete dir;
	}
}

FileEnumerator* RecursiveFileEnumerator::getEnum()
{
	FileEnumerator* dir = NULL;

	// while the stack is not empty,
	while ( ! _dirStack.empty() ) {

		// get the top of the stack.
		dir =  _dirStack.top();

		// break if that enumerator is good.
		if ( dir->more() ) {
			break;
		}

		// emerator is empty; pop it off the stack and delete it.
		_dirStack.pop();
		delete dir;
		dir = 0;
	}

	// return the enumerator that was found to be good, or NULL.
	return dir;
}

bool RecursiveFileEnumerator::more() 
{
	FileEnumerator* dir = getEnum();

	if ( dir ) {
		return dir->more();
	} else {
		return false;
	}
}

//lint -sem(push,custodial(0))
File RecursiveFileEnumerator::next()
{
	// get the enumerator.
	FileEnumerator* dir = getEnum();

	// get the next file.
	File file = dir->next();

	// if the next file is file, then return it.
	_tstring name = file.name();
	if ( file.isFolder() && name != _T(".") && name != _T("..")) {
		// the next file is a folder.
		// create an enumerator for it and push it to the top of the stack.
		_tstring path = file.path() + file.name() + _T("\\");
		dir = new FileEnumerator( path.c_str() );
		_dirStack.push(dir);

		// recurse. 
        // -e{429}
		return next();

	} else {
		// not a folder.
		return file;
	}
}


};
