#include <assert.h>
#include <windowsx.h>
#include "ImageWindow.h"
#include <stdio.h>
#include "DibImage.h"
#include "debug.h" // must be last

/**
 * Construct an iamge window.
 */
ImageWindow::ImageWindow(HINSTANCE hInstance) :
    ScrollWindow( hInstance ),
    _navWindow( hInstance )
{
	_hInstance = hInstance;
	_pszClassName = TEXT("ImageWindow");
	_pszTitle = TEXT("Image Window");

    _dwStyle |= WS_VISIBLE | WS_CHILD | WS_CLIPCHILDREN; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL;
    _WndClass.hCursor = (HCURSOR)LoadCursor(NULL, IDC_CROSS);
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
    _hatchBrush = CreateHatchBrush( HS_DIAGCROSS, RGB(200,200,200) );
    _zoom = 1.5;
    _firstSize = true;
    _zoomFullPage = false;
}


ImageWindow::~ImageWindow()
{
    DeleteObject( _hatchBrush );
}


/**
 * Sets the image to be displayed by the image window.
 */ 
void
ImageWindow::setImage( DibImageRef image ) 
{
    _image = image;
    //scrollTo(0,0);

    SetFocus(hwnd);

    if ( _image ) {
        _image->clearCache();
        if ( !_zoomFullPage ) {
            setZoom(_zoom);
        } else {
            zoomFullPage();
        }
        InvalidateRect( hwnd, NULL, FALSE );
    }

    _navWindow.setImage(_image);
}

/**
 * Sets the image to be displayed by the image window.
 Inefficient. Use the DibImageRef version instead.
 */
void 
ImageWindow::setImage(Bitmap* bmp)
{
    DibImageRef dib = new DibImage();
    dib->createFromBitmap( *bmp );
    setImage( dib );
}

/**
 * Scales the image so that the entire image is visible in the control. This
 * setting is sticky -- if the control is resized, the image will be resized to
 * fit again. This mode can be exited by calling one of the other zoom
 * functions.
 */
void
ImageWindow::zoomFullPage()
{
    _zoomFullPage = true;
    if ( _image == 0 ) {
        return;
    }

    RECT rect = getClientRect();
    int cx = rect.right - rect.left;
    int cy = rect.bottom - rect.top;
    int imWidth = _image->width();
    int imHeight = _image->height();

    // calculate zoom factor.
    double scaleX = (double)(cx) / imWidth;
    double scaleY = (double)(cy) / imHeight;
    int newHeight, newWidth;

    double zoom;

    if ( (double)imHeight * scaleX > cy ) {
        // use scaleY
        newWidth = (int)(imWidth * scaleY);
        newHeight = (int)(imHeight * scaleY);
        zoom = scaleY;
    } else {
        // use scaleX
        newWidth = (int)(imWidth * scaleX);
        newHeight = (int)(imHeight * scaleX);
        zoom = scaleX;
    }

    setZoom(zoom);
}

void 
ImageWindow::getViewRect(RECT* src, RECT* viewPort)
{
	if ( !_image ) {
		src->left = src->top = 0;
		src->right = src->bottom = 1;
		viewPort->left = viewPort->top = 0;
		viewPort->bottom = viewPort->right = 1;
	}
    RECT rect = getClientRect();
    int offsetX=0, offsetY=0;
    if ( _zoom * _image->width() < rect.right ) {
        offsetX = (int)(rect.right/2 - _zoom * _image->width()/2);
    }

    if ( _zoom * _image->height() < rect.bottom ) {
        offsetY = (int)(rect.bottom/2 - _zoom * _image->height()/2);
    }

    viewPort->left = offsetX;
    viewPort->top = offsetY;
    viewPort->right = (int)(_zoom * _image->width()) + offsetX;
    viewPort->bottom = (int)(_zoom * _image->height()) + offsetY;

    if ( offsetX == 0 ) {
        viewPort->right = rect.right;
    }

    if ( offsetY == 0 ) {
        viewPort->bottom = rect.bottom;
    }

    if ( offsetX ) {
        src->left  = 0;
        src->right = _image->width();
    } else {
        src->left = (int)(getScrollX() / _zoom);
        src->right = src->left + (int)(rect.right / _zoom);
    }

    if ( offsetY ) {
        src->top  = 0;
        src->bottom = _image->height();
    } else {
        src->top = (int)(getScrollY() / _zoom);
        src->bottom = src->top + (int)(rect.bottom / _zoom);
    }
}

/**
  Set the zoom factor for the image. 1.0 means no zoom. It is probably a bad
  idea to use 0.0.
 */
void
ImageWindow::setZoom(double zoom)
{
    _zoom = zoom;
    if ( _image ) {
        setScrollBars((int)(_image->width()*_zoom), 
            (int)(_image->height()*_zoom), 1, 1 );
        InvalidateRect( hwnd, NULL, FALSE );
    }
}

/**
 * Create the ImageWindow, specifying its position and size.
 */
void
ImageWindow::create(HWND hParent, int x, int y, int cx, int cy)
{
    if ( NULL == Window::Create( x, y, cx, cy, hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

LRESULT 
ImageWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        case WM_MOUSEWHEEL: {
            if ( _image ) {
                int zDelta = GET_WHEEL_DELTA_WPARAM(wParam);
                double zoom = _zoom + (double)zDelta / 120 * 0.25 * _zoom;
                if ( _zoom > 4 ) {
                    zoom = 4;
                } else if ( _zoom < 0.01 ) {
                    zoom = 0.01;
                }
                setZoom(zoom);
                _zoomFullPage = false;
                return 0;
            }
        }

        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
        case WM_ERASEBKGND:
            return 1;
    }
    
    *pbProcessed = FALSE;
    return ScrollWindow::WindowProc(hwnd,msg,wParam,lParam,pbProcessed);
}

UINT
ImageWindow::onNcCalcSize(HWND hwnd, BOOL fCalcValidRects, NCCALCSIZE_PARAMS * lpcsp)
{
    if ( fCalcValidRects == FALSE ) {
        RECT* rect = (RECT*)lpcsp;
        rect->top += 64;
        rect->bottom -= 10;
        rect->left += 10;
        rect->right -= 10;
    } else {
        assert(0);
    }
    return 0;
}

void
ImageWindow::onClose(HWND)
{

}

void 
ImageWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    if ( hwndCtl == _navWindow.hwnd ) {
        switch(id) {
            case NavWindow::IDC_MOVE: {
                RECT displayRect = _navWindow.getDisplayArea();
                scrollTo( (int)(_zoom*displayRect.left), (int)(_zoom*displayRect.top) );
                break;
            }
            case NavWindow::IDC_FULL_SCREEN: {
                zoomFullPage();
                break;
            }
            case NavWindow::IDC_ZOOM_IN: {
                double zoom = _zoom * 1.25;
                if ( _zoom > 4 ) {
                    zoom = 4;
                }
                setZoom(zoom);
                _zoomFullPage = false;
                break;
            }
            case NavWindow::IDC_ZOOM_OUT: {
                double zoom = _zoom * 0.75;
                if ( _zoom < 0.01 ) {
                    zoom = 0.01;
                }
                setZoom(zoom);
                _zoomFullPage = false;
                break;
            }
            case NavWindow::IDC_ZOOM_1_1: {
                setZoom(1.0);
                _zoomFullPage = false;
                break;
            }
        }
    }
}

BOOL 
ImageWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    _navWindow.create(hwnd, lpCreateStruct->cx-200, lpCreateStruct->cy-150, 200, 150 );
    return TRUE;
}

void
ImageWindow::onDestroy(HWND hwnd)
{

}


void 
ImageWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
ImageWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{

}

void 
ImageWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
ImageWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{

}

/**
 * Returns the current scale factor.
 */
double
ImageWindow::getScale()
{
    return _zoom;
}

/**
 * Returns the x coordinate of the pixel in the original image that is
 * currently being displayed at the upper left corner, taking into account user
 * scrolling and zooming..
 */
int
ImageWindow::getOffsetX()
{
    if ( !_image ) {
        return 0;
    }

    RECT rect = getClientRect();

    if ( _zoom * _image->width() < rect.right ) {
        return (int)(rect.right/2 - _zoom * _image->width()/2);
    }

    return 0;
}

/**
 * Returns the y coordinate of the pixel in the original image that is
 * currently being displayed at the upper left corner, taking into account user
 * scrolling and zooming..
 */
int
ImageWindow::getOffsetY()
{
    if ( !_image ) {
        return 0;
    }

    RECT rect = getClientRect();

    if ( _zoom * _image->height() < rect.bottom ) {
        return (int)(rect.bottom/2 - _zoom * _image->height()/2);
    }

    return 0;
}

void 
ImageWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, 
        bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC dc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        dc = PaintStruct.hdc;
    } else {
        dc = (HDC)wParam;
    }

    RECT rect = getClientRect();
    if ( _image == 0 ) {
        FillRect( dc, &rect, _hatchBrush);
        return;
    }

    RECT src;
    RECT dest;
    getViewRect( &src, &dest );

    if ( dest.left > 0 || dest.top > 0 ) {
        //HRGN region = CreateRectRgnIndirect(&dest);
        //ExtSelectClipRgn( dc, region, RGN_DIFF );
        FillRectAround( dc, &dest, &rect, _hatchBrush );
        //ExtSelectClipRgn( dc, NULL, RGN_COPY );
        //DeleteObject(region);
    }

    if ( _zoom < 1.0 ) {
        _image->cacheStretch(dc, &src, &dest,2);
    } else {
        _image->stretch(dc, &src, &dest);
    }

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}

void
ImageWindow::onScroll(int x, int y)
{
    RECT rect = getClientRect();

    RECT displayRect;
    displayRect.left = (int)(x/_zoom);
    displayRect.right = displayRect.left + (int)(rect.right/_zoom);
    displayRect.top = (int)(y/_zoom);
    displayRect.bottom = displayRect.top + (int)(rect.bottom/_zoom);
    _navWindow.setDisplayArea( &displayRect );
}


void 
ImageWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    ScrollWindow::onSize( hwnd, state, cx, cy );
    RECT rect = getClientRect();
    //_navWindow.moveWindow(rect.right - 200, rect.bottom - 150, 200, 150); 
    _navWindow.fixIfOffscreen();
    onScroll(getScrollX(), getScrollY());

    if ( _zoomFullPage ) {
        zoomFullPage();
    }
}
