#include "List.h"
#include <fstream>
#include "LogFile.h"
#include "string.h"
#include <assert.h>
#include <stdarg.h>

/**
 * Construct a Logfile object.
 * @param destination The filename to write to.
 * @param backtrace The number of lines to store. As newer lines are added,
 * older ones are removed. The default is 1000 lines.
 */
LogFile::LogFile(const char* destination, int backtrace)
{
	construct();
	setLength(backtrace);
	setDestination(destination);
}

LogFile::LogFile()
{
	construct();
}

void LogFile::construct()
{
	_destination = 0;
	_enabled = false;
	_backtrace = 1000;
    _size = 0;
}

/**
 * The destructor flushes the lines to the file and frees all memory.
 */
LogFile::~LogFile()
{
	flush();
    char* str = _strings.first();
    while( str ) {
        free( str );
        str = _strings.next();
    }
	free(_destination);
}

/**
 * If the logfile is disabled, then printing will do nothing.
 */
void LogFile::enable()
{
	_enabled = true;
}

void LogFile::disable()
{
	_enabled = false;
}

/**
 * Implements a printf() style function for the logfile.
 */
void
LogFile::printf(const char* fmt, ... )
{
    char buffer[1024];
    va_list args;

    va_start( args, fmt );
    vsprintf( &buffer[0], fmt, args );
    va_end( args );
    addLine( buffer );
}

/**
 * Sets the number of lines of backtrace.
 */
void LogFile::setLength(int lines)
{
	assert(lines >= 0);
	_backtrace = lines;
}

/**
 * Sets the filename to which to log.
 */
void LogFile::setDestination(const char* file)
{
    char buffer[1024];
    free( _destination ) ;
    _destination = _strdup( file );
    _enabled = true;

	// attempt to read it in.
	FILE* openFile = fopen(_destination, "rt");
	if ( !openFile ) {
		return;
	}

    while( fgets( buffer, sizeof(buffer), openFile ) ) {
        addLine( buffer );
    }

    fclose( openFile );
}

/**
 * Appends a line to the log. As new lines are added, older ones may be
 * deleted.
 */
void LogFile::addLine(const char* str)
{
	if(!_enabled) {
		return;
	}

	if(_size >= _backtrace) {
        char* first = _strings.first();
        free( first );
        _strings.remove();
	}

	_strings.addTail( _strdup( str ) );
}

/**
 * Stores the lines to the disk file.
 */
void LogFile::flush()
{
	if(!_enabled) {
		return;
	}

    if ( !_destination ) {
        return;
    }

    FILE* file = fopen( _destination, "wt" );
    char* str = _strings.first();
    while( str ) {
        fprintf(file, "%s", str);
        str = _strings.next();
    }
    fclose(file);
}

