#include <assert.h>
#include <windowsx.h>
#include "NavWindow.h"
#include "DibImage.h"
#include "win95.h"
#include "utils-resource.h"
#include "debug.h" // must be last

NavWindow::NavWindow(HINSTANCE hInstance) :
    _toolBar( hInstance )
{
	_hInstance = hInstance;
	_pszClassName = TEXT("NavWindow");
	_pszTitle = TEXT("Nav Window");

    _dwStyle |= WS_CHILD | WS_VISIBLE | WS_CAPTION | WS_SIZEBOX | WS_CLIPCHILDREN | WS_CLIPSIBLINGS; 
    _dwExtendedStyle |= WS_EX_TOOLWINDOW;
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = (HCURSOR)LoadCursor(NULL, IDC_ARROW);
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;

    _scale = 1.0;
    _toolBarHeight = 16;
    memset( &_displayRect, 0, sizeof(_displayRect));
    memset( &_dragRect, 0, sizeof(_dragRect));
}

NavWindow::~NavWindow()
{

}

/**
 * Create the child window in the given position and size.
 */
void
NavWindow::create(HWND hParent, int x, int y, int cx, int cy)
{
    _hParent = hParent;
    if ( NULL == Window::Create( x,y,cx,cy, hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

LRESULT 
NavWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
            #if 0
        case WM_MOVING: // doesn't work
            RECT* rect = (RECT*)lParam;
            RECT parent;
            GetWindowRect( _hwndParent, &parent );
            SnapWindowRect( rect, &parent, 5, false );
            break;
            #endif
    }
    
    *pbProcessed = FALSE;
    return 0;
}

/**
 * If any part of the window is outside of its parent, we shift the window
 * until it is visiable again.
 */
void 
NavWindow::fixIfOffscreen()
{
    WindowFunctions wparent;
    wparent.hwnd = _hwndParent;
    RECT rect = wparent.getChildRect(hwnd);;
    RECT parent = wparent.getClientRect();
    SnapWindowRect( &rect, &parent, 5, true );
    moveWindow(&rect);
}


void
NavWindow::onClose(HWND)
{

}


void 
NavWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    if ( hwndCtl == _toolBar.hwnd ) {
        switch(id) {
            case IDC_FULL_SCREEN:
            case IDC_ZOOM_IN:
            case IDC_ZOOM_OUT:
            case IDC_ZOOM_1_1:
            SendMessage( _hParent, WM_COMMAND, MAKEWPARAM( id, BN_CLICKED ),
                    (LPARAM)hwnd );
                break;
            default:
                break;
        }
    }
}


BOOL 
NavWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    _toolBar.create(hwnd);
    _toolBar.addButton(IDC_FULL_SCREEN, IDB_ZOOMPAGE_OUT, IDB_ZOOMPAGE_OVER, IDB_ZOOMPAGE_DOWN, 0);
    _toolBar.addButton(IDC_ZOOM_IN, IDB_ZOOMIN_OUT, IDB_ZOOMIN_OVER, IDB_ZOOMIN_DOWN, 0);
    _toolBar.addButton(IDC_ZOOM_OUT, IDB_ZOOMOUT_OUT, IDB_ZOOMOUT_OVER, IDB_ZOOMOUT_DOWN, 0 );
    _toolBar.addButton(IDC_ZOOM_1_1, IDB_ZOOM11_OUT, IDB_ZOOM11_OVER, IDB_ZOOM11_DOWN, 0);

    return TRUE;
}

void
NavWindow::onDestroy(HWND hwnd)
{

}


void 
NavWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
NavWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    onMouseMove(hwnd, x, y, keyFlags);

}

void 
NavWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
NavWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( !_image ) {
        return;
    }

    if ( keyFlags & MK_LBUTTON ) {
        // calculate position in the real image.
        // center the display rect at that position.
        int xpos = (int)((x - _destRect.left)/_scale) - (_displayRect.right-_displayRect.left)/2;
        int ypos = (int)((y - _destRect.top)/_scale) - (_displayRect.bottom-_displayRect.top)/2;

        // If we are overhanging an edge, snap it back.
        if ( xpos > _image->width() - (_displayRect.right-_displayRect.left)) {
            xpos = _image->width() - (_displayRect.right-_displayRect.left);
        }

        if ( ypos > _image->height() - (_displayRect.bottom-_displayRect.top)) {
            ypos = _image->height() - (_displayRect.bottom-_displayRect.top);
        }

        if ( xpos < 0 ) {
            xpos = 0;
        }

        if ( ypos < 0 ) {
            ypos = 0;
        }


        RECT rect;
        rect.left = xpos;
        rect.top = ypos;
        rect.right = rect.left + _displayRect.right - _displayRect.left;
        rect.bottom = rect.top + _displayRect.bottom - _displayRect.top;

        // move.
        if ( 0 != memcmp( &rect, &_displayRect, sizeof(rect) )) {
            setDisplayArea( &rect );
            SendMessage( _hParent, WM_COMMAND, MAKEWPARAM( IDC_MOVE, BN_CLICKED ),
                    (LPARAM)hwnd );
        }
    }
}

void
NavWindow::calcLayout()
{
    // Calculates _scale, _destRect.
    if ( _image == NULL ) {
        memset( &_destRect, 0, sizeof(_destRect));
        _scale = 1.0;
        return;
    }

    RECT rect = getClientRect();
    rect.bottom -= _toolBarHeight;
    int cx = rect.right - rect.left;
    int cy = rect.bottom - rect.top;
    int imWidth = _image->width();
    int imHeight = _image->height();

    // calculate zoom factor.
    double scaleX = (double)(cx) / imWidth;
    double scaleY = (double)(cy) / imHeight;
    int newHeight, newWidth;
    int offsetX, offsetY;

    if ( (double)imHeight * scaleX > cy ) {
        // use scaleY
        newWidth = (int)(imWidth * scaleY);
        newHeight = (int)(imHeight * scaleY);
        _scale = scaleY;
    } else {
        // use scaleX
        newWidth = (int)(imWidth * scaleX);
        newHeight = (int)(imHeight * scaleX);
        _scale = scaleX;
    }

    offsetX = cx/2-newWidth/2;
    offsetY = cy/2-newHeight/2;

    _destRect.left = offsetX;
    _destRect.top = offsetY;
    _destRect.right = offsetX + newWidth;
    _destRect.bottom = offsetY + newHeight;
}


void 
NavWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, 
        bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    RECT rect = getClientRect();
    rect.bottom -= _toolBarHeight;

    if ( _image ) {

        // get image size.
        int imWidth, imHeight;
        int cx, cy;

        imWidth = _image->width();
        imHeight = _image->height();

        // get our size.
        cx = rect.right;
        cy = rect.bottom;

        RECT src;
        src.left = 0;
        src.top = 0;
        src.right = imWidth;
        src.bottom = imHeight;

        _image->cacheStretch(hdc, &src, &_destRect, 2);
        FillRectAround( hdc, &_destRect, &rect, (HBRUSH)GetStockObject(BLACK_BRUSH));

        BLENDFUNCTION blend;
        blend.BlendOp = AC_SRC_OVER;
        blend.BlendFlags = 0;
        blend.SourceConstantAlpha = 128;
        blend.AlphaFormat = 0;
        
        SafeAlphaBlend( hdc, _dragRect.left, _dragRect.top, _dragRect.right-_dragRect.left,
            _dragRect.bottom-_dragRect.top, _dragger, 0, 0, _dragger.width(), _dragger.height(),
            blend);

    } else {
        FillRect( hdc, &rect, (HBRUSH)GetStockObject(BLACK_BRUSH) );
    }

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
NavWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    RECT rect = getClientRect();

    _toolBar.moveWindow(0,rect.bottom-_toolBarHeight,rect.right,_toolBarHeight);
    placeDragger();
}

/**
 * Sets the image to display.
 */
void
NavWindow::setImage(DibImageRef image )
{
    _image = image;
    placeDragger();
    InvalidateRect(hwnd, NULL, FALSE );
}

/**
 * Returns a rectangle corresponding to the ordered display area in the
 * original image.
 */
RECT
NavWindow::getDisplayArea()
{
    return _displayRect;
}

/**
 * Sets the area that is being displayed in the original image. The thumbnail
 * will change to reflect this to the user.
 */
void 
NavWindow::setDisplayArea( RECT* rect )
{
    if ( 0 == memcmp( rect, &_displayRect, sizeof(*rect) )) {
        return;
    }
    _displayRect = *rect;
    placeDragger();
}

void
NavWindow::placeDragger()
{
    calcLayout();

    if ( _image == NULL ) {
        return;
    }

    int dwidth = _displayRect.right - _displayRect.left;
    int dheight = _displayRect.bottom - _displayRect.top;

    RECT dragRect;
    dragRect.left = (int)(_scale * _displayRect.left + _destRect.left);
    dragRect.top = (int)(_scale * _displayRect.top + _destRect.top);
    dragRect.right = (int)(_scale * _displayRect.right + _destRect.left);
    dragRect.bottom = (int)(_scale * _displayRect.bottom + _destRect.top);

    if ( _dragRect.right-_dragRect.left != dragRect.right - dragRect.left ) {
        _dragger.createCompatible(hwnd, dragRect.right-dragRect.left,
            dragRect.bottom-dragRect.top);
        RECT rect;
        rect.top = 0;
        rect.left = 0;
        rect.right = dragRect.right - dragRect.left;
        rect.bottom = dragRect.bottom - dragRect.top;
        Brush brush(RGB(0,0,255));
        FillRect( _dragger, &rect, brush);
    }

    InvalidateRect( hwnd, &_dragRect, FALSE );
    _dragRect = dragRect;
    InvalidateRect( hwnd, &_dragRect, FALSE );
}

