/*****************************************************************************
   SocketObj
      
   Object wrapper for socket.
 **************************************************************************/

#ifndef Socket_H
#define Socket_H

#include <tchar.h>
#include <tstring>
#include "Trace.h"
#ifdef LINUX
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>

typedef int SOCKET;
#else
#include <winsock2.h>
#endif

/**
 * Encapsulates a socket class. This works on both linux and win32 platforms.
 */
class Socket
{

 public:

  enum Type {
    STREAM,
    DGRAM
  };

  /**
   * Encapsulates an address to be passed to the socket. For example:
   *
   * - 192.168.1.3:1024
   *
   * - www.google.com:80
   */
  class Address {
  public:
    Address( const std::_tstring& dest );
    Address();
    Address( const Address& other );
    Address& operator=( const Address& other );
    bool good() const;
    std::_tstring toString() const;
  private:
    friend class Socket;
    sockaddr_in addr;
    bool _good;
  };

    /***********************************************************************
      Initializes a socket. Throws exception if it fails.
     ***********************************************************************/
    Socket(Type p = STREAM);
    ~Socket();

    static bool UnitTest();

    /***********************************************************************
       Returns true when read or accept will not block, or returns false
       after timeout.
      **********************************************************************/
    bool waitForData(int timeoutMS);

    /***********************************************************************
       Takes first listening connection off queue and returns a connected
       socket representing connection.
      **********************************************************************/
    Socket* accept();

    /**********************************************************************
        Performs a bind operation, binding the socket to the given port.
        Throws exception if socket could not be bound.
     *********************************************************************/
    void bind(int port);

    /*********************************************************************
        Connects to the given host and port or returns false.
     ********************************************************************/
    bool connect(const _TCHAR* hostport);
    bool connect(const _TCHAR* host, int port);

    /********************************************************************
	Performs listen operation on socket.
     *******************************************************************/
    void listen();

    /*********************************************************************
	Sends data to connected socket. Returns length of data actually
        recieved.
        Throws exception if not connected.
     *********************************************************************/  
    int recv( void* data, int length );
    int recvfrom( Address* pAddress, void* data, int length );

    /*********************************************************************
	Recieves data to connected socket. Returns length sent.
        Throws exception if not connected.
     *********************************************************************/  
    int send( void* data, int length );
    int sendto( const Address* address, void* data, int length );

    /*********************************************************************
	shuts down socket.
        throws no exceptions.
     *********************************************************************/  
    void shutdown();

    bool connected();

    std::_tstring getLastError();

 private:
    Socket( SOCKET desc );

    enum State {
      IDLE,
      CONNECTED
    } _state;

    SOCKET _socket;
    TraceLabel trace;
};

#endif // SocketObj_H
