#include "Stream.h"
#include <windows.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "WideChar.h"
#include "zlib.h"
#include <tchar.h>
#include "debug.h" // must be last

Stream::Stream()
{

}

Stream::~Stream()
{

}


unsigned 
Stream::read( void* buffer, int rec_size, int num_recs )
{
    assert(0);
    return 0;
}

unsigned 
Stream::write( void* buffer, int rec_size, int num_recs )
{
    assert(0);
    return 0;
}

_TCHAR* 
Stream::gets(_TCHAR*, int)
{
    assert(0);
    return 0;
}

bool 
Stream::eof()
{
    return true;
}

FileStream::FileStream()
{
    _handle = 0;
}

FileStream::~FileStream()
{    close();
}


bool 
FileStream::open( const _TCHAR* name, const _TCHAR* mode )
{
    _handle = _tfopen(name, mode);
    return _handle != 0;
}

void
FileStream::close()
{
    if ( _handle ) {
        fclose(_handle);
        _handle = 0;
    }
}

_TCHAR*
FileStream::gets(_TCHAR* val, int n)
{
    return _fgetts(val, n, _handle);
}

unsigned 
FileStream::read( void* buffer, int rec_size, int num_recs)
{
    return (unsigned)fread(buffer, rec_size, num_recs, _handle);
}


unsigned 
FileStream::write( void* buffer, int rec_size, int num_recs)
{
    return (unsigned)fwrite(buffer, rec_size, num_recs, _handle);
}

bool 
FileStream::eof()
{
    return feof(_handle) != 0;
}


TempFileStream::TempFileStream(const _TCHAR* friendlyName)
{
    _TCHAR buffer[MAX_PATH*4];
    _TCHAR fileBuffer[MAX_PATH*4];
    _handle = 0;

    GetTempPath(sizeof(buffer)/sizeof(*buffer), buffer);
    GetTempFileName(buffer, _T("kbl"), 0, fileBuffer);
    _file = _tcsdup(fileBuffer);
    _name = _tcsdup(friendlyName);
}

TempFileStream::~TempFileStream()
{
    close();
    DeleteFile(_file);
    free(_name);
    free(_file);
}

const TCHAR* 
TempFileStream::name()
{
    return _name;
}

const TCHAR*
TempFileStream::file()
{
    return _file;
}

bool
TempFileStream::open(const _TCHAR* mode)
{
    return FileStream::open(_file, mode);
}
/*
void
TempFileStream::close()
{
    FileStream::close();
}

unsigned 
TempFileStream::read( void* buffer, int rec_size, int num_recs)
{
    return FileStream::read(buffer, rec_size, num_recs); 
}

unsigned 
TempFileStream::write( void* buffer, int rec_size, int num_recs)
{
    return FileStream::write(buffer, rec_size, num_recs);
}

bool 
TempFileStream::eof()
{
    return FileStream::eof();
}

*/

Archive::Archive(Stream* stream)
{
    _stream = stream;
}

Archive::~Archive()
{

}

bool 
Archive::addFile( const _TCHAR* filename, const _TCHAR* friendlyName )
{
    bool error = false;
    Header header;
    _TCHAR buffer[4096];
    ZeroMemory(&header, sizeof(header));

    // open file
    FILE* file = _tfopen(filename, _T("rb"));
    if ( file == 0 ) {
        error = true;
        goto cleanup;
    }

    // seek to end
    if ( fseek( file, 0, SEEK_END ) != 0 ) {
        error = true;
        goto cleanup;
    }

    // get length.
    _tcsncpy(header.name, friendlyName, sizeof(header.name)/sizeof(*header.name));
    header.name[sizeof(header.name)/sizeof(*header.name)-1] = 0;
    header.size = (unsigned)ftell(file);
    header.checksum = checksum((const char*)&header, sizeof(header));
    
    // go to beginning
    if ( fseek(file, 0, SEEK_SET) != 0 ) {
        error = true;
        goto cleanup;
    }
    // write header to stream
    if ( _stream->write(&header, sizeof(header), 1) != 1 ) {
        error = true;
        goto cleanup;
    }

    // write the file to the stream 4K at a time.
    while( ! _stream->eof() ) {
        unsigned bytes_read = (unsigned)fread(buffer, 1, sizeof(buffer), file);
        if ( bytes_read != _stream->write(buffer, 1, bytes_read) ) {
            error = true;
            goto cleanup;
        }

        if ( bytes_read == 0 ) {
            break;
        }
    }

cleanup:
    if ( file ) {
        fclose(file);
    }

    return !error;
}

TempFileStream*
Archive::extract()
{
    Header header;

    // read the header from the stream
    if ( 1 != _stream->read(&header, sizeof(header), 1) ) {
        return 0;
    }

    // calculate checksum
    unsigned check = header.checksum;
    header.checksum = 0;
    if ( checksum((const char*)&header, sizeof(header)) != check ) {
        OutputDebugString(_T("Checksum bad while reading from archive.\n"));
        return 0;
    }

    // create temp file.
    TempFileStream* ret = new TempFileStream(header.name);
    ret->open();

    int bytes_to_read = 0;
    int bytes_left = header.size;
    char buffer[4096];
    while( bytes_left ) {
        // bytes to read is min of 4K or bytes left.
        bytes_to_read = sizeof(buffer);
        if ( bytes_to_read > bytes_left ) {
            bytes_to_read = bytes_left;
        }
        
        // read from stream
        int bytes_read = _stream->read(buffer, 1, bytes_to_read);
        if ( bytes_read != bytes_to_read ) {
            delete ret;
            return 0;
        }

        // write to temp file
        if ( bytes_read != (int)ret->write(buffer, 1, bytes_read) ) {
            delete ret;
            return 0;
        }

        // subtract bytes read from bytes left
        bytes_left -= bytes_read;
    }

    ret->close();

    return ret;
}

unsigned 
Archive::checksum( const char* data, unsigned bytes )
{
    unsigned sum = 0;
    for( unsigned i = 0; i < bytes; i++ ) {
        sum += data[i];
    }

    return sum;
}

Record::Record(int max_properties)
{
    _max = max_properties;
    _used = 0;
    _props = new Property[max_properties];
}

Record::~Record()
{
    for( int i = 0; i < _used; i++ ) {
        free( _props[i].name );
        free( _props[i].value );
    }

    delete[] _props;
}

bool 
Record::read( Stream* file )
{
    _TCHAR buffer[1024];

    clear();

    // while not end of file,
    while( 0 != file->gets(buffer, sizeof(buffer)/sizeof(*buffer)) ) {
        buffer[sizeof(buffer)/sizeof(*buffer)-1] = 0;

        _TCHAR* cr = _tcschr(buffer, 10);
        if ( cr ) {
            *cr = 0;
        }

        // if the line is '.', it ends the record.
        if ( _tcscmp(buffer, _T(".")) == 0 ) {
            return true;
        }

        // if the line contains an equal sign,
        _TCHAR* equals = _tcschr(buffer, _T('='));
        if ( !equals ) {
            continue;
        }

        *equals = 0;
        equals++;
        // if the last character is 10, remove it
        if ( _used < _max ) {
            newProp( buffer, equals );
        }
    }

    // premature end of record.
    assert(0);
    // probably opened file in "rb" mode instead of "r"
    return false;
}

bool
Record::write( Stream* file )
{
    // for each property,
    for( int i = 0; i < _used; i++ ) {
        // write it out 
        file->write((void*)_props[i].name, sizeof(_TCHAR), 
            (int)_tcslen(_props[i].name));
        int a = errno;
        file->write(_T("="), sizeof(_TCHAR), 1);
        file->write((void*)_props[i].value, sizeof(_TCHAR), 
            (int)_tcslen(_props[i].value));
        file->write(_T("\n"), sizeof(_TCHAR), 1);
    }

    // write end of record marker.
    return 2 == file->write(_T(".\n"), sizeof(_TCHAR), 2); 
}

void
Record::clear()
{
    _used = 0;
}

void
Record::newProp(const _TCHAR* name, const _TCHAR* value)
{
    assert( _used < _max );
    if ( _used >= _max ) {
        return;
    }
    
    _props[_used].name = _tcsdup(name);
    if ( _props[_used].name == 0 ) {
        return;
    }

    _props[_used].value = _tcsdup(value);
    if ( _props[_used].value == 0 ) {
        free( _props[_used].name);
        return;
    }

    _used++;
}

void
Record::set( const _TCHAR* name, const _TCHAR* value )
{
    newProp( name, value );
}

const _TCHAR*
Record::get( const _TCHAR* name )
{
    for( int i = 0; i < _used; i++ ) {
        if ( _tcscmp( name, _props[i].name ) == 0 ) {
            return _props[i].value;
        }
    }

    return _T("");
}

bool 
Record::get( const _TCHAR* name, int& value )
{
    return 1 == _stscanf( get(name), _T("%d"), &value );
}

bool 
Record::get( const _TCHAR* name, RECT& value )
{
    assert( sizeof(value.top) == 4);
    return 4 == _stscanf( get(name), _T("%d %d %d %d"), &value.left, &value.top,
        &value.right, &value.bottom );
}

bool 
Record::get( const _TCHAR* name, unsigned& value )
{
    return 1 == _stscanf( get(name), _T("%u"), &value);
}

void 
Record::set( const _TCHAR* name, const int& value )
{
    _TCHAR buffer[20];
    _stprintf(buffer, _T("%d"), value);
    set(name, buffer);
}

void 
Record::set( const _TCHAR* name, const RECT& value )
{
    _TCHAR buffer[80];
    _stprintf(buffer, _T("%d %d %d %d"), value.left, value.top, value.right, value.bottom);
    set(name, buffer);
}

void 
Record::set( const _TCHAR* name, const unsigned& value )
{
    _TCHAR buffer[20];
    _stprintf(buffer, _T("%u"), value);
    set(name, buffer);
}

void 
Record::set( const _TCHAR* name, const float& value )
{
    _TCHAR buffer[20];
    _stprintf(buffer, _T("%f"), value);
    set(name, buffer);
}

void
Record::set( const _TCHAR* name, const unsigned long& value )
{
    _TCHAR buffer[30];
    _stprintf(buffer, _T("%lu"), value);
    set(name, buffer);
}

void
Record::set( const _TCHAR* name, const POINT& value )
{
    _TCHAR buffer[60];
    _stprintf( buffer, _T("%d %d"), value.x, value.y);
    set(name, buffer);
}

void
Record::set( const _TCHAR* name, const bool& value )
{   
    _TCHAR buffer[20];
    _stprintf (buffer, _T("%s"), value ? "true" : "false");
    set(name, buffer);
}

void
Record::set( const _TCHAR* name, const long& value )
{
    _TCHAR buffer[20];
    _stprintf( buffer, _T("%ld"), value );
    set(name, buffer);
}

bool 
Record::get( const _TCHAR* name, _TCHAR*& value )
{
    for( int i = 0; i < _used; i++ ) {
        if ( _tcscmp( name, _props[i].name ) == 0 ) {
            free( value );
            value = _tcsdup( _props[i].value );
            return true;
        }
    }

    return false;
}


bool 
Record::get( const _TCHAR* name, POINT& value )
{
    return 2 == _stscanf( get(name), _T("%d %d"), &value.x, &value.y );
}

bool 
Record::get( const _TCHAR* name, unsigned long& value )
{
    return 1 == _stscanf( get(name), _T("%lu"), &value );
}

bool 
Record::get( const _TCHAR* name, long& value )
{
    return 1 == _stscanf( get(name), _T("%ld"), &value );
}

bool 
Record::get( const _TCHAR* name, float& value )
{
    return 1 == _stscanf( get(name), _T("%f"), &value );
}

bool
Record::get( const _TCHAR* name, bool& value )
{
    if ( 0 == _tcscmp( get(name), _T("true") ) ) {
        value = true;
        return true;
    } else if ( 0 == _tcscmp( get(name), _T("false") ) ) {
        value = false;
        return true;
    }

    return false;
}

bool write( Stream* stream, const char* str )
{
    return write( stream, (const wchar_t*)WideChar(str));
}

bool write( Stream* stream, const wchar_t* str )
{
    size_t size = wcslen( str );
    size_t size2 = size;
    unsigned char c;

    while( size >= 255 ) {
        c = 255;
        if ( 1 != stream->write( &c, 1, 1 ) ) {
            return false;
        }
    }

    c = size;
    if ( 1 != stream->write( &c, 1, 1 ) ) {
        return false;
    }

    if ( size2 != stream->write( (void*)str, sizeof( *str ), size2 ) ) {
        return false;
    }

    return true;
}

#define ZDATA_BUFFER_SIZE    (1024*64)
class ZData 
{
public:
    ZData();
    ~ZData();
    bool init;
    char buffer[ZDATA_BUFFER_SIZE];
    z_stream d_stream;
};

ZData::ZData()
{
    init = false;
    d_stream.zalloc = (alloc_func)0;
    d_stream.zfree = (free_func)0;
    d_stream.opaque = (voidpf)0;

    d_stream.next_in  = (Byte*)buffer;
    d_stream.avail_in = (uInt)0;

    if ( Z_OK != inflateInit(&d_stream) ) {
        assert(0);
    }
}

ZData::~ZData()
{
    inflateEnd(&d_stream);
}

ZStream::ZStream( Stream* source )
{
    _source = source;
    _zdata = new ZData;
    _reading = _writing = false;

}

ZStream::~ZStream()
{
    delete _zdata;
    _source = 0;
}

unsigned
ZStream::read( void* buffer, int rec_size, int num_recs )
{
    unsigned to_read = (unsigned)(rec_size*num_recs);
    unsigned read = 0;
    assert(!_writing);
    _reading = true;
    _zdata->d_stream.next_out = (Byte*)buffer;
    _zdata->d_stream.avail_out = to_read;
    while( _zdata->d_stream.avail_out > 0 ) {
        if ( _zdata->d_stream.avail_in == 0 ) {
            if ( _source->eof() ) {
                return 0;
            }
            
            _zdata->d_stream.avail_in = 
                _source->read( _zdata->buffer, 1, ZDATA_BUFFER_SIZE );
            _zdata->d_stream.next_in = (Byte*)_zdata->buffer;

            if ( _zdata->d_stream.avail_in == 0 ) {
                return 0;
            }
        }

        assert( _zdata->d_stream.avail_in > 0 );
        int before = (int)_zdata->d_stream.avail_out;

        int err = inflate(&_zdata->d_stream, Z_NO_FLUSH );

        if ( err == Z_OK || err == Z_STREAM_END ) {
            
        } else {
            assert(0);
        }

        read += before - (int)_zdata->d_stream.avail_out;

        if ( before - (int)_zdata->d_stream.avail_out) {
            assert(false);
            return 0;
        }
    }

    return read;
}

bool
ZStream::eof()
{
    return _zdata->d_stream.avail_in == 0 && _source->eof();
}


unsigned
ZStream::write( void* buffer, int rec_size, int num_recs )
{
    assert(0);
    return 0;
}

GzFileStream::GzFileStream()
{
    _handle = 0;
}

GzFileStream::~GzFileStream()
{    
    close();
}


bool 
GzFileStream::open( const _TCHAR* name, const _TCHAR* mode )
{
    _handle = gzopen(NarrowChar(name), NarrowChar(mode));
    return _handle != 0;
}

void
GzFileStream::close()
{
    if ( _handle ) {
        gzclose(_handle);
        _handle = 0;
    }
}

wchar_t*
GzFileStream::gets(wchar_t* val, int n)
{
    char* temp = (char*)malloc(n+1);
    if ( gzgets(_handle, temp, n) ) {
        wcscpy(val, WideChar(temp));
        free( temp );
        return val;
    } else {
        free( temp );
        return NULL;
    }
}

char*
GzFileStream::gets(char* val, int n)
{
    return gzgets(_handle, val, n);
}

unsigned 
GzFileStream::read( void* buffer, int rec_size, int num_recs)
{
    unsigned size = rec_size * num_recs;

    return (unsigned)gzread(_handle, buffer, size) / rec_size;
}


unsigned 
GzFileStream::write( void* buffer, int rec_size, int num_recs)
{
    unsigned size = rec_size * num_recs;
    return (unsigned)gzwrite(_handle, buffer, size ) / rec_size;
}

bool 
GzFileStream::eof()
{
    return gzeof(_handle) != 0;
}

