#include "UndoStack.h"
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include "debug.h" // must be last

UndoClient::UndoClient()
{

}

UndoClient::~UndoClient()
{

}

/**
 * Construct an UndoRecord.
 * @param pdescription A description of the action to be performed. This will
 * be returned by UndoStack::undoDescription() and
 * UndoStack::redoDescription().
 */
UndoRecord::UndoRecord( const _TCHAR* pdescription )
{
    description = _tcsdup( pdescription );
    prev = next = 0;
    done = false;
}

UndoRecord::~UndoRecord()
{
    free( description );
}

/**
 * Initialize the undostack.
 *
 * @param client A pointer to your UndoClient, which is called to perform and
 * undo actions.
 *
 * @param maxRecords The maximum size of the undo stack. The oldest records
 * will be deleted and no longer undoable if the stack exceeds this size.
 */
UndoStack::UndoStack( UndoClient* client, unsigned maxRecords )
{
    _maxRecords = maxRecords;
    _numRecords = 0;
    _firstRecord = 0;
    _undoRecord = 0;
    _client = client;
}

UndoStack::~UndoStack()
{
    clear();
}

/**
 * Deletes all the undo and redo records. This is useful when loading a new
 * document.
 */
void
UndoStack::clear()
{
    destroyRecords( _firstRecord );
    _numRecords = 0;
    _firstRecord = 0;
    _undoRecord = 0;
}

/**
 * Clears all actions which have been undone, so they are no longer redoable.
 * This is called automatically whenever a new action is performed.
 */
void
UndoStack::clearRedo()
{
    if ( _undoRecord ) {
        destroyRecords( _undoRecord->next );
    } else if ( _firstRecord ) {
        destroyRecords( _firstRecord );
    }
}

/**
 * Returns true if it is possible to undo something.
 */
bool
UndoStack::canUndo()
{
    return _undoRecord != 0;
}

/**
 * Returns true if is it possible to redo something.
 */
bool
UndoStack::canRedo()
{
    if ( _undoRecord ) {
        return _undoRecord->next != 0;
    } else {
        return _firstRecord != 0;
    }
}

/**
 * Adds an item to the undo stack, and optionally performs the action
 * immediately.
 *
 * @param record The UndoRecord which describes the action.
 *
 * @param callClient Set to true to perform the action immediately. It is
 * highly recommended that you structure your application so that all actions
 * are performed by the undo/redo framework. If you do not set this, the record
 * is added to the undo stack without being done.
 */
void
UndoStack::action( UndoRecord* record, bool callClient )
{
    clearRedo();

    if ( _undoRecord ) {
        _undoRecord->next = record;
        record->prev = _undoRecord;
    }

    if ( _firstRecord == 0 ) {
        _firstRecord = record;
    }

    _undoRecord = record;

    if ( _numRecords == _maxRecords ) {
        UndoRecord* oldFirst = _firstRecord;
        if ( _undoRecord == _firstRecord ) {
            _undoRecord = _firstRecord->next;
        }
        _firstRecord = _firstRecord->next;
        _firstRecord->prev = NULL;
        delete oldFirst;
    } else {
        _numRecords++;
    }

    record->done = true;
    if ( callClient ) {
        _client->undoRedo( record, true );
    }
}

/**
 * Perform an undo action. It is an error to call this if canUndo() returns
 * false.
 */
void
UndoStack::undo()
{
    UndoRecord* record = _undoRecord;
    assert( record != 0 );
    _undoRecord = record->prev;
    record->done = false;
    _client->undoRedo( record, false );
}

/**
 * Perform a redo action. It is an error to call this if canRedo() returns
 * false.
 */ 
void
UndoStack::redo()
{
    UndoRecord* record;

    if ( _undoRecord ) {
        record = _undoRecord->next;
    } else {
        record = _firstRecord;
    }

    assert( record );

    _undoRecord = record;
    record->done = true;
    _client->undoRedo( record, true );
}

/**
 * Returns the description of the next item to undo. Do not call this if
 * canUndo() returns false.
 */
const TCHAR*
UndoStack::undoDescription()
{
    assert( _undoRecord );
    return _undoRecord->description;
}

/**
 * Returns the description of the ntext item to redo. Do not call this if
 * canUndo() returns false.
 */
const TCHAR*
UndoStack::redoDescription()
{
    if ( _undoRecord ) {
        assert( _undoRecord->next );
        return _undoRecord->next->description;
    } else {
        assert( _firstRecord );
        return _firstRecord->description;
    }
}


void
UndoStack::destroyRecords( UndoRecord*& record )
{
    UndoRecord* cur = record;

    while( cur ) {
        UndoRecord* next = cur->next;
        delete cur;
        cur = next;
        assert( _numRecords > 0 ) ;
        _numRecords--;
    }

    record = 0;
}

UndoRecord*
UndoStack::getUndoRecord()
{
    return _undoRecord;
}
