// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#ifndef DISPATCHER_H
#define DISPATCHER_H

#include <list>

/**
 * Base class for objects wishing to listen to events. To listen for events,
 * derive from this class, implement onEvent(). Also, you will have to 
 * register for events of interest by calling Dispatcher::registerEvent().
 */
class EventListener
{
public:
    EventListener();
    virtual ~EventListener();

    virtual void onEvent(unsigned event)=0;
};

/**
 * Event dispatcher. Allows classes to communicate indirectly by sending
 * events. Any class may register to receive an event.
 *
 * For example, it is convenient to give Win32 menu items the same ID as an
 * event. So when a menu item is chosen, the event is sent. Any interested
 * class can then handle that menu event.
 */
class Dispatcher {
public:
    Dispatcher();
    ~Dispatcher();

    void dispatch( unsigned event );
    void registerEvent( EventListener* listener, unsigned event );
    void unregister( EventListener* listener );

private:
    struct Event {
        EventListener* listener;
        unsigned event;
    };

    typedef std::list<Event> EventList;
    EventList _events;
};

extern Dispatcher* _dispatcher;

#endif // DISPATCHER_H
