// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include <assert.h>
#include <windowsx.h>
#include "ScalogramViewWindow.h"
#include "SoundImage.h"
#include "WaveletDoc.h"
#include "ScalogramCreator.h"
#include "DibImage.h"
#include "ScalogramRender.h"
#include "PlayTask.h"
#include "WaveletApplication.h"
#include "WaveletOptions.h"
#include "PlayWaveformTask.h"
#include "Clipboard.h"
#include "dbg.h" // must be last

ScalogramViewWindow::ScalogramViewWindow(HINSTANCE hInstance) :
    ViewWindow(hInstance)
{
	_pszClassName = TEXT("ScalogramViewWindow");
	_pszTitle = TEXT("ScalogramView Window");

    _dwStyle |= WS_CHILD | WS_VISIBLE; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = (HCURSOR)LoadCursor(NULL, IDC_ARROW);
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
    _dwExtendedStyle |= 0;
	_doc = 0;
    _drawTool = new DrawBoxTool(this);
    _type = SoundImage::LINEAR;
    _colourMapType = 0;
    _creatorTask = new ScalogramCreator( this );
    _renderTask = new ScalogramRender(this);
    _enhanceTask = new ScalogramRender(this);
    _playTask = new PlayTask( this );
    _playWholeFileTask = new PlayWaveformTask(this);
    _dib = new DibImage();

    RECT rect;
    rect.left = 10;
    rect.top = 10;
    rect.right = 16000;
    rect.bottom = 128;
    _dispatcher->registerEvent( this, IDC_PLAY_REQ );
    _dispatcher->registerEvent( this, IDC_STOP_REQ );
    _dispatcher->registerEvent( this, IDC_PLAYTOOL_REQ );
    _dispatcher->registerEvent( this, IDC_SELTOOL_REQ );
    _dispatcher->registerEvent( this, IDC_AUTO_ENHANCE_TOGGLE_CNF );
    _dispatcher->registerEvent( this, IDC_COPY_VIEW_TO_CLIPBOARD_REQ );
}


ScalogramViewWindow::~ScalogramViewWindow()
{
    delete _creatorTask;
    delete _renderTask;
    delete _enhanceTask;
    delete _playTask;
    delete _playWholeFileTask;
}


/**
  * Removes the enhancement.
  */
void
ScalogramViewWindow::removeEnhancement()
{
    _enhanceTask->stop();
    _enhanceDib = 0;
}

void
ScalogramViewWindow::onEvent( unsigned event )
{
    // Handle events -- eg, selecting a drawing tool, etc.
    switch( event ) {
        case IDC_PLAYTOOL_REQ:    
            _drawTool = new DrawBoxTool(this);
            break;
        case IDC_SELTOOL_REQ:    
            _drawTool = new DrawSelectionTool(this);
            break;

        case IDC_PLAY_REQ: 
        { 
            PlayWaveformTaskParams params;
            params.wave = _doc->wave;
            _playTask->stop();
            _playWholeFileTask->start( params );
            break;
        }

        case IDC_STOP_REQ:
            _playTask->stop();
            _playWholeFileTask->stop();
            break;


        case IDC_AUTO_ENHANCE_TOGGLE_CNF:
            if ( ((bool)_enhanceDib) ^ _options->autoEnhance ) {
                enhance();
            }
            break;

        case IDC_COPY_VIEW_TO_CLIPBOARD_REQ: {
            Clipboard clipBoard;
            if ( _enhanceDib ) {
                clipBoard.copy( hwnd, _enhanceDib.ptr() );
            } else {
                clipBoard.copy( hwnd, _dib.ptr() );
            }

            break;
        }
    }
}

void
ScalogramViewWindow::create(HWND hParent)
{
    _hParent = hParent;

    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                CW_USEDEFAULT, CW_USEDEFAULT, _hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

void
ScalogramViewWindow::setType( SoundImage::ImageType type )
{
    _type = type;
    renderAgain();
}

/**
  * Re-render the preview image using new parameters.
  */
void 
ScalogramViewWindow::renderAgain()
{
    if ( _creatorTask->done() ) {
        removeEnhancement();
        ScalogramRenderParams params;
        params.image = _doc->scalogram;
        params.type = _type;
        params.map = ColourMap( _colourMapType );
        params.dib = _dib.ptr();
        params.offset = 0;
        params.length = _doc->wave->size;
        _renderTask->start( params );
    } else {
        // Will be re-rendered anyways when the creator task finishes.
    }
}

void
ScalogramViewWindow::setColourMap( unsigned index )
{
    _colourMapType = index;
    renderAgain();
}

SoundImage::ImageType
ScalogramViewWindow::getType()
{
    return _type;
}


LRESULT 
ScalogramViewWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    _drawTool->processMessage( hwnd, msg, wParam, lParam );

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONDBLCLK, onLButtonDown );
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        HANDLE_MSG(hwnd, WM_CHAR, onChar);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}

void
ScalogramViewWindow::onChar(HWND hwnd, _TCHAR ch, int lParam )
{
}


void
ScalogramViewWindow::onClose(HWND)
{

}

void 
ScalogramViewWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{

}

BOOL 
ScalogramViewWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{

    return TRUE;
}

void
ScalogramViewWindow::onDestroy(HWND hwnd)
{

}


void 
ScalogramViewWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

/**
  * Enhance the currently viewed portion. Without the enhancement, the viewed
  * portion of the scalogram is simply a 1024 pixel wide image stretched to
  * simulate a continuous transform. To enhance the image, we re-render it at
  * the proper resolution, so less stretching is needed and the view is more
  * accurate and less blurry.
  */
void
ScalogramViewWindow::enhance()
{
    RECT rect = getClientRect();
    double scale = (double)_dib->width()/_doc->wave->size;
    int left = (int)((double)_viewStart*scale);
    int right = (int)((double)(_viewEnd)*scale);

    if ( !_creatorTask->done() ) {
        return;
    }

    // If the viewed portion is zoomed in, and auto-enhancement is enabled,
    if ( right - left < rect.right - rect.left && _options->autoEnhance ) {
        // Restart the enhancement task.
        _enhanceTask->stop();
        // Create a new DibImage to contain the enhanced part, that is exactly
        // the right size to match the current window size.
        _enhanceDib = new DibImage();
        _enhanceDib->Create( rect.right, _doc->scalogram->bands, 24, 0 );

        ScalogramRenderParams params;
        params.image = _doc->scalogram;
        params.type = _type;
        params.map = ColourMap( _colourMapType );
        params.dib = _enhanceDib.ptr();
        params.offset = _viewStart;
        params.length = _viewEnd - _viewStart;
        InvalidateRect( hwnd, NULL, FALSE );
        _enhanceTask->start( params );
    } else {
        removeEnhancement();
        InvalidateRect( hwnd, NULL, FALSE );
    }
}

void 
ScalogramViewWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    if ( !_doc ) {
        return;
    }
    if ( fDoubleClick ) {
    }
}

void 
ScalogramViewWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
ScalogramViewWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{

}

/**
  * Renders the scalogram to the screen, handling any stretches.
  *
  * \param hdc Device context to use.
  *
  * \param dib DibImage to render. 
  *
  * \param offset Starting sample to render.
  *
  * \param samples Number of samples (columns) to render.
  *
  * \param rect Destination rect to render
  *
  * \param realSize Total number of samples (columns) in the SoundImage.
  */
static void
DrawScalogram( HDC hdc, DibImageRef dib, unsigned offset, unsigned samples, RECT* rect,
    int realSize)
{
    int size = rect->bottom - rect->top;
    double scale = (double)dib->width()/realSize;
    RECT src;
    src.left = (int)((double)offset*scale);
    src.top = 0;
    src.right = (int)((double)(offset + samples)*scale);
    src.bottom = dib->height();

    if ( 1 || src.right - src.left > rect->right - rect->left ) {
        SetStretchBltMode( hdc, HALFTONE );
    } else {
        SetStretchBltMode( hdc, COLORONCOLOR );
    }
    dib->stretch( hdc, &src, rect );
}

/**
  * Highlight all the user selections.
  */
void
ScalogramViewWindow::drawSelection( DeviceContext dc )
{
    if ( _doc == 0 ) {
        return;
    }

    int mode = SetROP2( dc, R2_NOT );
    Pen pen( 1, PS_DASH, 0 );
    dc.selectPen( pen );
    dc.selectBrush( (HBRUSH)GetStockObject(NULL_BRUSH) );

    int count = _doc->selection.rects.size();
    // for each rect,
    for ( int i = 0; i < count; i++ ) {
        // transform into pixel coordinates and draw it.
        RECT rect = _doc->selection.rects[i];
        untransform( &rect );
        Rectangle( dc, rect.left, rect.top, rect.right, rect.bottom );
    }

    SetROP2( dc, mode );
}

void 
ScalogramViewWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, 
        bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    RECT rect = getClientRect();

    // Paint window black.
    if ( _enhanceDib ) {
        double progress = _enhanceTask->getProgress();
        RECT src;
        src.left = 0;
        src.right = _enhanceDib->width();
        src.top = 0;
        src.bottom = (int)((double)_enhanceDib->height() * progress);
        RECT dest = rect;
        dest.bottom = (int)(progress * dest.bottom);

        _enhanceDib->stretch( hdc, &src, &dest );
    } else if ( _doc ) {

        DrawScalogram( hdc,
              _dib,
              _viewStart,
              _viewEnd - _viewStart,
              &rect, _doc->wave->size);

        if ( !_creatorTask->done() ) {
            _TCHAR text[100];
            int len = sprintf( text, "Rendering... %.3f%%", _creatorTask->getProgress() * 100 );
            SetBkColor(hdc, RGB(255,255,255));
            SetTextColor( hdc, RGB(0,0,0));
            DrawText( hdc, text, len, &rect, DT_CENTER | DT_VCENTER | DT_SINGLELINE );
        }

          _drawTool->onPaint( hdc );
	} else {
        FillRect( hdc, &rect, Brush(0) );
    }

    drawSelection( hdc );

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}

void 
ScalogramViewWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{

}

/**
  * Called when user opens a new document. Starts creating the wavelet
  * transform.
  */
void 
ScalogramViewWindow::setDoc( WaveletDocRef doc )
{
    _creatorTask->stop();
    _renderTask->stop();
    _enhanceTask->stop();
    _playTask->stop();
    _playWholeFileTask->stop();

    ViewWindow::setDoc( doc );

    _viewStart = 0;
    _viewEnd = _doc->wave->size;

    ScalogramCreatorParams params;
    params.wave = _doc->wave;
    params.image = _doc->scalogram;
    params.lowerScale = _doc->scalogram->lowerScale;
    params.upperScale = _doc->scalogram->upperScale;
    params.f0 = 10;
    params.type = _type;
    params.map = ColourMap( _colourMapType );
    params.dib = _dib.ptr();

    _dib->Create( 1024, _doc->scalogram->bands, 24, 0 );
    removeEnhancement();
    _creatorTask->start( params );
}

void
ScalogramViewWindow::onTaskNotify( AbstractTask* task )
{
    InvalidateRect( hwnd, NULL, FALSE );

    if ( task == _creatorTask && _creatorTask->done() ) {
        // On the fly-rendering is inaccurate since the SoundImage had no mmax
        // and mmin data -- so we need to render again, otherwise the top half
        // of the image is over-emphasized.
        renderAgain();
    } else if ( task == _renderTask && _renderTask->done() ) {
        enhance();
    }
}

/**
  * Called when the user used the Play Tool to draw a rectangular region.
  */
void 
ScalogramViewWindow::onDrawRect( RECT* rect )
{
    if ( !_doc ) {
        return;
    }

    _playWholeFileTask->stop();
    _playTask->stop();
    RECT trect = *rect;
    transform( &trect );
   
    // Start the PlayTask to filter and play the highlighted region.
    PlayTaskParams params;
    params.image = _doc->scalogram;
    params.rowStart = trect.top;
    params.numRows = trect.bottom - trect.top+1;
    params.offset = trect.left;
    params.length = trect.right - trect.left + 1;
    _playTask->start(params);
}

/**
  * Transform the given coordinate from pixels into sample coordinates.
  */
int 
ScalogramViewWindow::transformX( int x )
{
    RECT rect = getClientRect();
    return (int)((float)x / (rect.right-rect.left) *  (_viewEnd - _viewStart) + _viewStart );
}

/**
  * Transform the given coordinate from pixels into sample coordinates.
  */
int 
ScalogramViewWindow::transformY( int y )
{
    RECT rect = getClientRect();
    return (int)((float)y / (rect.bottom-rect.top) * _doc->scalogram->bands );
}

/**
  * Transform the given coordinate from pixels into sample coordinates.
  */
void
ScalogramViewWindow::transform( RECT* rect )
{
    rect->left = transformX( rect->left );
    rect->right = transformX( rect->right );
    rect->top = transformY( rect->top );
    rect->bottom = transformY( rect->bottom );
}

/**
  * Transform the given coordinate from samples into pixel coordinates.
  */
int 
ScalogramViewWindow::untransformX( int x )
{
    RECT rect = getClientRect();
    return (int)((float)(x-_viewStart) * (rect.right-rect.left) /  (_viewEnd - _viewStart) );
}

/**
  * Transform the given coordinate from samples into pixel coordinates.
  */
int 
ScalogramViewWindow::untransformY( int y )
{
    RECT rect = getClientRect();
    return (int)((float)y * (rect.bottom-rect.top) / _doc->scalogram->bands );
}

/**
  * Transform the given coordinate from samples into pixel coordinates.
  */
void
ScalogramViewWindow::untransform( RECT* rect )
{
    rect->left = untransformX( rect->left );
    rect->right = untransformX( rect->right );
    rect->top = untransformY( rect->top );
    rect->bottom = untransformY( rect->bottom );
}

/**
  * Change the viewed region.
  *
  * \param start First sample index to view.
  *
  * \param end Last sample index to view.
  */
void 
ScalogramViewWindow::setViewExtents( int start, int end )
{
    _viewStart = start;
    _viewEnd = end;
    removeEnhancement();

    InvalidateRect( hwnd, NULL, FALSE );
}
