// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#ifndef SOUND_DEVICE_H
#define SOUND_DEVICE_H

#include <stdio.h>

class SampleBuffer;

class AudioSink
{
public:
	virtual ~AudioSink() {}
	virtual void handleAudioData( unsigned char* data, unsigned int numBytes )=0;
};

/**
 The SoundDevice class allows you to play and record audio data. It is a
 singleton class accessed through SoundDevice::instance(). Before accessing the
 instance, you can set the desired sample rate using SoundDevice::Initialize().

 Following initialization, memory buffers may be allocated using
 allocateBlock(), and freed using releaseBlock().  However, once write() is
 called, the block will be released by the system automatically, so there
 should be no need to release.

 The size of the blocks cannot be set. Instead, the size in samples can be
 retrieved using samplesPerBlock().

 The audio data must be in signed 16 bit numbers. Each channel is interleaved
 in the data.

 When write() is called while playing, it will block if too much data is queued
 up. It is recommended that you set up another thread to do the writing.
 */
class SoundDevice
{
 public:
 /**
  * This should be the first function called. If called after the sound device is used, it is shut
  * down and restarted. 
  * 
  */
    static bool Initialize( unsigned SamplesPerSec, unsigned channels );
	static SoundDevice* instance();
    static void shutdown();

	virtual unsigned char* allocateBlock()=0;
	virtual void releaseBlock( unsigned char* )=0;

	virtual void beginRecord( AudioSink* pAudioSink )=0;
	virtual bool isRecording()=0;
	virtual void endRecord()=0;

	virtual void beginPlay()=0;
	virtual void write( unsigned char* data )=0;
	virtual void endPlay()=0;

	static void close();

	virtual unsigned int sampleRate()=0;
	virtual unsigned int samplesPerBlock()=0;
    virtual unsigned int numChannels()=0;

	enum Format {
		Signed_16_LE
	};

	virtual Format format()=0;
};

#endif //SOUND_DEVICE_H
