// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <assert.h>
#include "SoundImage.h"
#include "DibImage.h"
#include "ColourMap.h"
#include "OverlappedFile.h"
#include <algorithm>
#include "dbg.h"

static const bool no_alloc = false;

SoundImage::SoundImage() :
    bands(0),
    samples(0),
    sampleRate(0),
    mmin((float)1.0e36),
    mmax((float)0.0)
{
}

SoundImage::~SoundImage()
{
}

/**
  Calculate memory requirements in megabytes. Used to decide whether to store
  to disk, or keep in memory.
  */
unsigned 
SoundImage::calcMemorySizeMB( unsigned sampleRate, unsigned samples )
{
    float dummy1, dummy2;
    unsigned bands = calcBands( (float)sampleRate, &dummy1, &dummy2);
    unsigned __int64 bytes = (__int64)bands * samples * sizeof(float) * 2;
    printf("Rate: %d Bands: %d samples: %d\n", sampleRate, bands, samples);
    return (unsigned)(bytes/1024/1024);
}

/**
  * Updates mmax and mmin with the minimum and maximum value in the row.
  * This should be called by the addRow() function in derived classes.
  * Otherwise, mmax and mmin will be inaccurate and rendering the image will
  * not work.
  */
void 
SoundImage::findMaxInRow( float* data )
{
    for ( unsigned i = 0; i < samples; i += 2 ) {
        float val = (float)fabs( data[i*2] );
        if ( val < mmin ) {
            mmin = val;
        }
        
        if ( val > mmax ) {
            mmax = val;
        }
    }
}

static float calcLinear(float* data)
{
    return sqrt(data[0]*data[0]+data[1]*data[1]);
}

static float calcLog(float* data)
{
    return log(sqrt(data[0]*data[0]+data[1]*data[1]));
}

static float calcLogReal(float* data)
{
    if ( data[0] == 0.0 ) {
        return 0.0;
    } 
    return log(fabs(data[0]));
}

static float calcLogComplex(float* data)
{
    return log(fabs(data[1]));
}

const static float pi = (float)3.14159265358979323846;
static float phase( float x, float y )
{
    if ( y == 0.0 ) {
        if ( x >= 0 ) {
            return 0.0;
        } else {
            return pi;
        }
    } else {
        float sgn = 1.0;
        if ( y < 0.0 ) {
            sgn = -1.0;
        }

        if ( x >= 0.0 ) {
            return abs(atan( y / x ))*sgn;
        } else if ( x == 0.0 ) {
            return (float)pi/2*sgn;
        } else {
            return (float)(pi - abs(atan( y / x )))*sgn;
        }
    }
}

static float calcPhase(float* data)
{
    return (phase( data[0], data[1]) + pi) / (2 * pi);
}

SoundImageReader::SoundImageReader()
{

}

SoundImageReader::~SoundImageReader()
{

}

/**
  * Renders the given row to a DibImage. If the width of the DibImage is
  * smaller than the number of samples, the row is shrunk to fit using a
  * running average.
  *
  * \warning If the DibImage is larger than the number of samples, unexpected
  * results will occur since stretching is not implemented.
  *
  * \param row The row in the image to render.
  *
  * \param dib The DibImage to which to render. This is expected to contain the
  * same number of rows as the SoundImage::band paramter.
  *
  * \param type The type of image to render.
  * 
  * \param map The colour map to use.
  *
  * \param offset At which column to start. The given sample will be placed at
  * column 0 in the dib.
  *
  * \paramlength The number of samples to render.
  *
  * \param Pointer to the samples, which must have previously been obtained
  * using getRow().
  */
void 
SoundImage::renderRowToDib( unsigned row, DibImage* dib, ImageType type, ColourMap& map,
    unsigned offset, unsigned length, float* samples )
{
    if ( mmax <= mmin ) {
        return;
    }

    if ( samples == 0 ) {
        return;
    }

    double filter = (float)dib->width() / length;
    int lastpix = 0;
    double value = 0.0;
    double val = 0.0;
    double mymax = mmax;
    double mymin = mmin;

    float (*func)(float* data);

    switch( type ) {
        case LOGARITHMIC: func = &calcLog; break;
        case LOGARITHMIC_REAL: func = &calcLogReal; break;
        case LOGARITHMIC_COMPLEX: func = &calcLogComplex; break;
        case LINEAR: func = &calcLinear; break;
        case PHASE: func = &calcPhase; break;
        default: assert(0); break;
    }             

    if ( type == LOGARITHMIC || type == LOGARITHMIC_REAL || type ==
            LOGARITHMIC_COMPLEX ) 
    {
        mymax = log(mymax);
        if ( mymin != 0.0 ) {
            mymin = log(mymin);
        } else {
            mymin = 0.0;
        }
    } else if ( type == PHASE ) {
        mymin = 0.0;
        mymax = 1.0;
    }

    if ( 1 ) {
        for( int j = 0; j < dib->width(); j++ ) {
            int index = (int)((double)j/filter)+offset;
            val = func(&samples[index*2]);
            dib->setPixel( j, row, map.map( (float)(val-mymin) /
                        (float)(mymax-mymin) ));
        }
    } else {
        // old way smooths, but is much slower.
        for( int j = offset; j < (int)(length+offset); j++ ) {
            val = (1.0-filter)*val+filter*func(&samples[j*2]);
            int pix = (int)((double)(j-offset)*filter); 
            if ( lastpix != pix ) {
                dib->setPixel( pix, row, map.map( (float)(val-mymin) /
                            (float)(mymax-mymin) ));
                lastpix = pix;
            }
        }
    }
}

class MemorySoundImageReader : public SoundImageReader
{
public:
    MemorySoundImage* image;
    MemorySoundImageReader( MemorySoundImage* image ) : 
        image(image)
    {

    }

    virtual ~MemorySoundImageReader()
    {

    }

    virtual float* getRow( unsigned row, unsigned offset, unsigned length )
    {
        return &image->values[row][offset*2];    
    }
};

MemorySoundImage::MemorySoundImage() :
    values(0)
{

}

MemorySoundImage::~MemorySoundImage()
{
    if ( values ) {

        for( unsigned i = 0; i < bands; i++ ) {
            free( values[i] );
        }

        free( values );
        values = 0;
    }
}

SoundImageReader* 
MemorySoundImage::createReader()
{
    return new MemorySoundImageReader( this );
}

unsigned
SoundImage::calcBands( float sampleRate, float* upperScale, float* lowerScale )
{
    float lowFreq = 20;
    float highFreq = 20000;
    *upperScale = sampleRate / lowFreq;
    *lowerScale = _cpp_max( sampleRate / highFreq, (float)2.0 );
    printf("sampleRate: %g upperScale:%g lowerScale:%g\n", sampleRate,
            *upperScale, *lowerScale);
    return (unsigned)(*upperScale - *lowerScale)+1;
}

bool 
MemorySoundImage::alloc( unsigned numSamples, unsigned sampleRate )
{
    bands = calcBands((float)sampleRate, &upperScale, &lowerScale);
    this->sampleRate = sampleRate;
    samples = numSamples;
    
    values = (float**)malloc( bands * sizeof(float*) );
    if ( values == NULL ) {
        //malloc failure.
        return false;
    }

    for( unsigned i = 0; i < bands; i++ ) {
        values[i] = (float*)malloc( samples*2 * sizeof( float ) );
        if ( values[i] == NULL ) {
            //malloc failure.
            for( unsigned j = 0; j < i; j++ ) {
                free( values[j] );
            }
            free( values );
            values = 0;
            return false;
        }
        memset( values[i], 0, samples*2 * sizeof(float) );
    }

    return true;
}

void 
MemorySoundImage::addRow( unsigned row, float* data )
{
    memcpy( values[row], data, samples * 2 * sizeof(float) );
    findMaxInRow( data );
}

class DiskSoundImageReader : public SoundImageReader
{
public:
    DiskSoundImage* image;
    OverlappedOperation* _reader;
    float* _buffer;
    unsigned _rowSize;
    unsigned _haveRow[2];
    DiskSoundImageReader( DiskSoundImage* image ) : 
        image(image)
    {
        _reader = new OverlappedOperation( *image->_file );
        _rowSize = image->samples*2*sizeof(float);

        // alloc enough space for two rows.
        //_buffer = (float*)malloc( 2*_rowSize );
        _buffer = (float*)VirtualAlloc(NULL,  2*_rowSize, MEM_RESERVE|MEM_COMMIT, 
            PAGE_READWRITE);

        // try to read first two lines.
        _reader->read( 0, _rowSize*2, _buffer );
        _haveRow[0] = 0;
        _haveRow[1] = 1;
    }

    virtual ~DiskSoundImageReader()
    {
        // free memory.
        dbgprint((DDISK, "DiskSoundReader::~DiskSoundReader() -- wait"));
        delete _reader;
        dbgprint((DDISK, "DiskSoundReader::~DiskSoundReader() -- done"));
        //free( _buffer );
        VirtualFree( _buffer, 0, MEM_RELEASE );
    }

    virtual float* getRow( unsigned row, unsigned roffset, unsigned length )
    {
        // wait for pending read to complete.
        dbgprint((DDISK, "DiskSoundReader::getRow( %d )", row));
        _reader->wait();

        // is this the last row?
        bool last = row == image->bands-1;
        int offset = -1;

        if ( _haveRow[0] == row ) {
            offset = 0;
        } else if ( _haveRow[1] == row ) {
            offset = 1;
        } 

        // check if we've got the given row.
        if ( offset >= 0 ) {
            // if rows left,
            if ( !last ) {
                // initiate read for next row after
                _haveRow[1-offset] = row+1;
                _reader->read( (__int64)(row+1)*_rowSize, _rowSize, _buffer + (1-offset)*image->samples*2 );
            }
        } else {
            // calculate number of rows to read (two if possible, one if row is last one)
            if ( last ) {
                _haveRow[0] = row;
                _reader->read( (__int64)row*_rowSize, _rowSize, _buffer );
            } else {
                _haveRow[0] = row;
                _haveRow[1] = row+1;
                _reader->read( (__int64)row*_rowSize, _rowSize*2, _buffer );
            }
            offset = 0;
            // initiate read
            // wait for it.
            _reader->wait();
        }

        // return the row.
        return _buffer + offset * image->samples * 2 + roffset*2;
    }
};

DiskSoundImage::DiskSoundImage() 
{
    _file = 0;
}

DiskSoundImage::~DiskSoundImage()
{
    close();
}

SoundImageReader* 
DiskSoundImage::createReader()
{
    return new DiskSoundImageReader( this );
}

bool
DiskSoundImage::create( const char* filename )
{
    close();
    OverlappedFile* file = new OverlappedFile();
    if ( !file->create( filename, FILE_FLAG_NO_BUFFERING | FILE_FLAG_DELETE_ON_CLOSE ) ) {
        delete file;
        return false;
    }

    _file = file;
    _writer = new OverlappedOperation( *_file );
    return true;
}

bool
DiskSoundImage::alloc( unsigned numSamples, unsigned sampleRate )
{
    assert( _file );
    bands = calcBands((float)sampleRate, &upperScale, &lowerScale);
    this->sampleRate = sampleRate;

    samples = numSamples;

    __int64 space = (__int64)sizeof(float)*samples*2*bands/1000/1000;

    printf("Requires %d MB disk space.\n", (unsigned)space);

    return true;
}

void 
DiskSoundImage::addRow( unsigned row, float* data )
{
    findMaxInRow( data );
    _writer->wait();
    unsigned __int64 offset = (__int64)row*samples*2*sizeof(float);
    unsigned length = samples*2*sizeof(float);
    _writer->write( offset, length, data );
}

void
DiskSoundImage::close()
{
    if ( _file ) {
        delete _file;
        delete _writer;
        _file = 0;
    }
}
