#ifndef SoundImage_H
#define SoundImage_H

#include <stdio.h>

class DibImage;
class ColourMap;
class SoundImageReader;

/**
  * Class to contain the continuous wavelet transform.
  */
class SoundImage
{
public:
    enum ImageType {
        LOGARITHMIC,
        LOGARITHMIC_REAL,
        LOGARITHMIC_COMPLEX,
        LINEAR,
        PHASE
    };

	SoundImage();
	virtual ~SoundImage();

    virtual SoundImageReader* createReader()=0;

    static unsigned calcMemorySizeMB(unsigned sampleRate, unsigned numSamples);
    static unsigned calcBands( float sampleRate, float* upperScale, float*
            lowerScale );

    virtual bool alloc( unsigned numSamples, unsigned sampleRate )=0;

    virtual void addRow( unsigned row, float* data )=0;

    void renderRowToDib( unsigned row, DibImage* dib, SoundImage::ImageType type, ColourMap& map,
        unsigned offset, unsigned length, float* samples);

    /// Number of rows in the image.
    unsigned bands;

    /// Number of samples in the image. Each sample has a real and complex
    /// part.
    unsigned samples;

    /// Sample rate of the represented audio file.
    unsigned sampleRate;

    /// minimum value of any sample.
    float mmin;

    /// maximum value of any sample.
    float mmax;

    /// highest scale factor corresponding to last band.
    float upperScale;

    /// Lowest scale factor, corresponding to the first band.
    float lowerScale;

protected:
    void findMaxInRow( float* data );

private:
};

/**
  * The SoundImage class cannot be read directly. To optimize disk access, the 
  * you must create a SoundImageReader using createReader(), and delete it when
  * you are finished reading rows.
  */
class SoundImageReader
{
public:
    SoundImageReader();
    virtual ~SoundImageReader();

    virtual float* getRow( unsigned row, unsigned offset, unsigned length )=0;
};

/**
  * A MemorySoundImage holds the continuous wavelet transform completely in
  * memory. For this reason, its size is limited.
  */
class MemorySoundImage : public SoundImage
{
public:
	MemorySoundImage();
	virtual ~MemorySoundImage();

    SoundImageReader* createReader();

    virtual bool alloc( unsigned numSamples, unsigned sampleRate );

    virtual void addRow( unsigned row, float* data );

private:
    friend class MemorySoundImageReader;
    float** values;
};

class OverlappedFile;
class OverlappedOperation;

/**
  * A DiskSoundImage holds the continuous wavelet transform on disk.
  * Its size is not limited. Access to the rows is optimized for reading each
  * row in order from the first to the last. Reading them any other way will
  * result in performance problems. Each call to getRow() will asynchronously
  * fetch the next row, so you can process the row while the next one is read
  * from disk.
  */
class DiskSoundImage : public SoundImage
{
public:
    DiskSoundImage();
    virtual ~DiskSoundImage();
    SoundImageReader* createReader();

    bool create( const char* filename );
    bool alloc( unsigned numSamples, unsigned sampleRate );
    void close();

    virtual void addRow( unsigned row, float* data );

private:
    friend class DiskSoundImageReader;
    OverlappedFile* _file;
    OverlappedOperation* _writer;
};

#endif // SoundImage_H
