// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include "Waveform.h"
#include <memory.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>
#include "sndfile.h"
#include "BaseWnd.h"
#include "dbg.h" // must be last

double
Waveform::findMax()
{
    double max = 0.0;
    for( unsigned i = 0; i < size; i++ ) {
        double val = abs( samples[i] );
        if ( val > max ) {
            max = val;
        }
    }

    return max;
}

/**
  * Multiplies each sample in the waveform by the given scaling factor.
  */
void
Waveform::scale( double factor )
{
    if ( factor > 0.0 ) {
        for( unsigned i = 0; i < size; i++ ) {
            samples[i] *= factor;
        }
    }
}


/**
  * Scales all of the samples such that the maximum value is 1.0.
  */
void
Waveform::normalize()
{
    double max = findMax();
    if ( max > 0.0 ) {
        scale( 1.0 / max );
    }
}

WaveformObserver::~WaveformObserver()
{

}

Waveform::Waveform()
{
    samples = NULL;
    sampleRate = 8000;
    _observer = NULL;
    alloc( 0 );
}

Waveform::~Waveform()
{
    free( samples );
}

void
Waveform::setObserver( WaveformObserver* observer )
{
    _observer = observer;
}

void
Waveform::notifyChanged()
{
    if ( _observer ) {
        _observer->onWaveformChanged(this);
    }
}

void
Waveform::alloc( unsigned size )
{
    samples = (double*)realloc( samples, size * sizeof(*samples) );
    this->size = size;
}

Waveform* loadOther( const char* infilename );

/**
  * Loads a waveform from a file. Currently, GoldWave .txt files and .wav files
  * are supported.
  */
Waveform* 
Waveform::load( const char* filename )
{
    // if it's not a text extention, load "other".
    if ( !CompareExtension( filename, "txt" ) ) {
        return loadOther( filename );
    }

    // Load in GoldWave .txt format file.
    FILE* file = fopen( filename, "rt" );
    char buffer[256];
    char* line;
    unsigned freq, channels, size;
    Waveform* wave = NULL;

    if ( file == NULL ) {
        return NULL;
    }

    line = fgets( buffer, sizeof(buffer), file);

    if ( line == 0 ) {
        goto error;
    }

    if ( 3 != sscanf( line, "[ASCII %dHz, Channels: %d, Samples: %d,", &freq, &channels, 
        &size ) ) {
        goto error;
    }

    if ( channels != 1 ) {
        goto error;
    }

    wave = new Waveform();
    wave->alloc( size );
    wave->sampleRate = freq;

    for ( unsigned i = 0; i < size; i++ ) {
        int num;
        line = fgets( buffer, sizeof(buffer), file);
        if ( line == 0 ) {
            break;
        }

        if ( 1 != sscanf( line, "%d", &num ) ) {
            goto error;
        }

        wave->samples[i] = (double)num/32767;
    }

    //wt1( wave->samples, wave->size, 1, daub4 );
    wave->name = filename;

    return wave;

error:
    fclose( file );
    delete wave;
    return NULL;
}

/** 
  * Clears the given section of the waveform.
  */
void
Waveform::clear( unsigned start, unsigned end )
{
    memset( samples + start, (end - start) * sizeof(*samples), 0 ); 
}

/**
  * Sets all samples in the waveform to silence.
  */
void
Waveform::clear()
{
    memset( samples, 0, sizeof(*samples)*size );
}

/**
  * Resample to new sampling rate.
  */
bool 
Waveform::resample( unsigned rate )
{
    if ( rate == sampleRate ) {
        return true;
    }
    double scale = (double)sampleRate / rate;
    unsigned newSize = (unsigned)((double)size/scale);
    double* newSamples = (double*)malloc(sizeof(double)*newSize);

    printf("\n scale:%g samples: %d     size: %d sampleRAte: %d\n",
            scale, samples, size, sampleRate);

    if ( newSamples == 0 ) {
        return false;
    }

    for ( unsigned i = 0; i < newSize; i++ ) {
        newSamples[i] = samples[(unsigned)floor(scale*i)];
    }

    free( samples );
    samples = newSamples;
    size = newSize;
    sampleRate = rate;

    printf(" scale:%g samples: %d     size: %d sampleRAte: %d\n",
            scale, samples, size, sampleRate);
    return true;
}

/**
  * Save the waveform as a GoldWave .txt file.
  */
bool
Waveform::save( const char* filename )
{
    FILE* file = fopen( filename, "wt" );

    fprintf( file, "[ASCII %dHz, Channels: 1, Samples: %d, Flags: 1]\n",
        sampleRate, size );

    for ( unsigned i =0 ; i < size; i++ ) {
        fprintf(file, "%d\n", (int)(samples[i] * 32767));
    }

    fclose( file );

    return true;

}

/*    This will be the length of the buffer used to hold.frames while
**    we process them.
*/
#define		BUFFER_LEN	1024

/**
  * Load a .wav file. Adapted from libsndfile example program.
  */
Waveform* loadOther( const char* infilename )
{   /* This is a buffer of double precision floating point values
    ** which will hold our data while we process it.
    */
    static double data [BUFFER_LEN] ;

    /* A SNDFILE is very much like a FILE in the Standard C library. The
    ** sf_open function return an SNDFILE* pointer when they sucessfully
	** open the specified file.
    */
    SNDFILE      *infile;

    /* A pointer to an SF_INFO stutct is passed to sf_open.
    ** On read, the library fills this struct with information about the file.
    ** On write, the struct must be filled in before calling sf_open.
    */
    SF_INFO		sfinfo ;
    int			readcount ;

    /* Here's where we open the input file. We pass sf_open the file name and
    ** a pointer to an SF_INFO struct.
    ** On successful open, sf_open returns a SNDFILE* pointer which is used
    ** for all subsequent operations on that file.
    ** If an error occurs during sf_open, the function returns a NULL pointer.
	**
	** If you are trying to open a raw headerless file you will need to set the
	** format and channels fields of sfinfo before calling sf_open(). For
	** instance to open a raw 16 bit stereo PCM file you would need the following
	** two lines:
	**
	**		sfinfo.format   = SF_FORMAT_RAW | SF_FORMAT_PCM_16 ;
	**		sfinfo.channels = 2 ;
    */
    if (! (infile = sf_open (infilename, SFM_READ, &sfinfo)))
    {   /* Open failed so print an error message. */
        printf ("Not able to open input file %s.\n", infilename) ;
        /* Print the error message from libsndfile. */
        return NULL ;
    }

    Waveform* wave = new Waveform();
    wave->alloc( (int)sfinfo.frames );
    wave->sampleRate = sfinfo.samplerate;

    /* While there are.frames in the input file, read them, process
    ** them and write them to the output file.
    */

    int readAtOnce = BUFFER_LEN/sfinfo.channels;
    readAtOnce *= (int)sfinfo.channels;
    int sample = 0;
    while ((readcount = (int)sf_read_double (infile, data, readAtOnce))) {  
        // discard everything except the first channel.
        for( int i = 0; i < readcount / sfinfo.channels; i++ ) {
            assert( sample < (int)wave->size);
            wave->samples[sample] = data[i*sfinfo.channels];
            sample++;
        }
    }

    printf("%d samples loaded.\n", wave->size);

    /* Close input and output files. */
    sf_close (infile) ;

    wave->name = infilename;

    return wave;
} /* main */

bool 
Waveform::getFileInfo( const _TCHAR* filename, WaveInformation* info )
{
    return false;
}
