// code reviewed on 2004-01-24/13:16
//*******************************************************************
//BaseWnd.cpp
//By Steve Hanov (smhanov@uwaterloo.ca)
//November 1999
//
// A Window wrapper that allows inheritance through
// a virtual window procedure and command handlers.
//******************************************************************
#include <windows.h>
#include <commctrl.h>
#include <assert.h>
#include "BaseWnd.h"
#include <stdio.h>
#include "richedit.h"
#include "debug.h" // must be last

#ifndef GWLP_USERDATA
#define GWLP_USERDATA GWL_USERDATA
#define GWLP_WNDPROC GWL_WNDPROC
#define LONG_PTR LONG
#define SetWindowLongPtr SetWindowLongA
#define GetWindowLongPtr GetWindowLongA
#endif

// ---------------------------------------------------------------------------
// WindowMap
//
// The WindowMap object is used internally by the framework. It allows
// object-oriented wrappers for sublcassed windows, when it is undesirable to
// store a pointer using SetWindowLong().
static class WindowMap
{
public:
    WindowMap();
    ~WindowMap();

    void mapDialog( HWND hwnd, Dialog* dialog );
    Dialog* getDialog( HWND );
    void removeDialog( HWND hwnd, Dialog* dialog );

    void mapSubclassed( HWND hwnd, SubclassedWindow* dialog );
    SubclassedWindow* getSubclassed( HWND );
    void removeSubclassed( HWND hwnd, SubclassedWindow* subclassed );

private:
    struct MapItem {
        MapItem( HWND );
        HWND hwnd;
        Dialog* dialog;
        SubclassedWindow* subclassed;
        MapItem* next;
    }; 

    MapItem* _hash[67];

    MapItem* lookup( HWND hwnd );
    void add( MapItem* item );
    void clean( MapItem* item );
} _windowMap;

WindowMap::MapItem::MapItem( HWND phwnd )
{
    hwnd = phwnd;
    dialog = 0;
    subclassed = 0;
    next = 0;
}

WindowMap::WindowMap()
{
    memset( _hash, 0, sizeof(_hash) );
}

WindowMap::~WindowMap()
{
    unsigned int i;
    for ( i = 0; i < sizeof(_hash)/sizeof(*_hash); i++ ) {
        MapItem* item = _hash[i];
        while( item ) {
            MapItem* next = item->next;
            delete item;
            item = next;
        }
    }
}

// --------------------------------------------------------------------------
// Insert a <hwnd, dialog> pair into the window map.
void 
WindowMap::mapDialog( HWND hwnd, Dialog* dialog )
{
    MapItem* item = lookup( hwnd );
    if ( !item ) {
        item = new MapItem( hwnd );
        add( item );
    }
    
    assert( item->dialog == 0 );
    item->dialog = dialog;
}

// --------------------------------------------------------------------------
// Lookup a Dialog object given a window handle.
Dialog*
WindowMap::getDialog( HWND hwnd )
{
    MapItem* item = lookup( hwnd );
    if ( item ) {
        return item->dialog;
    }

    return 0;
}

// --------------------------------------------------------------------------
// Remove an <hwnd, dialog> pair from the window map.
void
WindowMap::removeDialog( HWND hwnd, Dialog* dialog )
{
    MapItem* item = lookup( hwnd );
    if ( item ) {
        if ( item->dialog == dialog ) {
            item->dialog = 0;
            clean( item );
            return;
        }
    }

    assert(0);
}


// --------------------------------------------------------------------------
// Add an <hwnd, SubclassedWindow pointer> pair to the window map.
void 
WindowMap::mapSubclassed( HWND hwnd, SubclassedWindow* subclassed )
{
    MapItem* item = lookup( hwnd );
    if ( !item ) {
        item = new MapItem( hwnd );
        add( item );
    }
    
    assert( item->subclassed == 0 );
    item->subclassed = subclassed;
}

// --------------------------------------------------------------------------
// Lookup an instance of SubclassedWindow given a window handle.
SubclassedWindow*
WindowMap::getSubclassed( HWND hwnd )
{
    MapItem* item = lookup( hwnd );
    if ( item ) {
        return item->subclassed;
    }

    return 0;
}

// --------------------------------------------------------------------------
// Remove a SubclassedWindow object from the map.
void
WindowMap::removeSubclassed( HWND hwnd, SubclassedWindow* subclassed )
{
    MapItem* item = lookup( hwnd );
    if ( item ) {
        if ( item->subclassed == subclassed ) {
            item->subclassed = 0;
            clean( item );
            return;
        }
    }

    assert(0);
}

WindowMap::MapItem* 
WindowMap::lookup( HWND hwnd )
{
    unsigned bucket = (unsigned)hwnd % (sizeof(_hash)/sizeof(*_hash));
    MapItem* item = _hash[bucket];

    while( item ) {
        if ( item->hwnd == hwnd ) {
            return item;
        }
        
        item = item->next;
    }

    return 0;
}

void 
WindowMap::add( MapItem* item )
{
    unsigned bucket = (unsigned)item->hwnd % (sizeof(_hash)/sizeof(*_hash));
    item->next = _hash[bucket];
    _hash[bucket] = item;
}

void 
WindowMap::clean( MapItem* item )
{
    if ( item->dialog == 0 && item->subclassed == 0 ) {
        unsigned bucket = 
            (unsigned)item->hwnd % (sizeof(_hash)/sizeof(*_hash));
        MapItem* mitem = _hash[bucket];
        MapItem** prev = &_hash[bucket];

        while( mitem ) {
            MapItem* next = mitem->next;
            if ( mitem == item ) {
                delete mitem;
                *prev = next;
                return;
                
            }
            
            prev = &mitem->next;
            mitem = mitem->next;
        }

        assert(0);
    }
}

static bool _baseWndInit = false;
double _mmX = 1.0; // mm per pixel
double _mmY = 1.0;
double _scaling = 1.0;
static HINSTANCE _ghInstance = 0;


#define MAX_MODELESS_DIALOGS 4
static HWND ModelessDialogs[MAX_MODELESS_DIALOGS] = 
{ 0,0,0,0 };

#define MAX_MODELESS_PROPERTY_SHEETS 4
static HWND ModelessPropertySheets[MAX_MODELESS_PROPERTY_SHEETS] = 
{ 0,0,0,0 };

// --------------------------------------------------------------------------
// Initialize the BaseWnd framework. The scaling parameter affects the values
// returned by the TwipsX(), TwipxY(), MillimetresX(), and MillimetresY()
// functions and is normally set to 1.0.
void
InitBaseWnd(HINSTANCE hInstance, double scaling)
{
    _baseWndInit = true;
    _ghInstance = hInstance;

#if 0
    HDC dc = GetDC(NULL);

    int old = GetMapMode( dc );
    SetMapMode( dc, MM_TWIPS );

    POINT p = {100, 100};
    DPtoLP( dc, &p, 1 );
    SetMapMode( dc, old );
    ReleaseDC( NULL, dc );

    if ( p.x < 0 ) {
        p.x = -p.x;
    }

    if ( p.y < 0 ) {
        p.y = -p.y;
    }

    _twipsX = p.x / 100;
    _twipsY = p.y / 100;

    _mmX = 25.4 / 1440.0 * p.x / 100;
    _mmY = 25.4 / 1440.0 * p.y / 100;

    SetMapMode( dc, old );
    ReleaseDC( NULL, dc );
#endif

    double screenX;
    double screenY;
    bool swapped = false;

    // get the width and height of the screen in pixels.
    screenX = GetSystemMetrics(SM_CXSCREEN);
    screenY = GetSystemMetrics(SM_CYSCREEN);
    
    // if the width is < height then swap them.
    if ( screenX < screenY ) {
        swapped = true;
        double temp = screenX;
        screenX = screenY;
        screenY = temp;
    }
    
    if ( screenX <= 1024 ) {
        // assume screen is on a 15 inch monitor. That's 286 mm x 215 mm 
        // calculate mmY and mmX
        // (Millimetres per Pixel)
        _mmX = 286.0 / screenX;
        _mmY = 215.0 / screenY;
    } else {
        // Hmmm... Screen is higher res. They probably have a bigger monitor.
        // Based on 19" monitor
        _mmX = 377.0 / screenX;
        _mmY = 302.0 / screenY;
    }

    _mmX /= scaling;
    _mmY /= scaling;

    // if we swapped, then swap the mmX and mmY
    if ( swapped ) {
        double temp = _mmX;
        _mmX = _mmY;
        _mmY = temp;
    }
}

// --------------------------------------------------------------------------
// Returns the module instance passed to InitBaseWnd()
HINSTANCE
GetInstance()
{
    assert( _baseWndInit );
    return _ghInstance;
}

// --------------------------------------------------------------------------
// Convert a twips value to pixels.
int
TwipsX(int twips)
{
    assert( _baseWndInit);
    return MillimetresX( (int)( (double)twips * 25.4 / 1440.0 ));
}

// --------------------------------------------------------------------------
// Convert a twips value to pixels.
int
TwipsY(int twips)
{
    assert( _baseWndInit);
    return MillimetresY( (int)((double)twips * 25.4 / 1440.0 ));
}

// --------------------------------------------------------------------------
// Convert a value in millimetres to pixels.
int
MillimetresX(int mm)
{
    assert( _baseWndInit);
    return (int)((double)mm / _mmX);
}

// --------------------------------------------------------------------------
// Convert a value in millimetres to pixels.
int 
MillimetresY(int mm)
{
    assert( _baseWndInit);
    return (int)((double)mm / _mmY);
}


// ---------------------------------------------------------------------------
// WindowFunctions

WindowFunctions::WindowFunctions()
{
    hwnd = 0;
}

WindowFunctions::WindowFunctions( HWND hwnd ) :
    hwnd(hwnd)
{
}

WindowFunctions::~WindowFunctions()
{
    hwnd = 0;
}

// Attach an instance of the WindowFunctions() class to a given window handle
void
WindowFunctions::attach(HWND hwndn)
{
    hwnd = hwndn;
}


// Object Oriented wrapper for MoveWindow().
void 
WindowFunctions::moveWindow( int x, int y, int width, int height, bool repaint)
{
    MoveWindow( hwnd, x, y, width, height, repaint ? TRUE : FALSE);
}

// Object Oriented wrapper to set the size of a window, without changing its
// position.
void
WindowFunctions::sizeWindow( int x, int y )
{
    RECT rect;
    GetWindowRect(hwnd, &rect);
    if ( ! MoveWindow( hwnd, rect.left, rect.top, x, y, TRUE ) ) {
        int a = GetLastError();
        assert(0);
    }
}


// Object Oriented wrapper to set the size of a window, and change its position
// at the same time.
void 
WindowFunctions::moveWindow( RECT* rect, bool repaint )
{
    RECT vrect = *rect;
    ValidateRect(vrect);
    MoveWindow(hwnd, vrect.left, vrect.top,
        vrect.right - vrect.left, vrect.bottom - vrect.top, repaint ? TRUE : FALSE );
}

// Object Oriented wrapper for GetClientRect()
RECT
WindowFunctions::getClientRect()
{
    RECT rect;
    GetClientRect( hwnd, &rect );
    return rect;
}

void 
WindowFunctions::invalidate()
{
    if ( hwnd ) {
        InvalidateRect( hwnd, NULL, FALSE );
    }
}

// Object Oriented wrapper for SetWindowText()
void
WindowFunctions::setWindowText( const _TCHAR* text )
{
    SetWindowText( hwnd, text );
}

// Returns true if the window is maximized.
bool
WindowFunctions::isMaximized()
{
    WINDOWPLACEMENT placement;
    if ( !GetWindowPlacement( hwnd, &placement ) ) {
        assert(0);
        return false;
    }

    return placement.showCmd == SW_MAXIMIZE ||
        placement.showCmd == SW_SHOWMAXIMIZED;
}

// Returns the rectangle of a child window.
RECT
WindowFunctions::getChildRect( HWND hChild )
{
    RECT crect;

    GetWindowRect( hChild, &crect );
    ScreenToClient( hwnd, (LPPOINT)&crect.left );
    ScreenToClient( hwnd, (LPPOINT)&crect.right );

    return crect;
}

// Returns a handle to the window's font.
HFONT 
WindowFunctions::getFont()
{
    HFONT hFont = (HFONT)SendMessage( hwnd, WM_GETFONT, 0, 0 );
    if ( hFont == NULL ) {
        // Control is using system font.
        hFont = (HFONT)GetStockObject( SYSTEM_FONT );
    }

    return hFont;
}

// Sets the window's font.
void
WindowFunctions::setFont( HFONT hFont )
{
    SendMessage( hwnd, WM_SETFONT, (WPARAM)hFont, MAKEWORD( TRUE, 0 ) );
}



// ---------------------------------------------------------------------------
// Window
//
// Sets default properties for a window. If you override this class, you should
// override these properties in the constructor.
Window::Window()
{
	//Set the default data for the window styles and WNDCLASS
	//These can be reset in the child class's constructor.

	memset(&_WndClass, 0, sizeof(_WndClass));
	
	_WndClass.cbSize = sizeof(_WndClass);
	_WndClass.style = CS_DBLCLKS;
	_WndClass.lpfnWndProc = BaseWndProc;
	_WndClass.cbClsExtra = 0;
	_WndClass.cbWndExtra = 0;
	_WndClass.hInstance = NULL; 
	_WndClass.hIcon = NULL;
	_WndClass.hCursor = NULL;
	_WndClass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	_WndClass.lpszMenuName = NULL;
	_WndClass.hIconSm = NULL;

	_dwExtendedStyle = NULL;
	_dwStyle = NULL;
	_pszClassName = TEXT("Window");
	_pszTitle = TEXT("");

	_hwndParent = NULL;

	_bLocalMessages = false;
    _hInstance = 0;
    _attached = false;
}

Window::~Window()
{
	_bLocalMessages = true;
    _hwndParent = NULL;
    _hInstance = 0;
    _pszClassName = 0;
    _pszTitle = 0;
    if ( !_attached && IsWindow(hwnd) ) {
		SetWindowLongPtr(hwnd, GWLP_USERDATA, 0);
        DestroyWindow(hwnd);
    }
}

// Object oriented wrapper for RegisterClassEx() and CreateWindow()
HWND Window::Create(int x, int y, int nWidth, int nHeight,
			HWND hParent, HMENU hMenu, HINSTANCE hInstance)
{

	_WndClass.lpszClassName = _pszClassName;
	_WndClass.hInstance = hInstance;
	_hInstance = hInstance;
	
	//If we're already registered, this call will fail.
	DWORD err = RegisterClassEx(&_WndClass);

	if ( err == 0 ) {
		err = GetLastError();
//		assert(0);
	}

	hwnd = CreateWindowEx(_dwExtendedStyle, _pszClassName,
		_pszTitle, _dwStyle, x, y,	nWidth, nHeight, hParent,
		hMenu, hInstance,	(void*)this);

	_hwndParent = hParent;

	return hwnd;
}

#pragma warning(disable:4311)
#pragma warning(disable:4312)
#pragma warning(disable:4244)

LRESULT CALLBACK Window::BaseWndProc(HWND phwnd, UINT msg,
						  WPARAM wParam, LPARAM lParam)
{
	//A pointer to the object is passed in the CREATESTRUCT
	if(msg == WM_NCCREATE)
	{
		SetWindowLongPtr(phwnd, GWLP_USERDATA,
			(LONG_PTR)((LPCREATESTRUCT)lParam)->lpCreateParams);
		Window* pObj = (Window*)((LPCREATESTRUCT)lParam)->lpCreateParams;
		pObj->hwnd = phwnd;
	}
	
	BOOL bProcessed = FALSE;
	LRESULT lResult = 0;
	
	//Retrieve the pointer
	Window *pObj = (Window *)GetWindowLongPtr(phwnd, GWLP_USERDATA);

	//Filter message through child classes
	if(pObj && !pObj->_bLocalMessages) {
		lResult = pObj->WindowProc(phwnd, msg, wParam, lParam,
				&bProcessed);
    }

	if(pObj && msg == WM_DESTROY)
		pObj->hwnd = NULL;

	if(bProcessed)
		return lResult;
	else
	{
		//If message was unprocessed, send it back to Windows.		
		return DefWindowProc(phwnd, msg, wParam, lParam);
	}
}

// Window handler to override and provide window specific behaviour. In the
// subclass, set the value of *pbProcessed to TRUE to indicate the message was
// processed. Set it to FALSE to invoke DefWindowProc().
LRESULT Window::WindowProc(HWND , UINT , WPARAM , LPARAM , PBOOL pbProcessed)
{
	//This may be overridden to process messages.
	*pbProcessed = FALSE;
	return NULL;
}

// ---------------------------------------------------------------------------
// SubclassedWindow
SubclassedWindow::SubclassedWindow()
{
    hwnd = 0;
    _oldWndProc = 0;
}

SubclassedWindow::~SubclassedWindow()
{
    SetWindowLongPtr(hwnd, GWLP_WNDPROC, (LONG)_oldWndProc);
    _oldWndProc = 0;
    if ( hwnd ) {
        _windowMap.removeSubclassed( hwnd, this );
    }
    hwnd = 0;
}

void
SubclassedWindow::subclass(HWND phwnd)
{
    assert( this->hwnd == 0 );
    this->hwnd = phwnd;
    _oldWndProc = (LONG)(LONG_PTR)SetWindowLongPtr(phwnd, GWLP_WNDPROC, 
            (LONG)BaseWndProc);
    _windowMap.mapSubclassed( phwnd, this );
}

LRESULT 
SubclassedWindow::WindowProc(HWND , UINT , WPARAM , LPARAM , PBOOL pbProcessed)
{
    *pbProcessed = FALSE;
    return 0;
}

LRESULT CALLBACK 
SubclassedWindow::BaseWndProc(HWND phwnd, UINT msg,
	WPARAM wParam, LPARAM lParam)
{
	BOOL bProcessed = FALSE;
	LRESULT lResult = 0;
	
	//Retrieve the pointer
	SubclassedWindow *pObj = _windowMap.getSubclassed( phwnd );
    if ( pObj == 0 ) {
        return DefWindowProc(phwnd, msg, wParam, lParam);
    } 

	//Filter message through child classes
    lResult = pObj->WindowProc(phwnd, msg, wParam, lParam,
            &bProcessed);

	if( msg == WM_DESTROY )
		pObj->hwnd = NULL;

	if(bProcessed)
		return lResult;
    else if ( pObj->_oldWndProc != 0 ) {
        return CallWindowProc((WNDPROC)pObj->_oldWndProc, phwnd, msg, 
                wParam, lParam);
    } else {
        return DefWindowProc(phwnd, msg, wParam, lParam);
    }
}


// ---------------------------------------------------------------------------
// Dialog

void
AddModelessDialog( HWND hwnd )
{
    for ( int i = 0; i < MAX_MODELESS_DIALOGS; i++ ) {
        if ( ModelessDialogs[i] == 0 ) {
            ModelessDialogs[i] = hwnd;
            return;
        }
    }

    // need more modeless dialog entries!!
    assert(0);
}

void 
RemoveModelessDialog( HWND hwnd )
{
    for ( int i = 0; i < MAX_MODELESS_DIALOGS; i++ ) {
        if ( ModelessDialogs[i] == hwnd ) {
            ModelessDialogs[i] = 0;
            break;
        }
    }
}

Dialog::Dialog( HINSTANCE hInstance, LPCTSTR pszTemplate, HWND hParent)
{
	//These can be reset in the child class's constructor.
    _hwndParent = hParent;
	_hInstance = hInstance;
    _pszTemplate = pszTemplate;
}

Dialog::~Dialog()
{
    if ( IsWindow(hwnd) ) {
        RemoveModelessDialog( hwnd );
        DestroyWindow(hwnd);
    }
    _hwndParent = 0;
    hwnd = 0;
    _hInstance = 0;
    _pszTemplate = 0;
}

int Dialog::DoModal()
{
	return DialogBoxParam(_hInstance, 
        _pszTemplate, 
        _hwndParent, (DLGPROC)BaseDialogProc, (LPARAM)this);
}

HWND Dialog::CreateModeless()
{
    hwnd = CreateDialogParam( _hInstance, 
        _pszTemplate, _hwndParent, (DLGPROC)BaseDialogProc,
        (LPARAM)this);
    if ( hwnd == 0 ) {
        int a = GetLastError();
        assert(0);
    } else {
        AddModelessDialog( hwnd );
    }
    return hwnd;
}

BOOL CALLBACK Dialog::BaseDialogProc( HWND phwnd, UINT msg, WPARAM wParam, 
								 LPARAM lParam)
{
	//A pointer to the object is passed in lParam of WM_INITDIALOG
	if(msg == WM_INITDIALOG)
	{
		SetWindowLong(phwnd, GWL_USERDATA, (LONG)lParam);
		Dialog *pObj = (Dialog *)lParam;
		pObj->hwnd = phwnd;
	}

	//Retrieve the pointer
	Dialog *pObj = (Dialog *)GetWindowLong(phwnd, GWL_USERDATA);

	//Filter message through child classes
	if(pObj)
		return pObj->DialogProc(phwnd, msg, wParam, lParam);
	else
		return FALSE;
}

LRESULT Dialog::DialogProc( HWND , UINT , WPARAM , LPARAM )
{
	//Override this.
	return FALSE;
}

bool
Dialog::getCheck(unsigned int id )
{
    return SendMessage(GetDlgItem(hwnd, id), BM_GETCHECK, 0, 0) 
        == BST_CHECKED;
}

void 
Dialog::setCheck( unsigned id, bool value )
{
    SendMessage(GetDlgItem(hwnd, id), BM_SETCHECK, 
        value ? (WPARAM)BST_CHECKED : (WPARAM)BST_UNCHECKED, 0);    
}

HWND 
Dialog::getDlgItem(unsigned id)
{
    return GetDlgItem(hwnd, id);    
}

bool
Dialog::isDialogMessage( LPMSG msg )
{
    for ( int i = 0; i < MAX_MODELESS_DIALOGS; i++ ) {
        if ( ModelessDialogs[i] ) {
            assert( IsWindow( ModelessDialogs[i] ) );
            if ( ::IsDialogMessage( ModelessDialogs[i], msg ) ) {
                return true;
            }
        }
    } 

    return false;
}

// ---------------------------------------------------------------------------
// PropertyPage
PropertyPage::PropertyPage(HINSTANCE hInstance, LPCTSTR pszTemplate) :
    Dialog(hInstance, pszTemplate, NULL )
{
    Dialog* dlg = this;
	memset(&_PageStruct, 0, sizeof(_PageStruct));
	_PageStruct.pszTemplate = _pszTemplate;
	_PageStruct.dwSize = sizeof(_PageStruct);
	_PageStruct.dwFlags = 0;
	_PageStruct.hInstance = 0;
	_PageStruct.pszTitle = _T(""); //MUST Override!
	_PageStruct.pfnDlgProc = (DLGPROC)BaseDialogProcProp;
	_PageStruct.lParam = (LPARAM)dlg;
	_PageStruct.pfnCallback = NULL;
	_PageStruct.hInstance = hInstance;
    _hInstance = hInstance;
}

PropertyPage::~PropertyPage()
{

}

BOOL CALLBACK 
PropertyPage::BaseDialogProcProp(HWND phwnd, UINT msg, WPARAM wParam,
														  LPARAM lParam)
{
	//A pointer to the object is passed in lParam of WM_INITDIALOG
	if(msg == WM_INITDIALOG)
	{
		SetWindowLong(phwnd, GWL_USERDATA, (LONG)lParam);
		PROPSHEETPAGE* pHeader = (PROPSHEETPAGE*)lParam;
		SetWindowLong(phwnd, GWL_USERDATA, (LONG)pHeader->lParam);
		PropertyPage *pObj = (PropertyPage*)pHeader->lParam;
		pObj->hwnd = phwnd;
	}

	//Retrieve the pointer
	PropertyPage *pObj = (PropertyPage *)GetWindowLong(phwnd, GWL_USERDATA);

	//Filter message through child classes
	if(pObj)
		return pObj->DialogProc(phwnd, msg, wParam, lParam);
	else
		return FALSE;
}

HPROPSHEETPAGE PropertyPage::create()
{
	HPROPSHEETPAGE hPage;
	hPage = CreatePropertySheetPage(&_PageStruct);
	return hPage;
}

bool PropertyPage::areChangesValid()
{
	return true;
}

bool PropertyPage::onApply()
{
	return true;
}


LRESULT 
PropertyPage::DialogProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam)
{
	switch(msg)
	{
		case WM_NOTIFY:
		{
			switch( ((NMHDR FAR *)lParam)->code )
			{
				case PSN_KILLACTIVE:
				{
					//say, "yes, the changes are valid. you can apply them now."
					SetWindowLong(hwnd, DWL_MSGRESULT, areChangesValid() ? FALSE : TRUE);
					return true;
				}
		
				case PSN_APPLY:
				{
					onApply();
					return true;
				}

				default:
					break;
			}
		}
	}

	return FALSE;
}

// ---------------------------------------------------------------------------
// PropertySheetDialog

void
AddModelessPropertySheet( HWND hwnd )
{
    for ( int i = 0; i < MAX_MODELESS_PROPERTY_SHEETS; i++ ) {
        if ( ModelessPropertySheets[i] == 0 ) {
            ModelessPropertySheets[i] = hwnd;
            return;
        }
    }

    // need more modeless dialog entries!!
    assert(0);
}

void 
RemoveModelessPropertySheet( HWND hwnd )
{
    for ( int i = 0; i < MAX_MODELESS_PROPERTY_SHEETS; i++ ) {
        if ( ModelessPropertySheets[i] == hwnd ) {
            ModelessPropertySheets[i] = 0;
            break;
        }
    }
}

bool 
PropertySheetDialog::isDialogMessage( LPMSG msg )
{
    for ( int i = 0; i < MAX_MODELESS_PROPERTY_SHEETS; i++ ) {
        if ( ModelessPropertySheets[i] ) {
            assert( IsWindow( ModelessPropertySheets[i] ) );
            if ( PropSheet_IsDialogMessage( ModelessPropertySheets[i], msg ) ) {
                return true;
            } else if ( PropSheet_GetCurrentPageHwnd( 
                            ModelessPropertySheets[i] ) ) 
            {
                HWND hwnd = ModelessPropertySheets[i];
                RemoveModelessPropertySheet( hwnd );

                DestroyWindow( hwnd );
                return true;    
            }
        }
    } 

    return false;
}

PropertySheetDialog::PropertySheetDialog(HINSTANCE hInstance)
{
	memset(&_PropSheetHeader, 0, sizeof(_PropSheetHeader));

	_PropSheetHeader.dwSize = sizeof(_PropSheetHeader);
	_PropSheetHeader.dwFlags = PSH_NOAPPLYNOW;
	_PropSheetHeader.hIcon = NULL;
	_PropSheetHeader.pszCaption = _title;
	_PropSheetHeader.nPages = 0;
	_PropSheetHeader.nStartPage = 0;
	_PropSheetHeader.phpage = _PageArray;
	_PropSheetHeader.pfnCallback = NULL;
	_PropSheetHeader.hInstance = hInstance;
    _hInstance = hInstance;

    _title[0] = 0;

	_iNumPages = 0;
}

PropertySheetDialog::~PropertySheetDialog()
{
    if ( IsWindow(hwnd) ) {
        RemoveModelessPropertySheet( hwnd );
    }
}
	
bool PropertySheetDialog::addPage(PropertyPage* pNewPage)
{
	HPROPSHEETPAGE hPage = pNewPage->create();

	if(!hPage)
		return FALSE;

	_PageArray[_iNumPages++] = hPage;
	return true;
}

void
PropertySheetDialog::setTitle( const _TCHAR* title )
{
    _tcscpy( _title, title );
}

void 
PropertySheetDialog::modelessEnded( bool okClicked )
{
}

int 
PropertySheetDialog::showModal(HWND hwndParent, int nStartPage)
{
	_PropSheetHeader.hwndParent = hwndParent;
	
	if(nStartPage >= 0)
		_PropSheetHeader.nStartPage = nStartPage;

	if(!_iNumPages)
		return 0;

	_PropSheetHeader.nPages = _iNumPages;

	int a = ::PropertySheet(&_PropSheetHeader);
	if ( a == -1 ) {
		DWORD err = GetLastError();
            // Fails if showModal() called twice in a row -- the
            // pages need to be recreated.
		int b = 0;
	}
	return a;
}

HWND
PropertySheetDialog::showModeless(HWND hwndParent, int nStartPage)
{
	_PropSheetHeader.hwndParent = hwndParent;
	
	if(nStartPage >= 0)
		_PropSheetHeader.nStartPage = nStartPage;

	if(!_iNumPages)
		return 0;

	_PropSheetHeader.nPages = _iNumPages;
    _PropSheetHeader.dwFlags |= PSH_MODELESS;

    hwnd = (HWND)::PropertySheet( &_PropSheetHeader );

    return hwnd;
}

// ---------------------------------------------------------------------------
// Menu
Menu::Menu(bool popUp)
{
	if ( popUp ) {
		hMenu = CreatePopupMenu();
	} else {
		hMenu = CreateMenu();
	}

    _destroy = true;
    _size = 0;
}

Menu::Menu()
{
    hMenu = CreatePopupMenu();
    _destroy = true;
    _size = 0;
}

Menu::Menu(HMENU menu) 
{
    hMenu = menu;
    _destroy = false;
    _size = 0;
}

Menu::~Menu()
{
    if ( _destroy ) {
	    DestroyMenu(hMenu);
    }
    hMenu = 0;
}

void 
Menu::append( const _TCHAR* text, Menu& menu )
{
	AppendMenu(hMenu, MF_STRING | MF_POPUP, (UINT)menu.hMenu, text);
    _size++;
}


void Menu::appendBar()
{
	AppendMenu( hMenu, MF_SEPARATOR, 0, 0);
    _size++;
}

void Menu::append( const _TCHAR* text, unsigned int id )
{
	AppendMenu(hMenu, MF_STRING, id, text);
    _size++;
}

void
Menu::setDefaultItem( unsigned int id )
{
    SetMenuDefaultItem(hMenu, id, FALSE);
}

void
Menu::removeAll()
{
    for( int i = 0; i < _size; i++ ) {
        RemoveMenu( hMenu, i, MF_BYPOSITION);
    }
    _size = 0;

}

void
Menu::setCheck( unsigned int id, bool checked)
{
    CheckMenuItem(hMenu, id, checked ? MF_BYCOMMAND | MF_CHECKED :
        MF_BYCOMMAND | MF_UNCHECKED);
}

void 
Menu::enable(unsigned int id, bool enabled)
{
    assert( GetMenuItemCount(hMenu) > 0 );
    assert( IsMenu(hMenu) );
    // dbgprint((DWND, "Menu::enable([%08x] %d, %d)", hMenu, id, enabled));
    EnableMenuItem(hMenu, id, enabled ? (MF_BYCOMMAND | MF_ENABLED) :
        (MF_BYCOMMAND | MF_GRAYED));
    // dbgprint((DWND, "Ret: %d LastError: %d", Ret, GetLastError()));
}

void 
Menu::setText(unsigned id, const _TCHAR* text )
{
    MENUITEMINFO info;
    memset( &info, 0, sizeof(info) );
    info.cbSize = sizeof(info);
    info.fMask = MIIM_TYPE;
    info.fType = MFT_STRING;
    info.dwTypeData = (LPTSTR)text;
    info.cch = _tcslen( text );
    SetMenuItemInfo( hMenu, id, FALSE, &info );
}


// ---------------------------------------------------------------------------
// FileSelectDialog
FileSelectDialog::FileSelectDialog(HWND hwnd, const _TCHAR* title)
{
	_filter[0] = 0;
	_filter[1] = 0;
	_filterLen = 2;
	memset(&_openFileName, 0, sizeof(_openFileName));

	_openFileName.lStructSize = sizeof(OPENFILENAME);
	_openFileName.hwndOwner = hwnd;
	_openFileName.hInstance = 0;
	_openFileName.lpstrFilter = _filter;
	_openFileName.lpstrCustomFilter = NULL;
	_openFileName.nMaxCustFilter = 0;
	_openFileName.nFilterIndex = 1;
	//_openFileName.lpstrFile = szFiles;
	//_openFileName.nMaxFile = 10000;
	_openFileName.lpstrFileTitle = NULL; //szFileName;
	_openFileName.nMaxFileTitle = 255;
	_openFileName.lpstrInitialDir = NULL; //szFilePath;
	_openFileName.lpstrTitle = title;
	_openFileName.Flags = OFN_EXPLORER;
	//_openFileName.Flags = OFN_PATHMUSTEXIST | OFN_ALLOWMULTISELECT | OFN_EXPLORER | OFN_HIDEREADONLY;
	_openFileName.nFileOffset = 0;
	_openFileName.nFileExtension = 0;
	_openFileName.lpstrDefExt = NULL;
    _multisel = 0;
    _buffer = 0;
    _pos = 0;
    _bufferLen = 0;
}

FileSelectDialog::~FileSelectDialog()
{
    delete[] _multisel;
    _buffer = 0;
}

bool 
FileSelectDialog::getOpenFileName(TCHAR* buffer, int bufferLen, 
        bool multiselect)
{
	// buffer[0] = 0;
	_openFileName.lpstrFile = buffer;
	_openFileName.nMaxFile = bufferLen;
	if ( multiselect ) {
		_openFileName.Flags |= OFN_ALLOWMULTISELECT;
	}
	_openFileName.Flags |= OFN_PATHMUSTEXIST;

    _buffer = buffer;
    _bufferLen = bufferLen;
    _pos = 0;

	if ( !GetOpenFileName(&_openFileName) ) {
		DWORD err = GetLastError();
        err = CommDlgExtendedError();
        // Did you forget to set buffer[0] = 0 or a valid file??
		return false;
	}

	return true;
}

const _TCHAR* 
FileSelectDialog::nextMultiSelectFile()
{
    bool wasFirst = false;
    if ( _multisel == 0 ) {
        _multisel = new _TCHAR[MAX_PATH*4];
    }

    if ( _buffer == 0 ) {
        assert(0);
        return 0;
    }
    
    if ( _pos >= _bufferLen ) {
        return 0;
    }

    _tcscpy( _multisel, _buffer );

    if ( _pos == 0 ) {
        wasFirst = true;
        _pos = _tcslen( _multisel ) + 1;
    }

    if ( _buffer[_pos] ) {
        AppendSlash( _multisel );
        _tcscat( _multisel, &_buffer[_pos] );
        _pos += _tcslen(&_buffer[_pos]) + 1;
        assert( _pos <= _bufferLen );
        return _multisel;
    } else if ( wasFirst ) {
        return _multisel;
    }

    return 0;
}

bool
FileSelectDialog::getSaveFileName(TCHAR* buffer, int bufferlen)
{
	_openFileName.lpstrFile = buffer;
	_openFileName.nMaxFile = bufferlen;
//	buffer[0] = 0;

	_openFileName.Flags |= OFN_HIDEREADONLY;

	if ( !GetSaveFileName(&_openFileName) ) {
		//DWORD dwErr = CommDlgExtendedError();
		// CDERR_STRUCTSIZE
		return false;
	}

	return true;
}

void
FileSelectDialog::addFilter(const _TCHAR* name, const _TCHAR* filter)
{
	if ( _tcslen( name ) + _tcslen( filter ) + 2 + _filterLen >
		sizeof(_filter) ) 
	{
		assert(0);
		return;
	}

	int insertStart = 0;

	if ( _filterLen > 2 ) {
		insertStart = _filterLen;
	}

	_tcscpy(&_filter[insertStart], name);
	insertStart += (int)_tcslen(name) + 1;
	_tcscpy(&_filter[insertStart], filter );
	_filterLen = insertStart + (int)_tcslen(filter) + 1;
	_filter[_filterLen] = 0;
}

// ---------------------------------------------------------------------------
// Accelerators
Accelerators::Accelerators()
{
	_haccel = 0;
	items = 0;
    memset( _array, 0, sizeof(_array));
	_registered = false;
}

Accelerators::~Accelerators()
{
	if ( _haccel ) {
		DestroyAcceleratorTable(_haccel);
	}
    _haccel = 0;
}

HACCEL Accelerators::haccel()
{
	if ( _haccel == 0 ) {
		_haccel = CreateAcceleratorTable(&_array[0], items);
		assert(_haccel);
	}

	return _haccel;
}

void Accelerators::add(BYTE fVirt, WORD key, WORD cmd)
{
	if ( _haccel ) {
		DestroyAcceleratorTable(_haccel);
		_haccel = 0;
	}

	assert( items < sizeof(_array)/sizeof(*_array) );
	_array[items].fVirt = fVirt;
	_array[items].key = key;
	_array[items].cmd = cmd;

	items++;
}

struct AcceleratorList {
    HWND hwnd;
    Accelerators* accel;
} _acceleratorList[10];

static unsigned _numAccelerators = 0;

void
Accelerators::registerWindow( HWND hwnd )
{
    if ( _numAccelerators == 
            sizeof(_acceleratorList)/sizeof(*_acceleratorList) ) {
        assert( 0 );
        return;
    }

    if ( _registered ) {
        return;
    }

    _acceleratorList[_numAccelerators].hwnd = hwnd;
    _acceleratorList[_numAccelerators].accel = this;

    _numAccelerators++;
    _registered = true;
}

void 
Accelerators::unregister()
{
    if ( !_registered ) {
        return;
    }

    bool found = false;
    assert( _numAccelerators > 0 );

    for( unsigned i = 0; i < _numAccelerators - 1; i++ ) 
    {
        if ( !found ) {
            if ( _acceleratorList[i].accel == this ) {
                found = true;
            }
        }
            
        if ( found ) {
            _acceleratorList[i] = _acceleratorList[i+1];
        }
    }

    _numAccelerators--;
    _registered = false;
}

bool
Accelerators::translate(MSG* msg)
{
    bool found = false;

    for( unsigned i = 0; i < _numAccelerators; i++ ) {
        assert( IsWindow( _acceleratorList[i].hwnd ) );
        if ( TranslateAccelerator( _acceleratorList[i].hwnd, 
                    _acceleratorList[i].accel->haccel(),
                    msg ) ) 
        {
            found = true;
        }
    }

    return found;
}


// ---------------------------------------------------------------------------
// TrackBarControl
// 
TrackBarControl::TrackBarControl()
{

}

TrackBarControl::~TrackBarControl()
{

}

HWND 
TrackBarControl::create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
        unsigned id )
{
    hwnd = CreateWindowEx( 
        0,                             // no extended styles 
        TRACKBAR_CLASS,                // class name 
        "Trackbar Control",            // title (caption) 
        style|WS_CHILD | WS_VISIBLE,
        10, 10,                        // position 
        200, 30,                       // size 
        hParent,                      // parent window 
        (HMENU)id,             // control identifier 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL                           // no WM_CREATE parameter 
        ); 
    return hwnd;
}


int TrackBarControl::getPos()
{
    return SendMessage(hwnd, TBM_GETPOS, 0, 0);
}


void TrackBarControl::setPos( int pos )
{
    SendMessage(hwnd, TBM_SETPOS, TRUE, (LPARAM)pos);
}

void TrackBarControl::setRange( int minimum, int maximum )
{
    SendMessage(hwnd, TBM_SETRANGE, TRUE, MAKELONG( minimum, maximum ) );
}


// ---------------------------------------------------------------------------
// EditControl
//
EditControl::EditControl()
{

}

EditControl::~EditControl()
{

}

HWND 
EditControl::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    hwnd = CreateWindow(
        TEXT("EDIT"),     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | style, 
        x, y, cx, cy, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 
    return hwnd;
}

// ---------------------------------------------------------------------------
// StaticControl
//
StaticControl::StaticControl()
{

}

StaticControl::~StaticControl()
{

}

HWND 
StaticControl::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    hwnd = CreateWindow(
        TEXT("STATIC"),     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | style, 
        x, y, cx, cy, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 
    return hwnd;
}

// ---------------------------------------------------------------------------
// UpDownControl
//
UpDownControl::UpDownControl()
{

}

UpDownControl::~UpDownControl()
{

}

HWND 
UpDownControl::create( unsigned style, HWND hParent, unsigned id )
{
    hwnd = CreateWindow(
        UPDOWN_CLASS,     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | WS_BORDER | style, 
        0, 0, 100, 100, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 
    return hwnd;
}

void 
UpDownControl::setBuddy( HWND hBuddy )
{
    SendMessage( hwnd, UDM_SETBUDDY, (WPARAM)hBuddy, 0 );
}

#ifdef UDM_SETPOS32
void
UpDownControl::setRange( int iLow, int iHigh )
{
    SendMessage( hwnd, UDM_SETRANGE32, (WPARAM)iLow, (LPARAM)iHigh );
}

void
UpDownControl::setPos( int pos )
{
    SendMessage( hwnd, UDM_SETPOS32, 0, (LPARAM)pos );
}

bool
UpDownControl::getPos(int* pos)
{
    BOOL Error;
    *pos = SendMessage( hwnd, UDM_GETPOS32, 0, &Error );

    return Error != 0;
}

#else
void
UpDownControl::setRange( short iLow, short iHigh )
{
    SendMessage( hwnd, UDM_SETRANGE, (WPARAM)0, (LPARAM)MAKELONG(iHigh, iLow) );
}

void
UpDownControl::setPos( short pos )
{
    SendMessage( hwnd, UDM_SETPOS, 0, (LPARAM)MAKELONG(pos, 0) );
}

bool
UpDownControl::getPos(short* pos)
{
    LPARAM lParam = SendMessage( hwnd, UDM_GETPOS, 0, 0 );

    if ( lParam & 0xffff0000 ) {
        DWORD err = GetLastError();
        return false;
    }

    *pos = (short)lParam;
    return true;
}

#endif

// ---------------------------------------------------------------------------
// RichEditControl
//
RichEditControl::RichEditControl()
{

}

RichEditControl::~RichEditControl()
{

}

void 
RichEditControl::setBkColour(COLORREF clr)
{
    SendMessage( hwnd, EM_SETBKGNDCOLOR, 0, clr );
}

HWND 
RichEditControl::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    // ES_MULTILINE
    // ES_READONLY
    static bool loaded = false;
    
    if ( ! loaded ) {
        LoadLibraryA("RICHED32.DLL");
        loaded = true;
    }
    
    hwnd = CreateWindowEx(
        0, //extended style
        TEXT("RICHEDIT"),     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | style, 
        x, y, cx, cy, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 
    return hwnd;
}

// ---------------------------------------------------------------------------
// ProgressBarControl
//
ProgressBarControl::ProgressBarControl()
{

}

ProgressBarControl::~ProgressBarControl()
{

}

void 
ProgressBarControl::setBkColour(COLORREF clr)
{
    SendMessage( hwnd, PBM_SETBKCOLOR, 0, clr );
}

void 
ProgressBarControl::setBarColour(COLORREF clr)
{
    SendMessage( hwnd, PBM_SETBARCOLOR, 0, clr );
}

void
ProgressBarControl::setRange( int lower, int higher )
{
    SendMessage( hwnd, PBM_SETRANGE, 0, MAKELPARAM( lower, higher ) );
}

void
ProgressBarControl::setPos( int pos )
{
    SendMessage( hwnd, PBM_SETPOS, pos, 0 );
}

HWND 
ProgressBarControl::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    hwnd = CreateWindowEx(
        0, //extended style
        PROGRESS_CLASS,     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | style, 
        x, y, cx, cy, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 

    return hwnd;
}

// ---------------------------------------------------------------------------
// ListViewControl
//
ListViewControl::ListViewControl()
{
}

ListViewControl::~ListViewControl()
{
}

HWND 
ListViewControl::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    hwnd = CreateWindow(WC_LISTVIEW, NULL, 
        WS_CHILD | WS_VISIBLE | style,
        x, y,
        cx,
        cy,
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed

    return hwnd;
}

void
ListViewControl::setCurSel(int index)
{
    ListView_SetItemState( hwnd, index, LVIS_SELECTED | LVIS_FOCUSED, 0xff );
}

int
ListViewControl::getNextSel(int iStart)
{
    return ListView_GetNextItem( hwnd, iStart, LVNI_SELECTED );
}

void
ListViewControl::setExtendedStyle(unsigned style)
{
    ListView_SetExtendedListViewStyle(hwnd, style);
}

void 
ListViewControl::insertItem( int index, const _TCHAR* text, void* data )
{
    LVITEM item;
    memset( &item, 0, sizeof(item) );

    item.mask = LVIF_PARAM | LVIF_TEXT;
    item.pszText = (TCHAR*)text;
    item.lParam = (LPARAM)data;
    item.iItem = index;
    item.iSubItem = 0;

    ListView_InsertItem( hwnd, &item ); 
}

void 
ListViewControl::setItemText( int index, int subItem, const _TCHAR* text )
{
    LVITEM item;
    memset( &item, 0, sizeof(item) );

    item.mask = LVIF_TEXT;
    item.pszText = (TCHAR*)text;
    item.iItem = index;
    item.iSubItem = subItem;

    ListView_SetItem( hwnd, &item ); 
}

void 
ListViewControl::setBkColour( COLORREF colour )
{
    ListView_SetBkColor( hwnd, colour );
}

void
ListViewControl::insertColumn( int iCol, int cx, const _TCHAR* text )
{
    LVCOLUMN col;
    memset( &col, 0, sizeof(col) );

    col.mask = LVCF_TEXT | LVCF_WIDTH;
    col.cx = cx;
    col.pszText = (TCHAR*)text;

    ListView_InsertColumn( hwnd, iCol, &col );
}

void
ListViewControl::deleteColumn( int iCol )
{
    ListView_DeleteColumn( hwnd, iCol );
}

void
ListViewControl::setColumnWidth( int iCol, int width )
{
    ListView_SetColumnWidth( hwnd, iCol, width );
}

int
ListViewControl::getColumnWidth( int iCol )
{
    return ListView_GetColumnWidth( hwnd, iCol );    
}

int
ListViewControl::getCount()
{
    return ListView_GetItemCount(hwnd);
}

void
ListViewControl::clear()
{
    ListView_DeleteAllItems( hwnd );

}

// ---------------------------------------------------------------------------
// HotkeyControl
//
HotkeyControl::HotkeyControl()
{

}

HotkeyControl::~HotkeyControl()
{

}

HWND 
HotkeyControl::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    hwnd = CreateWindow(
        HOTKEY_CLASS,     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | style, 
        x, y, cx, cy, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 

    if ( hwnd ) {
        SendMessage( hwnd, HKM_SETRULES, 
                HKCOMB_NONE,
                HOTKEYF_ALT );
    }

    return hwnd;
}

void
HotkeyControl::setHotkey( unsigned key )
{
    SendMessage( hwnd, HKM_SETHOTKEY, key, 0 );
}

unsigned
HotkeyControl::getHotkey()
{
    return SendMessage( hwnd, HKM_GETHOTKEY, 0, 0 );
}

// ---------------------------------------------------------------------------
// TabControl
TabControl::TabControl()
{

}

TabControl::~TabControl()
{
}

void
TabControl::insertItem( int index,  const _TCHAR* text )
{
    TCITEM tie;

    tie.mask = TCIF_TEXT | TCIF_IMAGE;
    tie.iImage = -1;
    tie.pszText = (TCHAR*)text;
    TabCtrl_InsertItem( hwnd, index, &tie );
}

int 
TabControl::getCurSel()
{
    return TabCtrl_GetCurSel( hwnd );
}

// ---------------------------------------------------------------------------
// ComboBox
ComboBox::ComboBox()
{

}

ComboBox::~ComboBox()
{

}

HWND 
ComboBox::create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
        unsigned id )
{
    hwnd = CreateWindowEx( 
        0,                             // no extended styles 
        "COMBOBOX",                // class name 
        "Trackbar Control",            // title (caption) 
        style | WS_CHILD | WS_VISIBLE,
        10, 10,                        // position 
        200, 30,                       // size 
        hParent,                      // parent window 
        (HMENU)id,             // control identifier 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL                           // no WM_CREATE parameter 
        ); 
    
    return hwnd;
}

void
ComboBox::clear()
{
    SendMessage( hwnd, CB_RESETCONTENT, 0, 0 );
}

int
ComboBox::getCount()
{
    int result = SendMessage( hwnd, CB_GETCOUNT, 0, 0 );
    if ( result == CB_ERR ) {
        result = 0;
    }
    return result;
}

int
ComboBox::addString( const _TCHAR* text )
{
    int ret = SendMessage( hwnd, CB_ADDSTRING, 0, (LPARAM)text );

    if ( ret == CB_ERR ) {
        return -1;
    } else {
        return ret;
    }
}

int
ComboBox::getCurSel()
{
    return SendMessage( hwnd, CB_GETCURSEL, 0, 0 );
}

void
ComboBox::setCurSel(int sel)
{
    SendMessage( hwnd, CB_SETCURSEL, sel, 0 );
}

TCHAR*
ComboBox::getText( int sel )
{
    if ( !IsWindow( hwnd ) ) {
        return 0;

    }

    int len = SendMessage( hwnd, CB_GETLBTEXTLEN, sel, 0 );
    if ( len == CB_ERR ) {
        return 0;
    }

    _TCHAR* str = (TCHAR*)malloc((len+1)*sizeof(TCHAR));
    if ( CB_ERR == SendMessage( hwnd, CB_GETLBTEXT, sel, (LPARAM)str ) ) {
        free( str );
        return 0;
    }

    return str;
}

// ---------------------------------------------------------------------------
// StatusBar
StatusBar::StatusBar()
{
    hwnd = 0;
}

StatusBar::~StatusBar()
{
    if ( hwnd ) {
        DestroyWindow( hwnd );
    }
}

HWND 
StatusBar::create( LONG style, LPCTSTR lpszText, HWND hwndParent, unsigned id )
{
    hwnd = CreateStatusWindow( style | WS_CHILD | WS_VISIBLE,
            lpszText, hwndParent, id );
    return hwnd;
}



// ---------------------------------------------------------------------------
// ListBox
ListBox::ListBox()
{

}

ListBox::~ListBox()
{

}

HWND 
ListBox::create( unsigned style, int x, int y, int cx, int cy, 
        HWND hParent, unsigned id )
{
    hwnd = CreateWindow(
        TEXT("LISTBOX"),     // predefined class 
        NULL,       // no window title 
        WS_CHILD | WS_VISIBLE | style, 
        x, y, cx, cy, // set size in WM_SIZE message 
        hParent,       // parent window 
        (HMENU) id, // edit control ID 
        (HINSTANCE) GetWindowLong(hParent, GWL_HINSTANCE), 
        NULL);                // pointer not needed 
    return hwnd;
}

void
ListBox::clear()
{
    SendMessage( hwnd, LB_RESETCONTENT, 0, 0 );
}

void
ListBox::addString( const _TCHAR* text )
{
    SendMessage( hwnd, LB_ADDSTRING, 0, (LPARAM)text );
}

void
ListBox::addString( const _TCHAR* text, void* data )
{
    LRESULT index = SendMessage( hwnd, LB_ADDSTRING, 0, (LPARAM)text );
    if ( index == LB_ERR || index == LB_ERRSPACE ) {
        return;
    }

    SendMessage( hwnd, LB_SETITEMDATA, (WPARAM)index, (LPARAM)data );
}

int
ListBox::getCount()
{
    return SendMessage( hwnd, LB_GETCOUNT, 0, 0 );
}


void*
ListBox::getItemData( int index )
{
    assert( index < getCount() );
    return (void*)SendMessage( hwnd, LB_GETITEMDATA, index, 0 );
}

int 
ListBox::findStringExact( const _TCHAR* text )
{
    LRESULT result = SendMessage( hwnd, LB_FINDSTRINGEXACT, -1, 
            (LPARAM)text );

    if ( result == LB_ERR ) {
        result = -1;
    }

    return (int)result;
}

void
ListBox::deleteString( int index )
{
    SendMessage( hwnd, LB_DELETESTRING, (WPARAM)index, 0 );
}

int
ListBox::getCurSel()
{
    return SendMessage( hwnd, LB_GETCURSEL, 0, 0 );
}

void
ListBox::setCurSel(int sel)
{
    SendMessage( hwnd, LB_SETCURSEL, sel, 0 );
}

TCHAR*
ListBox::getText( int sel )
{
    if ( !IsWindow( hwnd ) ) {
        return 0;

    }

    int len = SendMessage( hwnd, LB_GETTEXTLEN, sel, 0 );
    if ( len == CB_ERR ) {
        return 0;
    }

    _TCHAR* str = (TCHAR*)malloc((len+1)*sizeof(TCHAR));
    if ( CB_ERR == SendMessage( hwnd, LB_GETTEXT, sel, (LPARAM)str ) ) {
        free( str );
        return 0;
    }

    return str;
}

// ---------------------------------------------------------------------------
// ScreenDC
ScreenDC::ScreenDC() :
    DeviceContext( ::GetDC( NULL ) )
{
    _release = true;
}

// ---------------------------------------------------------------------------
// DeviceContext
DeviceContext::DeviceContext( HDC pdc ) :
    dc(pdc)
{
    _hOldFont = 0;
    _hOldPen = 0;
    _hOldBrush = 0;
    _release = false;
}

DeviceContext::DeviceContext( const DeviceContext& other )
{
    *this = other;
    _release = false;
}

DeviceContext::~DeviceContext()
{
    if ( _hOldFont ) {
        SelectObject(dc, _hOldFont);
        _hOldFont = 0;
    }
    if ( _hOldPen ) {
        SelectObject(dc, _hOldPen );
        _hOldPen = 0;
    }
    if ( _hOldBrush ) {
        SelectObject(dc, _hOldBrush );
        _hOldBrush = 0;
    }
    if ( _release ) {
        ReleaseDC( NULL, dc );
    }
    dc = 0;
}

DeviceContext::operator HDC() 
{
    return dc;
}

void
DeviceContext::selectFont(HFONT hFont)
{
    if ( _hOldFont ) {
        SelectObject(dc, hFont);
    } else {
        _hOldFont = (HFONT)SelectObject(dc, hFont);
    }
}

void
DeviceContext::selectPen(HPEN hPen)
{
    if ( _hOldPen ) {
        SelectObject(dc, hPen);
    } else {
        _hOldPen = (HPEN)SelectObject(dc, hPen);
    }
}

void
DeviceContext::selectBrush(HBRUSH hBrush)
{
    if ( _hOldBrush ) {
        SelectObject(dc, hBrush);
    } else {
        _hOldBrush = (HBRUSH)SelectObject(dc, hBrush);
    }
}

// ---------------------------------------------------------------------------
// Font

Font::Font()
{
    construct(TEXT("Tahoma"), 10);
}

Font::Font( HFONT hFont )
{
    construct(TEXT("Tahoma"), 10);
    GetObject( hFont, sizeof(_logfont), &_logfont );
}

Font::Font(const _TCHAR* FaceName, int height)
{
    construct(FaceName, height);
}

Font::Font( const Font& other )
{
    _logfont = other._logfont;
    _hFont = 0;
}

void
Font::construct(const _TCHAR* FaceName, int height)
{
    if ( height < -5000 || height > 5000 ) { 
        assert( height >= -500 && height <= 500 );
        int a = 0;
    }
    ZeroMemory(&_logfont, sizeof(_logfont));
    _logfont.lfWeight = FW_DONTCARE;
    _logfont.lfCharSet = DEFAULT_CHARSET;
    _logfont.lfOutPrecision = OUT_DEFAULT_PRECIS;
    _logfont.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    _logfont.lfQuality = DEFAULT_QUALITY;
    _logfont.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;
    _logfont.lfHeight = height;
    _underline = false;
    
    _tcscpy( _logfont.lfFaceName, FaceName);

    _hFont = 0;
    _tm.tmHeight = 0;
}

Font::~Font()
{
    destroy();
}

Font::operator HFONT() const
{
    if ( _hFont == 0 ) {
        _hFont = CreateFontIndirect(&_logfont);
    }

    return _hFont;
}

const LOGFONT* 
Font::logfont() const
{
    return &_logfont;
}

bool 
Font::chooseFontDialog(HWND hwnd)
{
    CHOOSEFONT choose;

    ZeroMemory(&choose, sizeof(choose));
    choose.lStructSize = sizeof(choose);
    choose.hwndOwner = hwnd;
    choose.lpLogFont = &_logfont;
    choose.Flags = CF_INITTOLOGFONTSTRUCT | CF_FORCEFONTEXIST | CF_SCREENFONTS;

    if ( ::ChooseFont(&choose) ) {
        destroy();
        return true;
    } 

    return false;
}

void 
Font::destroy()
{
    if ( _hFont ) {
        DeleteObject(_hFont);
        _hFont = 0;
        _tm.tmHeight = 0;
    }
}

void
Font::getTm() const
{
    if ( _tm.tmHeight == 0 ) {
        HDC dc = GetDC( NULL );
        HFONT hOldFont = (HFONT)SelectObject( dc, (HFONT)*this);
        GetTextMetrics( dc, &_tm );
        SelectObject( dc, hOldFont );
        ReleaseDC( NULL, dc );
    }
}

void 
Font::setFaceName(const _TCHAR* name)
{
    destroy();
    _tcscpy( _logfont.lfFaceName, name );
}

int
Font::getExternalLeading() const
{
    getTm();
    return _tm.tmExternalLeading;
}

int
Font::getAscent() const
{
    getTm();
    return _tm.tmAscent;
}

int
Font::getDescent() const
{
    getTm();
    return _tm.tmDescent;
}

int
Font::getInternalLeading() const
{
    getTm();
    return _tm.tmInternalLeading;
}

void
Font::setLogHeight( int height )
{
    destroy();
    _logfont.lfHeight = height;
}

void
Font::setBold( bool bold )
{
    destroy();
    if ( bold ) {
        _logfont.lfWeight = FW_BOLD;
    } else {
        _logfont.lfWeight = FW_DONTCARE;
    }
}

void
Font::setUnderline( bool underline )
{
    _underline = underline;
}

bool
Font::getUnderline() const
{
    return _underline;
}

void
Font::setItalic( bool italic )
{
    destroy();
    _logfont.lfItalic = italic;
}

int
Font::getCharWidth( _TCHAR ch )
{
    HDC dc = GetDC( NULL );
    HFONT hOldFont = (HFONT)SelectObject( dc, (HFONT)*this);

#if 0
    ABCFLOAT abc;
    GetCharABCWidthsFloat( dc, ch, ch, &abc );

    // does not work on win95/98
    if ( (int)(abc.abcfA + abc.abcfB + abc.abcfC) < -1000 ) {
        assert(0);
    }
#endif
    int w = 10;

    GetCharWidth( dc, ch, ch, &w );
    SelectObject( dc, hOldFont );
    ReleaseDC( NULL, dc );

    return w;
}

const _TCHAR*
Font::getFaceName() const
{
    return _logfont.lfFaceName;
}

int
Font::getHeight() const
{
    getTm();
    return _tm.tmHeight;
}

int
Font::getLogHeight() const
{
    // If logfont.lfHeight is < 0, then it is really the logfont height minus internal leading.
    return _logfont.lfHeight;
}

bool
Font::getBold() const
{
    return _logfont.lfWeight == FW_BOLD;
}

bool
Font::getItalic() const 
{
    return _logfont.lfItalic != FALSE;
}

// ---------------------------------------------------------------------------
// Brush
Brush::Brush(COLORREF colour)
{
    _hBrush = CreateSolidBrush(colour);
}

Brush::Brush()
{
    _hBrush = CreateSolidBrush(0);
}


Brush::~Brush()
{
    DeleteObject(_hBrush);
    _hBrush = 0;
}


void
Brush::setColour( COLORREF clr )
{
    DeleteObject(_hBrush);
    _hBrush = CreateSolidBrush( clr );
}


Brush::operator HBRUSH() 
{
    return _hBrush;
}


// ---------------------------------------------------------------------------
// Pen
//
Pen::Pen(int style, int width, COLORREF colour)
{
    _hPen = CreatePen(style, width, colour);
}

Pen::~Pen()
{
    DeleteObject(_hPen);
    _hPen = 0;
}

Pen::operator HPEN() 
{
    return _hPen;
}

// ---------------------------------------------------------------------------
// RegistryKey

RegistryKey::RegistryKey(HKEY hParent)
{
    hKey = hParent;
    _parent = 0;
    _bClose = false;
    _bCloseParent = false;
    _name = 0;
}

RegistryKey::~RegistryKey()
{
    if ( _bClose ) {
	    RegCloseKey(hKey);
    }

    if ( _bCloseParent ) {
        RegCloseKey( _parent );
    }

    free(_name);
    _parent = 0;
}

bool
RegistryKey::becomeSubKey( const _TCHAR* subKeyName )
{
    HKEY hSubKey = NULL;
    LONG lError = RegOpenKeyEx( hKey, subKeyName, 0, KEY_READ, &hSubKey );

    if ( lError == ERROR_SUCCESS ) {
        if ( _bClose ) {
            RegCloseKey(hKey);
        }
        free( _name );
        _name = _tcsdup( subKeyName );
        _bCloseParent = _bClose;
        _bClose = true;
        hKey = hSubKey;
        return true;
    }

    return false;
}

RegistryKey::RegistryKey(RegistryKey& parent, const _TCHAR* SubKey)
{
    DWORD dwDisposition;	//useless windows thing
    _parent = parent.hKey;

    if(RegCreateKeyEx( parent.hKey, SubKey, 0, NULL, REG_OPTION_NON_VOLATILE,
        KEY_ALL_ACCESS, NULL, &hKey, &dwDisposition) != ERROR_SUCCESS) 
    {
        assert(0);        
    }
    _bClose = true;
    _bCloseParent = false;
    _name = _tcsdup(SubKey);
}

bool 
RegistryKey::hasSubKey( const _TCHAR* subKey )
{
    HKEY hSubKey = NULL;
    LONG lResult = RegOpenKey( hKey, subKey, &hSubKey ); 

    if ( lResult == ERROR_SUCCESS ) {
        RegCloseKey( hSubKey );
        return true;
    }

    return false;
}

void
RegistryKey::writeValue(const _TCHAR* subKey, const _TCHAR* value)
{
	unsigned long Size = (unsigned long)(_tcslen(value) + 1) * sizeof(_TCHAR);

	 RegSetValueEx(hKey, subKey, 0, REG_SZ, (unsigned char*)value, Size);
}

bool 
RegistryKey::readValue(const _TCHAR* subKey, _TCHAR* value, int size)
{
    DWORD Type;
    DWORD dsize = size;
    _TCHAR* str = (TCHAR*)malloc((size+1)*sizeof(TCHAR));
    if ( str == 0 ) {
        return false;
    }

    *str = 0;

	long lError = RegQueryValueEx(hKey, subKey, 0, &Type, 
            (unsigned char*)str, &dsize);
    
    if ( Type == REG_SZ && lError == ERROR_SUCCESS ) {
        _tcscpy( value, str );
        free(str);
        return true;
    }

    free( str );
    
    return false;
}
    

void
RegistryKey::writeValue(const _TCHAR* subKey, long value)
{
	RegSetValueEx(hKey, subKey, 0, REG_DWORD, 
            (unsigned char*)&value, sizeof(value)); 
}

void
RegistryKey::writeValue(const _TCHAR* subKey, int value)
{
    long lvalue = value;
	RegSetValueEx(hKey, subKey, 0, REG_DWORD, 
            (unsigned char*)&lvalue, sizeof(lvalue)); 
}

void
RegistryKey::writeValue(const _TCHAR* subKey, bool value)
{
    long lvalue = value;
	RegSetValueEx(hKey, subKey, 0, REG_DWORD, 
            (unsigned char*)&lvalue, sizeof(lvalue)); 
}

void
RegistryKey::writeValue(const _TCHAR* subKey, double value)
{
	RegSetValueEx(hKey, subKey, 0, REG_BINARY, 
            (unsigned char*)&value, sizeof(value)); 
}

void
RegistryKey::writeValue(const _TCHAR* subKey, unsigned long value)
{
	RegSetValueEx(hKey, subKey, 0, REG_DWORD, 
            (unsigned char*)&value, sizeof(value)); 
}


bool
RegistryKey::readValue( const _TCHAR* subKey, unsigned long& value )
{
    long lvalue;
    if ( readValue( subKey, lvalue ) ) {
        value = (unsigned long)lvalue;
        return true;
    }
    return false;
}

bool
RegistryKey::readValue(const _TCHAR* subKey, long& value)
{
    DWORD bytes = sizeof(value);
    DWORD Type;
    long dvalue;
    long lError = RegQueryValueEx(hKey, subKey, 0, &Type,
            (unsigned char*)&dvalue, &bytes);  

    if ( lError == ERROR_SUCCESS && Type == REG_DWORD ) {
        value = dvalue;
        return true;
    }

    return false;
}

bool
RegistryKey::readValue(const _TCHAR* subKey, double& value)
{
    DWORD bytes = sizeof(value);
    DWORD Type;
    double dvalue;
    long lError = RegQueryValueEx(hKey, subKey, 0, &Type,
            (unsigned char*)&dvalue, &bytes);  

    if ( lError == ERROR_SUCCESS && Type == REG_BINARY ) {
        value = dvalue;
        return true;
    }

    return false;
}

bool
RegistryKey::readValue(const _TCHAR* subKey, bool& value)
{
    DWORD bytes = sizeof(DWORD);
    DWORD Type;
    long dvalue;
    long lError = RegQueryValueEx(hKey, subKey, 0, &Type,
            (unsigned char*)&dvalue, &bytes);  

    if ( lError == ERROR_SUCCESS && Type == REG_DWORD ) {
        if ( dvalue ) {
            value = true;
        } else {
            value = false;
        }
        return true;
    }

    return false;
}

bool
RegistryKey::readValue(const _TCHAR* subKey, int& value)
{
    DWORD bytes = sizeof(long);
    DWORD Type;
    long dvalue;
    long lError = RegQueryValueEx(hKey, subKey, 0, &Type,
            (unsigned char*)&dvalue, &bytes);  

    if ( lError == ERROR_SUCCESS && Type == REG_DWORD ) {
        value = dvalue;
        return true;
    }

    return false;
}

void 
RegistryKey::remove(const _TCHAR* key)
{
    RegDeleteKey(hKey, key);
}

void 
RegistryKey::remove()
{
    if ( _bClose ) {
	    RegCloseKey(hKey);
        RegDeleteKey(_parent, _name);
        _bClose = false;
    }
}

// ---------------------------------------------------------------------------
// CommandLine

CommandLine::CommandLine(const _TCHAR* cmdline, int max_items)
{
    _cmdline = _tcsdup(cmdline);
    _size = 0;
    _token = new _TCHAR*[max_items];
    ZeroMemory(_token, max_items * sizeof(TCHAR*));

    enum {
        no_token,
        in_token,
        quoted_token_piece,
        start_quoted,
        quoted
    } state = no_token;

    _TCHAR* ch = _cmdline;
    while( *ch ) {

        if ( state == no_token ) {
            if ( *ch == ' ' ) {
                
            } else if ( *ch == '\"' ) {
                state = start_quoted;
            } else {
                _token[_size++] = ch;
                state = in_token;
            }
        } else if ( state == in_token ) {
            if ( *ch == ' ' ) {
                *ch = 0;
                if ( (int)_size == max_items ) {
                    break;
                }
                state = no_token;
            } else if ( *ch == '\"' ) {
                state = quoted_token_piece;
            } else {
            }
        } else if ( state == quoted_token_piece ) {
            if ( *ch == '\"' ) {
                state = in_token;
            } else {
            }
        } else if ( state == start_quoted ) {
            if ( *ch == '\"' ) {
                *ch = 0;
                _token[_size++] = ch;
                if ( (int)_size == max_items ) {
                    break;
                }
                state = no_token;
            } else {
                _token[_size++] = ch;
                state = quoted;
            }
        } else if ( state == quoted ) {
            if ( *ch == '\"' ) {
                *ch = 0;
                state = no_token;
            }
        } else {
            assert(0);
        }

        ch++;
    }

}

CommandLine::~CommandLine()
{
    free(_cmdline);
    delete[] _token;
}

unsigned
CommandLine::size()
{
    return _size;
}

const _TCHAR*
CommandLine::get(unsigned index)
{
    if ( index < _size ) {
        return _token[index];
    }

    assert(0);
    return 0;
}

// ---------------------------------------------------------------------------
// Base Application
BaseApplication::BaseApplication(HINSTANCE hInstance, const _TCHAR* name, 
        float version)
{
    _TCHAR buffer[32];
    
    _hInstance = hInstance;
    _name = _tcsdup( name );
    _version = version;

    size_t len = _tcslen( _name ) + _stprintf( buffer, _T(" %.2f"), (float)version );
    _fullName = (_TCHAR*)malloc((len+1)*sizeof(_TCHAR));
    _tcscpy( _fullName, _name );
    _tcscat( _fullName, buffer );
}

BaseApplication::~BaseApplication()
{
    free(_name);
    free(_fullName);
}

static BaseApplication* BaseApp = NULL;

int BaseApplication::run( const _TCHAR* strCommandLine )
{
    // First, we have to initialize the COM library
    if(!SUCCEEDED(CoInitialize(NULL))) {
        MessageBox(0, _T("Could not initialize the COM library."), name(), 
                MB_OK | MB_ICONEXCLAMATION);
        return -1;
    }

    int ret = 0;
    if ( initialize( strCommandLine ) ) {
        ret = MessageLoop();

        uninitialize();
    }

    CoUninitialize();

    return ret;
}

bool
BaseApplication::initialize( const _TCHAR* lpszCommandLine )
{
    return true;
}

void
BaseApplication::uninitialize()
{
    
}

const _TCHAR*
BaseApplication::name()
{
    return _name;
}

const _TCHAR*
BaseApplication::fullName()
{
    return _fullName;
}

float
BaseApplication::getVersion()
{
    return _version;
}

HINSTANCE
BaseApplication::getInstance()
{
    return _hInstance;
}

// ---------------------------------------------------------------------------
// Misc functions

// Ensures that rect.right >= rect.left, and rect.bottom >= rect.top. If this
// is not the case, the values are swapped.
void ValidateRect(RECT& rect)
{	
	LONG temp;
	
	if(rect.left > rect.right)
	{
		temp = rect.left;
		rect.left = rect.right;
		rect.right = temp;
	}

	if(rect.top > rect.bottom)
	{
		temp = rect.top;
		rect.top = rect.bottom;
		rect.bottom = temp;
	}
}

// returns true if the given file exists.
bool FileExists(const _TCHAR* fullPath)
{
	HANDLE Handle = NULL;

	Handle = CreateFile(fullPath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

	if(Handle == INVALID_HANDLE_VALUE)
		return false;

	CloseHandle(Handle);
	return true;
}

// Forces the given top-level window to the foreground, taking into account
// counter-measures to prevent this in various versions of Microsoft Windows.
void ForceForegroundWindow(HWND hwnd)
{
	OSVERSIONINFO osvi;
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&osvi);
	
	BOOL bIsWindows98orLater = 
		(osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS) &&
		( (osvi.dwMajorVersion > 4) ||
		( (osvi.dwMajorVersion == 4) && (osvi.dwMinorVersion > 0) ) );

	BOOL bIsWindowsNT2000 =
		(osvi.dwPlatformId == VER_PLATFORM_WIN32_NT) &&
		(osvi.dwMajorVersion >= 5);

	if(bIsWindows98orLater && !bIsWindowsNT2000)
	{
		//For Windows 98
		DWORD dwTimeoutMS;

		// Get the current lock timeout.
		SystemParametersInfo (0x2000, 0, &dwTimeoutMS, 0);

		// Set the lock timeout to zero
		SystemParametersInfo (0x2001, 0, 0, 0);

		// Perform the SetForegroundWindow
		::SetForegroundWindow (hwnd);

		// Set the timeout back
		SystemParametersInfo (0x2001, 0, (LPVOID)dwTimeoutMS, 0);
	}
	else if(bIsWindowsNT2000)
	{
		HWND hCurrWnd;
		int iMyTID;
		int iCurrTID;

		hCurrWnd = GetForegroundWindow ();
		iMyTID   = GetCurrentThreadId ();
		iCurrTID = GetWindowThreadProcessId (hCurrWnd,0);

		// Connect to the current process.
		AttachThreadInput (iMyTID, iCurrTID, TRUE);

		// Now we look like the foreground process, we can set the 
		// foreground window.
		SetForegroundWindow (hwnd);

		// Now detach.
		AttachThreadInput (iMyTID, iCurrTID, FALSE);
	}
	else
		SetForegroundWindow(hwnd);
}

// Modifies the position of NewRect so that it is centered inside OldRect.
// Example: CentreRect( &MyRect, &ScreenRect )
BOOL CentreRect(RECT* NewRect, const RECT* OldRect)
{
	int cx = OldRect->left + (int)(((double)OldRect->right - OldRect->left) / 2);
	int cy = OldRect->top + (int)(((double)OldRect->bottom - OldRect->top) / 2);

	int nx = NewRect->left + (int)(((double)NewRect->right - NewRect->left) / 2);
	int ny = NewRect->top + (int)(((double)NewRect->bottom - NewRect->top) / 2);

	int shiftx = nx - cx;
	int shifty = ny - cy;

	NewRect->left -= shiftx;
	NewRect->right -= shiftx;
	NewRect->top -= shifty;
	NewRect->bottom -= shifty;

	return TRUE;
}

// Given an outer rect and an inner rect, fills the area outside of the inner
// rect, but inside the outer rect with the given brush.
void 
FillRectAround( HDC hDc, RECT* inner, RECT* outer, HBRUSH hBrush )
{
    RECT side;

    if ( inner->top != outer->top ) {
        // Now draw the top side
        side.left = 0;
        side.top = 0;
        side.right = outer->right;
        side.bottom = inner->top;
        FillRect( hDc, &side, hBrush );
    }

    if ( inner->bottom != outer->bottom ) {
        // Now draw the bottom side
        side.left = 0;
        side.top = inner->bottom;
        side.right = outer->right;
        side.bottom = outer->bottom;
        FillRect( hDc, &side, hBrush );
    }

    if ( inner->left != outer->left ) {
        // Now draw the left side
        side.left = 0;
        side.top = inner->top;
        side.right = inner->left;
        side.bottom = inner->bottom;
        FillRect( hDc, &side, hBrush );
    }

    if ( inner->right != outer->right ) {
        // Now draw the right side
        side.left = inner->right;
        side.top = inner->top;
        side.right = outer->right;
        side.bottom = inner->bottom;
        FillRect( hDc, &side, hBrush );
    }
}

// Obtains the area of the screen for windows, excluding the start menu bar.
void
GetScreenRect(RECT* rect)
{
	SystemParametersInfo(SPI_GETWORKAREA, 0, rect, 0);
}


// Adds a backslash to the given path, if one is not there already.
void AppendSlash(TCHAR* pszPath)
{
	int length = _tcslen(pszPath);
	
	if(pszPath[length - 1] != '\\')
	{
		_tcscat(pszPath, TEXT("\\"));
	}
}

// Removes a trailing backslash from the path, if it has one.
void RemoveSlash(TCHAR* pszPath)
{
	int length = (int)_tcslen(pszPath);
	
	if(pszPath[length - 1] == '\\')
	{
		pszPath[length - 1] = 0;
	}
}

// Wrapper for SafeSetLayeredWindowAttributes. Using this function makes it
// possible to run the program on versions of windows that do not include the
// function.
BOOL SafeSetLayeredWindowAttributes(HWND hwnd, COLORREF crKey, BYTE bAlpha, 
									DWORD dwFlags )
{
	typedef BOOL (WINAPI *SetLayeredWindowAttributesFunction)(HWND hwnd, COLORREF crKey, BYTE bAlpha, DWORD dwFlags);
	static SetLayeredWindowAttributesFunction pSetLayeredWindowAttributes = 0; 
	if ( pSetLayeredWindowAttributes == 0 ) {
		HMODULE hUser32 = GetModuleHandle(TEXT("USER32.DLL"));
		pSetLayeredWindowAttributes = 
            (SetLayeredWindowAttributesFunction)GetProcAddress(hUser32,
			"SetLayeredWindowAttributes");
	}

	if ( pSetLayeredWindowAttributes ) {
		return pSetLayeredWindowAttributes(hwnd, crKey, bAlpha, dwFlags);
	}

	return FALSE;
}

// Wrapper for AnimateWindow. Using this function makes it
// possible to run the program on versions of windows that do not include the
// function. If it doesn't exist, then ShowWindow() is used instead.
BOOL SafeAnimateWindow(HWND hwnd, DWORD dwTime, DWORD dwFlags )
{
	typedef BOOL (WINAPI *AnimateWindowFunction)(HWND hwnd, DWORD dwTime,
            DWORD dwFlags);
	static AnimateWindowFunction pAnimateWindow = 0; 
    static bool first = true;
	if ( first ) {
        first = false;
		HMODULE hUser32 = GetModuleHandle(TEXT("USER32.DLL"));
		pAnimateWindow = (AnimateWindowFunction)GetProcAddress(hUser32,
			"AnimateWindow");
	}

	if ( pAnimateWindow ) {
		return pAnimateWindow(hwnd, dwTime, dwFlags);
	} else {
        if ( dwFlags & AW_HIDE ) {
            return ShowWindow( hwnd, SW_HIDE );
        } else {
            return ShowWindow( hwnd,  SW_SHOWNORMAL );
        }
    }
}

// Invoke the framework's message loop. Use of this function allows
// accelerators and dialogs created by the framework to function properly.
int
MessageLoop()
{
	MSG Msg;
	BOOL bRet;

	while(( bRet = GetMessage(&Msg, NULL, 0, 0)) != 0 )
	{
		if ( bRet != -1 ) {
            if ( !Accelerators::translate( &Msg ) &&
                 !Dialog::isDialogMessage( &Msg ) &&
                 !PropertySheetDialog::isDialogMessage( &Msg ) ) {
                TranslateMessage(&Msg);
                DispatchMessage(&Msg);
            }
		}
	}

    return (int)Msg.wParam;
}

// Returns true if (x,y) lies within Rect.
bool PointInRect( const RECT* rect, int x, int y )
{
    return x >= rect->left && x < rect->right &&
           y >= rect->top && y < rect->bottom;    
}


// Returns a pointer to the windows error message, in the default language.
// This function is not safe to be called from multiple threads at the same
// time.
const char* 
GetWinErrorString()
{
    static char buffer[1024];
    LPVOID lpMsgBuf;
    if (!FormatMessage( 
        FORMAT_MESSAGE_ALLOCATE_BUFFER | 
        FORMAT_MESSAGE_FROM_SYSTEM | 
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        GetLastError(),
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
        (LPTSTR) &lpMsgBuf,
        0,
        NULL ))
    {
       // Handle the error.
       return "Error retrieving error";
    }

    // Display the string.
    strcpy( buffer, (const char*)lpMsgBuf );

    // Free the buffer.
    LocalFree( lpMsgBuf );

    return buffer;
}

// Snap the rectangle, typically a window, to the the edge of a larger
// rectangle, typically the screen, and return TRUE if modifications were made. 
// If bCorrectOffScreen is true, it will not be possible to move the window
// outside of the bounds of the screen.
BOOL SnapWindowRect(RECT* pWindowRect, RECT* pScreenRect, int iSnapRadius, 
        bool bCorrectOffScreen)
{
	RECT OriginalRect = *pWindowRect;

	LONG width = pWindowRect->right - pWindowRect->left;
	LONG height = pWindowRect->bottom - pWindowRect->top;

	if(bCorrectOffScreen)
	{
		//correct overedge
		pWindowRect->top = max(pScreenRect->top, pWindowRect->top);
		pWindowRect->left = max(pScreenRect->left, pWindowRect->left);
		if(pWindowRect->top + height > pScreenRect->bottom)
			pWindowRect->top = pScreenRect->bottom - height;
		if(pWindowRect->left + width > pScreenRect->right)
			pWindowRect->left = pScreenRect->right - width;
	}

	//Snap to edges
	if( abs(pWindowRect->top - pScreenRect->top) < iSnapRadius)
		pWindowRect->top = pScreenRect->top + 2;
	if( abs(pWindowRect->top + height - pScreenRect->bottom) < iSnapRadius )
		pWindowRect->top = pScreenRect->bottom - height - 2;
	if( abs(pWindowRect->left - pScreenRect->left) < iSnapRadius)
		pWindowRect->left = pScreenRect->left + 2;
	if( abs(pWindowRect->left + width - pScreenRect->right) < iSnapRadius )
		pWindowRect->left = pScreenRect->right - width - 2;

	pWindowRect->right = pWindowRect->left + width;
	pWindowRect->bottom = pWindowRect->top + height;

	return (OriginalRect.left != pWindowRect->left) || (OriginalRect.top != pWindowRect->top);
}

// Returns the position of the extension in the given path, or -1 if no
// extension exists.
int 
GetExtensionOffset(const char* pszPathAndName)
{
	//returns the position of the character after the last dot in the given string.
	//-1 if error.

	if(!pszPathAndName)
		return -1;

	int length;
	int pos = length = strlen(pszPathAndName);

	while(--pos >= 0)
	{
		if(pszPathAndName[pos] == '.')
		{
			if(++pos > length)
				pos = -1;
			break;
		}
	}

	return(pos);
}

#define strlwr _strlwr

// Determines if the given filename has the given extension, ignoring case.
bool 
CompareExtension(const char* pszFileName, const char* pszExt)
{
	int iOffset = GetExtensionOffset(pszFileName);


	if(iOffset == -1)
	{
		//return true if no extension given.
		return (*pszExt == 0);
	}

	int len = strlen(pszExt) + 1;
	
	char* pszExtLower1 = new char[len];
	char* pszExtLower2 = new char[len];

	memcpy(pszExtLower1, &pszFileName[iOffset], len - 1);
	pszExtLower1[len - 1] = 0;
	strcpy(pszExtLower2, pszExt);

	strlwr(pszExtLower1);
	strlwr(pszExtLower2);

	bool bEqual = (0 == strcmp(pszExtLower1, pszExtLower2));


	delete[] pszExtLower2;
	delete[] pszExtLower1;

	return bEqual;
}

// Removes any filename from the given path, and returns the new length.
unsigned
StripFilenameFromPath( _TCHAR* buffer )
{
    _TCHAR* slash = _tcsrchr( buffer, _T('\\') );
    if ( slash ) {
        slash[1] = 0;
        return slash - buffer + 1;
    } else {
        buffer[0] = 0;
        return 0;
    }
}

// Returns the path to the current module (typically the name of the currently
// executing EXE file). The filename is not included.
unsigned
GetModulePath( _TCHAR* buffer, unsigned size )
{
    unsigned totalLen;

    totalLen = GetModuleFileName( GetModuleHandle(NULL), buffer, size );
    buffer[totalLen] = 0;
    if ( totalLen == 0 ) {
        return 0;
    }

    return StripFilenameFromPath( buffer );
}

bool AttachToConsole() 
{
    // Look up the function in a DLL, since it is a newer one not declaredin header files.
    FARPROC fp;
    BOOL (CALLBACK *pAttachConsole)(DWORD);
    HMODULE hinst = GetModuleHandle(TEXT("KERNEL32"));
    fp = GetProcAddress(hinst, "AttachConsole");
    if (fp) {
        *(FARPROC *)&pAttachConsole = fp;
        pAttachConsole((DWORD)-1);
    }

    int m_nCRTIn = _open_osfhandle(
                 (long) GetStdHandle(STD_INPUT_HANDLE),
                  _O_TEXT );
    if( -1 == m_nCRTIn )
    {
        return false;
    }
        
    FILE* m_fpCRTIn = _fdopen( m_nCRTIn, "r" );
    
    if( !m_fpCRTIn )
    {
        return false;
    }

    FILE m_fOldStdIn = *stdin;
    *stdin = *m_fpCRTIn;
    // if clear is not done, any cout
    // statement before AllocConsole
    // will cause, the cin after
    // AllocConsole to fail, so very important

    //std::cin.clear();

    int m_nCRTOut= _open_osfhandle(
                (long) GetStdHandle(STD_OUTPUT_HANDLE),
                _O_TEXT );
    if( -1 == m_nCRTOut )
    {
        return false;
    }

    FILE* m_fpCRTOut = _fdopen( m_nCRTOut, "w" );

    if( !m_fpCRTOut )
    {
        return false;
    }

    FILE m_fOldStdOut = *stdout;
    *stdout = *m_fpCRTOut;

    // if clear is not done, any cout
    // statement before AllocConsole
    // will cause, the cout after
    // AllocConsole to fail, so very important

    //std::cout.clear();
    return true;

}

