//*******************************************************************
//BaseWnd.cpp
//By Steve Hanov (smhanov@uwaterloo.ca)//November 1999
//
// A Window wrapper that allows inheritance through
// a virtual window procedure and command handlers.
//
// Last Modified 2004-05-30
//******************************************************************

//All other windows derive from this object.

#ifndef __BaseWnd_INCLUDED__
#define __BaseWnd_INCLUDED__

#include <windows.h>
#include <prsht.h>
#include <tchar.h>
#include "commdlg.h"
#include <string>
#include <fcntl.h>
#include <io.h>

#ifndef WS_EX_LAYERED
#define WS_EX_LAYERED           0x00080000
#define LWA_COLORKEY			0x00000001
#define LWA_ALPHA               0x00000002
#endif // WS_EX_LAYERED

#ifndef AW_CENTER
#define AW_HOR_POSITIVE             0x00000001
#define AW_HOR_NEGATIVE             0x00000002
#define AW_VER_POSITIVE             0x00000004
#define AW_VER_NEGATIVE             0x00000008
#define AW_CENTER                   0x00000010
#define AW_HIDE                     0x00010000
#define AW_ACTIVATE                 0x00020000
#define AW_SLIDE                    0x00040000
#define AW_BLEND                    0x00080000
#endif

#ifndef WM_MOUSEWHEEL 
#define WM_MOUSEWHEEL 0x020A 
#endif 
#ifndef GET_WHEEL_DELTA_WPARAM 
#define GET_WHEEL_DELTA_WPARAM(wParam) ((short)HIWORD(wParam)) 
#endif 

/**
  The WindowFunctions class can be associated with any window handle, and
  allows you to call functions like moveWindow() and 
  sizeWindow() in an object oriented way.
 */
class WindowFunctions
{
public:
    WindowFunctions();
    WindowFunctions(HWND hwnd);
    virtual ~WindowFunctions();

    /// Attach to an existing window.
    void attach(HWND hwnd);

    ///
    /// Wrapper for ::MoveWindow()
    void moveWindow( int x, int y, int width, int height, bool repaint = true);
    void moveWindow( RECT* rect, bool repaint = true);

    /// Changes the size of a window without moving it.
    /// @param width New Width
    /// @param height New Height
    void sizeWindow( int width, int height );
    void setWindowText( const _TCHAR* text );

    bool isMaximized();

    void invalidate();

    /// Returns a handle to the window font.
    HFONT getFont();

    /// Sets the window font.
    void setFont( HFONT hFont );
    virtual RECT getClientRect();
    RECT getChildRect( HWND hChild );

    /// The window handle.
    HWND hwnd;

protected:
    bool _attached;
};

/**
 The Window class is the heart of the framework, allowing you to use windows
 in an object oriented way.
 */
class Window : public WindowFunctions
{
public:
	Window();
	virtual ~Window();

	HWND Create(int x, int y, int nWidth, int nHeight,
			HWND hParent, HMENU hMenu, HINSTANCE hInstance);

	HWND _hwndParent;
	HINSTANCE _hInstance;

protected:
	static LRESULT CALLBACK BaseWndProc(HWND hwnd, UINT msg,
		WPARAM wParam, LPARAM lParam);
	
	virtual LRESULT WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed);

	WNDCLASSEX _WndClass;
	DWORD _dwExtendedStyle;
	DWORD _dwStyle;
	LPCTSTR _pszClassName;
	LPCTSTR _pszTitle;

private:
	bool _bLocalMessages;
    bool _attached;
};	

/**
 * Allows you to subclass an existing window and process its messages.
 */
class SubclassedWindow : public WindowFunctions
{
public:
    SubclassedWindow();
    virtual ~SubclassedWindow();
    
    void subclass(HWND hwnd);

protected:
	virtual LRESULT WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed);

private:
    static LRESULT CALLBACK BaseWndProc(HWND hwnd, UINT msg,
		WPARAM wParam, LPARAM lParam);

    LONG _oldWndProc;
};

class PropertyPage;

/**
 * Encapsulates a windows dialog box, usually loaded from a resource file.
 */ 
class Dialog : public WindowFunctions
{
public:
	Dialog( HINSTANCE hInstance, LPCTSTR _pszTemplate, HWND hParent );
    virtual ~Dialog();

    HWND CreateModeless();
	virtual int DoModal();
    bool getCheck( unsigned id );
    void setCheck( unsigned id, bool value );
    HWND getDlgItem( unsigned id );

	HWND _hwndParent;
	HINSTANCE _hInstance;
    static bool isDialogMessage( LPMSG msg );

protected:

	virtual LRESULT DialogProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam);

private:
    friend class PropertyPage;
    
    static BOOL CALLBACK BaseDialogProc( HWND hwndDlg, UINT uMsg,
		WPARAM wParam, LPARAM lParam);


	LPCTSTR _pszTemplate;
};

/**
 * Encapsulates a windows property page.
 */ 
class PropertyPage : public Dialog
{
public:
	PropertyPage(HINSTANCE hInstance, LPCTSTR pszTemplate);
    virtual ~PropertyPage();

protected:
	virtual bool areChangesValid();
	virtual bool onApply();

	virtual LRESULT DialogProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam);

	PROPSHEETPAGE _PageStruct;
private:
    friend class PropertySheetDialog;
	HPROPSHEETPAGE create();
    static BOOL CALLBACK BaseDialogProcProp(HWND hwnd, UINT msg, WPARAM wParam,
                                                              LPARAM lParam);
};

#define MAX_PROPERTY_PAGES	25

/**
 * Encapsulates a windows property sheet dialog box.
 */ 
class PropertySheetDialog
{
public:
	PropertySheetDialog(HINSTANCE hInstance);
    virtual ~PropertySheetDialog();

    virtual void modelessEnded( bool okClicked );
	
	bool addPage(PropertyPage *pNewPage);
	int showModal(HWND hwndParent, int nStartPage = -1);
    HWND showModeless( HWND hwndParent, int nStartPage = -1 );
    void setTitle( const _TCHAR* title );
    static bool isDialogMessage( LPMSG pMsg );
	
protected:
    HINSTANCE _hInstance;
	HPROPSHEETPAGE _PageArray[MAX_PROPERTY_PAGES];
	unsigned int _iNumPages;
    _TCHAR _title[255];
    HWND hwnd;

	PROPSHEETHEADER _PropSheetHeader;
};


/**
 * Encapsulates a windows menu.
 */ 
class Menu
{
public:
	Menu( bool popup );
    Menu( HMENU menu );
    Menu();
	~Menu();

	void append( const _TCHAR* text, Menu& menu );
	void appendBar();
	void append( const _TCHAR* text, unsigned int id);
    void setDefaultItem( unsigned int id );
    void setCheck( unsigned int id, bool checked);
    void enable(unsigned int id, bool enabled);
    void setText(unsigned id, const _TCHAR* text );
    void removeAll();
	HMENU hMenu;
private:
    bool _destroy;
    int _size;
};

/**
 * Encapsulates a windows Open/Save dialog.
 */ 
class FileSelectDialog
{
public:
	FileSelectDialog(HWND hwndOwner, const _TCHAR* title);
	~FileSelectDialog();

	void addFilter( const _TCHAR* name, const _TCHAR* filter);

    // Buffer must be valid!!!
	bool getOpenFileName(TCHAR* buffer, int bufferLen, bool multiselect = false);

	bool getSaveFileName(TCHAR* buffer, int bufferLen);

    const _TCHAR* nextMultiSelectFile();

private:
    _TCHAR* _multisel;

    _TCHAR* _buffer;
    size_t _bufferLen;
    size_t _pos;
    
	TCHAR _filter[512];
	int _filterLen;
	OPENFILENAME _openFileName;
};

/**
 * Allows you to set windows accelerators throughout your application.
 * Accelerators allow you to use ALT+ shortcuts to access menu items.
 * You will have to call Accelerators::translate() in your message loop for
 * them to work. If you call the MessageLoop() function this will be handled
 * for you.
 */ 
class Accelerators
{
public:
	Accelerators();
	~Accelerators();

	HACCEL haccel();
    // add(FALT | FCONTROL | FSHIFT | FVIRTKEY, 'A', CMD);
    // In general you need FVIRTKEY, else its case sensitive.
    // Also, use upper case.
	void add(BYTE fVirt, WORD key, WORD cmd);

    void registerWindow(HWND hwnd);
    void unregister();
    static bool translate(MSG* msg);
private:
    bool _registered;
	HACCEL _haccel;
	ACCEL _array[8];
	int items;
};

/**
 * Encapsulates a windows slider control.
 */ 
class TrackBarControl : public WindowFunctions
{
public:
    TrackBarControl();
    ~TrackBarControl();

    int getPos();
    void setPos( int pos );
    void setRange( int minimum, int maximum );

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );
};

/**
 * Encapsulates a windows edit control.
 */ 
class EditControl : public WindowFunctions
{
public:
    EditControl();
    virtual ~EditControl();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );
};

/**
 * Encapsulates a windows "static" or label control.
 */ 
class StaticControl : public WindowFunctions
{
public:
    StaticControl();
    virtual ~StaticControl();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );
};

/**
 * Encapsulates a windows "Up Down" control.
 */ 
class UpDownControl : public WindowFunctions
{
public:
    UpDownControl();
    virtual ~UpDownControl();

    // style: UDS_ARROWKEYS | UDS_ALIGNRIGHT | UDS_SETBUDDYINT
    HWND create( unsigned style, HWND hParent, unsigned id );
    void setBuddy( HWND hBuddy );
#ifdef UDM_SETPOS32
    void setRange( int iLow, int iHigh );
    void setPos( int pos );
    bool getPos( int* pos );
#else
    void setRange( short iLow, short iHigh );
    void setPos( short pos );
    bool getPos( short* pos );
#endif
};

/**
 * Encapsulates a windows Rich Edit control.
 */ 
class RichEditControl : public WindowFunctions
{
public:
    RichEditControl();
    virtual ~RichEditControl();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );

    void setBkColour(COLORREF clr);
};

/**
 * Encapsulates a windows Progress Bar control.
 */ 
class ProgressBarControl : public WindowFunctions
{
public:
    ProgressBarControl();
    virtual ~ProgressBarControl();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );

    void setRange(int lower, int higher);
    void setPos( int pos );
    void setBkColour(COLORREF clr);
    void setBarColour( COLORREF clr );
};

/**
 * Encapsulates a windows List box control.
 */ 
class ListBox : public WindowFunctions
{
public:
    ListBox();
    virtual ~ListBox();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );

    void addString( const _TCHAR* text );
    void addString( const _TCHAR* text, void* data );
    void* getItemData( int sel );
    void clear();
    int getCurSel();
    void setCurSel( int sel );
    _TCHAR* getText( int sel );
    int getCount();
    int findStringExact( const _TCHAR* text );
    void deleteString( int index );
};

/**
 * Encapsulates a windows List View control.
 */ 
class ListViewControl : public WindowFunctions
{
public:
    ListViewControl();
    virtual ~ListViewControl();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );
    void setExtendedStyle( unsigned style );
    void setBkColour( COLORREF colour );
    void insertColumn( int iCol, int cx, const _TCHAR* text );
    void clear();
    int getCount();
    void setCurSel(int index);
    void deleteColumn(int index);
    int getColumnWidth( int index );
    void setColumnWidth( int index, int cx );

    // Pass in -1 to get the first selected item. 
    // To get the next one, pass in the previous value returned.
    int getNextSel(int start);
    void insertItem( int index, const _TCHAR* text, void* data );

    // SubItem 0 refers to the main item.
    void setItemText( int index, int subItem, const _TCHAR* text );
};

/**
 * Encapsulates a windows Hot Key control.
 */ 
class HotkeyControl : public WindowFunctions
{
public:
    HotkeyControl();
    virtual ~HotkeyControl();

    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );

    //MAKEWORD( HOTKEYF_ALT | HOTKEYF_CONTROL | HOTKEYF_EXT | HOTKEYF_SHIFT,
    //        VirtualKeyCode );
    void setHotkey( unsigned key );
    unsigned getHotkey();
};

/**
 * Encapsulates a windows Tab control.
 */ 
class TabControl : public WindowFunctions
{
public:
    TabControl();
    virtual ~TabControl();

    void insertItem( int index, const _TCHAR* text );
    int getCurSel();
};

/**
 * Encapsulates a windows ComboBox control.
 */ 
class ComboBox : public WindowFunctions
{
public:
    ComboBox();
    virtual ~ComboBox();

    // Styles:
    // CBS_DROPDOWNLIST - no editing the text.
    HWND create( unsigned style, int x, int y, int cx, int cy, HWND hParent, 
            unsigned id );

    int addString( const _TCHAR* text );
    void clear();
    int getCurSel();
    void setCurSel( int sel );
    _TCHAR* getText( int sel );
    int getCount();
};

/**
 * Encapsulates a windows Status bar control.
 */ 
class StatusBar : public WindowFunctions
{
public:
    StatusBar();
    virtual ~StatusBar();

    // styles: SBARS_SIZEGRIP, SBT_TOOLTIPS
    HWND create( LONG style, LPCTSTR lpszText, HWND hwndParent, unsigned id );
};


/**
 * Encapsulates a windows Device Context.
 */ 
class DeviceContext
{
public:
    DeviceContext(HDC dc);
    DeviceContext( const DeviceContext& other );
    ~DeviceContext();

    operator HDC();

    void selectFont( HFONT hFont );
    void selectPen( HPEN hPen );
    void selectBrush( HBRUSH hBrush );

    HDC dc;
protected:
    bool _release;
private:
    HFONT _hOldFont;
    HPEN _hOldPen;
    HBRUSH _hOldBrush;
};

class ScreenDC : public DeviceContext
{
public:
    ScreenDC();
};

/**
 * Encapsulates a windows Font
 */ 
class Font
{
public:
    Font(const _TCHAR* faceName, int height);
    Font( HFONT );
    Font( const Font& other );
    Font();
    ~Font();

    operator HFONT() const;
    const LOGFONT* logfont() const;

    void setFaceName(const _TCHAR* name);
    void setLogHeight( int height );
    void setBold( bool bold );
    void setItalic( bool italic );

    void setUnderline( bool underline ); // no effect, just storage.
    int getCharWidth( _TCHAR ch );

    const _TCHAR* getFaceName() const;
    int getHeight() const;
    int getLogHeight() const;
    int getExternalLeading() const;
    int getInternalLeading() const;
    int getAscent() const;
    int getDescent() const;
    bool getBold() const;
    bool getItalic() const;
    bool getUnderline() const;
    bool chooseFontDialog(HWND hwnd);

private:
    void construct(const _TCHAR* faceName, int height);
    void destroy();
    void getTm() const;
    LOGFONT _logfont;
    mutable TEXTMETRIC _tm;
    mutable HFONT _hFont;
    bool _underline;
};

/**
 * Encapsulates a windows GDI Brush
 */ 
class Brush
{
public:
    Brush( COLORREF colour );
    Brush();
    ~Brush();

    operator HBRUSH();
    void setColour( COLORREF clr );

private:
    HBRUSH _hBrush;
};

/**
 * Encapsulates a windows GDI Pen
 */ 
class Pen
{
public:
    Pen( int style, int width, COLORREF colour );
    ~Pen();

    operator HPEN() ;

private:
    HPEN _hPen;
};


/**
 * Provide access to the windows registry.
 */ 
class RegistryKey
{
public:
    // HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER,
    // HKEY_LOCAL_MACHINE, HKEY_USERS, or a parent.
    RegistryKey(HKEY hParent);
    RegistryKey(RegistryKey& parent, const _TCHAR* SubKey);
    bool hasSubKey( const _TCHAR* subKey );
    bool becomeSubKey( const _TCHAR* subKey );
    bool readValue(const _TCHAR* subKey, unsigned long& value );
    bool readValue(const _TCHAR* subKey, long& value);
    bool readValue(const _TCHAR* subKey, int& value);
    bool readValue(const _TCHAR* subKey, bool& value);
    bool readValue(const _TCHAR* subKey, double& value);
    bool readValue(const _TCHAR* subKey, _TCHAR* value, int size);
    void writeValue(const _TCHAR* subKey, const _TCHAR* value);
    void writeValue(const _TCHAR* subKey, long value);
    void writeValue(const _TCHAR* subKey, double value);
    void writeValue(const _TCHAR* subKey, int value);
    void writeValue(const _TCHAR* subKey, bool value);
    void writeValue( const _TCHAR* subkey, unsigned long value );
    void remove();
    void remove(const _TCHAR* subKey);
    ~RegistryKey();

    HKEY hKey;
private:
    bool _bClose;
    bool _bCloseParent;
    _TCHAR* _name;
    HKEY _parent;
};

/**
 * Parses the given command line, taking into account quote characters and
 * double quotes.
 */ 
class CommandLine
{
public:
    CommandLine(const _TCHAR* cmdline, int max_items = 10);        
    ~CommandLine();

    unsigned size();
    const _TCHAR* get(unsigned index);
private:
    unsigned _size;
    _TCHAR* _cmdline;
    _TCHAR** _token;

};

/**
 * A convenience class for your application. If you derive from this class, and
 * call run() from WinMain(), your message loop will be automatically handled
 * by the BaseWnd library.
 */ 
class BaseApplication
{
public:
	BaseApplication(HINSTANCE, const _TCHAR* name, float version);
	virtual ~BaseApplication();

    virtual bool initialize( const _TCHAR* lpszCommandLine );
    virtual void uninitialize();

	int run(const _TCHAR* lpszCommandLine);

    const _TCHAR* name();
    const _TCHAR* fullName();
    float getVersion();

    HINSTANCE getInstance();

private:
    float _version;
    _TCHAR* _name;
    _TCHAR* _fullName;
	HINSTANCE _hInstance;
};


// ---------------------------------------------------------------------------
// Misc functions

BOOL CentreRect(RECT* NewRect, const RECT* OldRect);
BOOL SnapWindowRect(RECT* pWindowRect, RECT* pScreenRect, int iSnapRadius, 
        bool bCorrectOffScreen);
void FillRectAround( HDC dc, RECT* inner, RECT* outer, HBRUSH hBrush );
bool PointInRect( const RECT* rect, int x, int y );
void ForceForegroundWindow(HWND hwnd);
void GetScreenRect(RECT* rect);
BOOL SafeSetLayeredWindowAttributes(HWND hwnd, COLORREF crKey, BYTE bAlpha, 
									DWORD dwFlags );
BOOL SafeAnimateWindow( HWND hwnd, DWORD dwTime, DWORD dwFlags );
void ValidateRect(RECT& rect);

void InitBaseWnd(HINSTANCE hInstance, double scaling = 1.0);
HINSTANCE GetInstance();

int TwipsX(int twips);
int TwipsY(int twips);
int MillimetresX( int mm );
int MillimetresY( int mm );

BOOL TranslateAccelerators( Accelerators* accel );
int MessageLoop();

const char* GetWinErrorString();

void AppendSlash(TCHAR* pszPath);
void RemoveSlash(TCHAR* pszPath);
bool FileExists(const _TCHAR* fullPath);
unsigned StripFilenameFromPath( _TCHAR* buffer );
unsigned GetModulePath( _TCHAR* buffer, unsigned size );
unsigned StripFilenameFromPath( _TCHAR* buffer );
bool CompareExtension(const char* pszFileName, const char* pszExt);
int GetExtensionOffset(const char* pszPathAndName);

/**
 Attaches to the console of the parent process. This will allow GUI applications to use
 printf functions to write to the console window, if they have been started from a console.
 */
bool AttachToConsole();

#endif // __BaseWnd_INCLUDED__
