#include <assert.h>
#include "Bitmap.h"
#include "BaseWnd.h"
#include "DibImage.h"

Bitmap::Bitmap()
{
    _hBitmap = 0;
    _hOldBitmap = 0;
    _hMask = 0;
    _hDC = 0;
}

Bitmap::~Bitmap()
{
    destroy();
    _hBitmap = 0;
    _hOldBitmap = 0;
    _hDC = 0;
    _hMask = 0;
}

/// Return true if the bitmap has not been created.
bool
Bitmap::empty()
{
    return _hDC == 0;
}

/// Destroys the bitmap, and releases all memory.
void
Bitmap::destroy()
{
    if ( _hDC ) {
        SelectObject( _hDC, _hOldBitmap );
        DeleteDC( _hDC );
        _hDC = 0;
        _hOldBitmap = 0;
    }

    if ( _hBitmap ) {
        DeleteObject( _hBitmap );
        _hBitmap = 0;
    }

    if ( _hMask ) {
        DeleteObject( _hMask );
        _hMask = 0;
    }
}

/// Sets the contents of this bitmap to be the same as another one.
void
Bitmap::copy( Bitmap* other )
{
    destroy();
    if ( other->_hDC == 0 ) {
        return;
    }

    createCompatible( other->_hDC, other->width(), other->height() );
    other->bitBlt( _hDC, 0, 0, width(), height(), 0, 0 );

    if ( other->_hMask ) {
        BOOL bFailed = TRUE;
        HBITMAP hMaskBMP = NULL;
        HDC dcMask = NULL;
        HBITMAP hOldMask = NULL;
        HDC dcScreen = NULL;
        HDC dcOtherMask = CreateCompatibleDC(_hDC);
        HBITMAP hOtherOldMask = (HBITMAP)SelectObject(dcOtherMask, other->_hMask);

        dcScreen = GetDC(NULL);

        if(!dcScreen)
            goto failure;

        hMaskBMP = CreateBitmap(width(), height(), 1, 1, NULL);

        if(!hMaskBMP)
            goto failure;

        dcMask = CreateCompatibleDC(dcScreen);
        hOldMask = (HBITMAP)SelectObject(dcMask, hMaskBMP);
        BitBlt(dcMask, 0, 0, width(), height(), dcOtherMask, 0, 0, SRCCOPY);
        bFailed = FALSE;
        _hMask = hMaskBMP;

    failure:

        SelectObject( dcOtherMask, hOtherOldMask );

        ReleaseDC( NULL, dcScreen );
        
        if(dcMask)
        {
            SelectObject(dcMask, hOldMask);
            DeleteDC(dcMask);
        }

        if(bFailed && hMaskBMP) {
            DeleteObject(hMaskBMP);
            hMaskBMP = 0;
        }
    }
}

/// Creates a bitmap compatible with the given device context.
bool
Bitmap::createCompatible( HDC hdc, int nWidth, int nHeight )
{
    destroy();
    _hDC = CreateCompatibleDC( hdc );
    _hBitmap = CreateCompatibleBitmap( hdc, nWidth, nHeight );
    _hOldBitmap = (HBITMAP)SelectObject( _hDC, _hBitmap );
    if ( _hBitmap ) {
        calcSize();
    }

    return _hBitmap != NULL;
}

/**
 Creates a bitmap that is compatible with the device context of the given
 window handle.
 */
bool 
Bitmap::createCompatible( HWND hwnd, int nwidth, int nheight )
{
    HDC dc = GetDC( hwnd );
    bool ret = createCompatible( dc, nwidth, nheight );
    ReleaseDC( hwnd, dc );
    return ret;
}

/// Loads a bitmap from the current instance's module file.
bool
Bitmap::loadResource( LPCTSTR lpBitmapName )
{
    destroy();
    HDC dcScreen = GetDC(NULL);
    _hDC = CreateCompatibleDC( dcScreen );
    ReleaseDC( NULL, dcScreen );
    _hBitmap = LoadBitmap( GetInstance(), lpBitmapName );
    if ( _hBitmap == 0 ) {
        int a = GetLastError();
        destroy();
        assert(0);
        return false;
    }

    _hOldBitmap = (HBITMAP)SelectObject( _hDC, _hBitmap );

    calcSize();

    return true;
}

/// Returns the handle to the bitmap.
Bitmap::operator HBITMAP()
{
    return _hBitmap;
}

/// Returns a device context for the bitmap.
Bitmap::operator HDC()
{
    return getDC();
}

/// Returns a device context for the bitmap.
HDC
Bitmap::getDC()
{
    return _hDC;
}

/// Object oriented wrapper for BitBlt(). Automatically handles a transparency
/// mask, if needed.
void 
Bitmap::bitBlt( HDC dc, int XDest, int YDest, int Width, int Height, 
        int XSrc, int YSrc)
{
    if ( !_hMask ) {
        BitBlt( dc, XDest, YDest, Width, Height, _hDC, XSrc, YSrc, SRCCOPY );
    } else {
        HDC dcMask = CreateCompatibleDC(dc);
        HBITMAP hOldMask = (HBITMAP)SelectObject(dcMask, _hMask);

        // !!! This affects transparent blitting!!!
        COLORREF clr = SetTextColor( dc, 0 );
        
        // SRCAND, SRCPAINT
        BitBlt(dc, XDest, YDest, Width, Height, dcMask, XSrc, YSrc, SRCAND);
        BitBlt(dc, XDest, YDest, Width, Height, _hDC, XSrc, YSrc, 
                SRCPAINT);
        
        SetTextColor( dc, clr );

        SelectObject(dcMask, hOldMask);
        DeleteDC(dcMask);
    }

}

/// Object oriented wrapper for StretchBlit(). Automatically handles a
/// transparency mask, if needed.
void 
Bitmap::stretchBlt( HDC dc, int XDest, int YDest, int WidthDest, int HeightDest,
        int XSrc, int YSrc, int WidthSrc, int HeightSrc )
{
    if ( !_hMask ) {
        StretchBlt( dc, XDest, YDest, WidthDest, HeightDest, _hDC, XSrc, YSrc, 
                WidthSrc, HeightSrc, SRCCOPY );
    } else {
        HDC dcMask = CreateCompatibleDC(dc);
        HBITMAP hOldMask = (HBITMAP)SelectObject(dcMask, _hMask);

        // !!! This affects transparent blitting!!!
        COLORREF clr = SetTextColor( dc, 0 );
        
        // SRCAND, SRCPAINT
        StretchBlt(dc, XDest, YDest, WidthDest, HeightDest, dcMask, XSrc, YSrc, 
                WidthSrc, HeightSrc, SRCAND);
        StretchBlt(dc, XDest, YDest, WidthDest, HeightDest, _hDC, XSrc, YSrc, 
                WidthSrc, HeightSrc, SRCPAINT);
        
        SetTextColor( dc, clr );

        SelectObject(dcMask, hOldMask);
        DeleteDC(dcMask);
    }
}

/// Scales the bitmap, using a resampling algorithm so it will look visually
/// pleasing.
void
Bitmap::scale( HDC dc, int newWidth, int newHeight )
{
    assert( newWidth > 0 );
    assert( newHeight > 0 );
    assert( _hDC );

    RECT rect;
    SetRect( &rect, 0, 0, width(), height() );

    DibImage* dib = new DibImage;
    DibImage* dibScaled = 0;
    dib->GetFromDC( _hDC, &rect );   
    dibScaled = dib->createScaled( newWidth, newHeight );
    delete dib;
    if ( !dibScaled ) {
        return;
    }

    destroy();
    createCompatible( dc, newWidth, newHeight );
    dibScaled->display( _hDC, 0, 0, newWidth, newHeight, 0, 0 );
    delete dibScaled;
}

/// Replaces the contents of this Bitmap by loading a bitmap file from disk. The
/// bitmap will be compatible with the given device context.
bool
Bitmap::load( HDC dc, const TCHAR* filename )
{
    DibImage* dib = DibImage::Load( filename );

    if ( !dib ) {
        return false;
    }

    destroy();
    int newWidth = dib->width();
    int newHeight = dib->height();

    createCompatible( dc, newWidth, newHeight );
    dib->display( _hDC, 0, 0, newWidth, newHeight, 0, 0 );
    delete dib;

    return true;
}

/// Replaces the contents of this Bitmap by loading a bitmap file from a Stream
/// object. The bitmap will be compatible with the given device context.
bool
Bitmap::load( HDC dc, Stream* stream )
{
    DibImage* dib = DibImage::LoadPng( stream );

    if ( !dib ) {
        return false;
    }

    destroy();
    int newWidth = dib->width();
    int newHeight = dib->height();

    createCompatible( dc, newWidth, newHeight );
    dib->display( _hDC, 0, 0, newWidth, newHeight,0,0 );
    delete dib;

    return true;
}

/// Returns true if this bitmap has a transparency mask associated with it.
bool
Bitmap::hasMask()
{
    return _hMask != 0;
}

/// Stores a bitmap as a file, written out to a Stream object.
bool 
Bitmap::save( Stream* stream )
{
    bool success = false;
    if ( empty() ) {
        assert(0);
        return false;
    }

    RECT rect;
    SetRect( &rect, 0, 0, width(), height() );

    DibImage* dib = new DibImage;
    DibImage* dibScaled = 0;
    dib->GetFromDC( _hDC, &rect );   
    success = SavePng( dib, stream );
    delete dib;

    return success;
}

/// Loads a transparency mask from the given Stream object. The mask must be the
/// same size as the existing bitmap, and it will be compatible with the device
/// context of the screen.
bool
Bitmap::loadMask( Stream* stream )
{
    if ( empty() ) {
        return false;
    }

    DibImage* dib = DibImage::LoadPng( stream );
    if ( dib == 0 ) {
        return false;
    }

    if ( dib->width() != width() || dib->height() != height() ) {
        return false;
    }

    HDC dcScreen = 0;
    HDC dcMask = 0;
    HBITMAP hMaskBMP = 0;
    HBITMAP hOldMask = 0;
    bool success = false;
    
    dcScreen = GetDC(NULL);
    if ( !dcScreen ) {
        goto failed;
    }
    
    dcMask = CreateCompatibleDC( dcScreen );
    if ( dcMask == 0 ) {
        goto failed;
    }
    
    hMaskBMP = CreateBitmap(width(), height(), 1, 1, NULL);
    if ( hMaskBMP == 0 ) {
        goto failed;
    }

    hOldMask = (HBITMAP)SelectObject(dcMask, hMaskBMP );
    if ( hOldMask == 0 ) {
        goto failed;
    }

    dib->display( dcMask, 0, 0, width(), height(),0,0 );

    if ( _hMask ) {
        DeleteObject( _hMask );
    }

    _hMask = hMaskBMP;
    hMaskBMP = 0;

    success = true;
failed:
    if ( dcScreen ) {
        ReleaseDC( NULL, dcScreen );
    }
    
    if ( hOldMask ) {
        SelectObject( dcMask, hOldMask );
    }

    if ( dcMask ) {
        DeleteDC( dcMask );
    }

    if ( hMaskBMP ) {
        DeleteObject( hMaskBMP );
    }

    delete dib;

    return success;
}

/// Stores the transparency mask to the stream, as a bitmap file.
bool 
Bitmap::saveMask( Stream* stream )
{
    bool success = false;
    if ( empty() ) {
        assert(0);
        return false;
    }

    if ( _hMask == 0 ) {
        assert(0);
        return false;
    }

    RECT rect;
    SetRect( &rect, 0, 0, width(), height() );

    HDC dcScreen = 0;
    HDC dcMask = 0;
    HBITMAP hOldMask = 0;
    DibImage* dib = 0;

    dcScreen = GetDC( NULL );
    if ( !dcScreen ) {
        goto failed;
    }

    dcMask = CreateCompatibleDC( dcScreen );
    if ( !dcMask ) {
        goto failed;
    }

    hOldMask = (HBITMAP)SelectObject( dcMask, _hMask );
    
    dib = new DibImage();
    dib->GetFromDC( dcMask, &rect );   
    success = SavePng( dib, stream );
failed:
    if ( dcScreen ) { 
        ReleaseDC( NULL, dcScreen );
    }

    if ( hOldMask ) {
        SelectObject( dcMask, hOldMask );
    }

    if ( dcMask ) {
        DeleteDC( dcMask );
    }

    delete dib;


    return success;
}

/// Returns the width of the bitmap, in pixels.
int 
Bitmap::width()
{
    if ( _hBitmap == 0 ) {
        return 1;
    }

    return _size.cx;
}

/// Returns the height of the bitmap, in pixels.
int
Bitmap::height()
{
    if ( _hBitmap == 0 ) {
        return 1;
    }

    return _size.cy;
}


/// Internal function used to compute the size of the bitmap.
void
Bitmap::calcSize()
{
    BITMAP bmp;
    if ( GetObject( (HGDIOBJ)_hBitmap, sizeof(bmp), &bmp ) ) {
        _size.cx = bmp.bmWidth;
        _size.cy = bmp.bmHeight;
        return;
    }

    assert(0);
    _size.cx = 1;
    _size.cy = 1;
}

/// Internal function used to create a transparency mask given a colour.
static HBITMAP 
CreateMask(HDC dcSrc, COLORREF MaskColour, int width, int height)
{
	BOOL bFailed = TRUE;
	HBITMAP hMaskBMP = NULL;
	HDC dcMask = NULL;
	HBITMAP hOldMask = NULL;
    HDC dcScreen = NULL;
    COLORREF a;
    COLORREF OldColour;

	dcScreen = GetDC(NULL);

	if(!dcScreen)
		goto failure;

	hMaskBMP = CreateBitmap(width, height, 1, 1, NULL);

	if(!hMaskBMP)
		goto failure;

	dcMask = CreateCompatibleDC(dcScreen);
	hOldMask = (HBITMAP)SelectObject(dcMask, hMaskBMP);
    a = GetPixel( dcSrc, 2, 2 );
    OldColour = SetBkColor(dcSrc, MaskColour);
    BitBlt(dcMask, 0, 0, width, height, dcSrc, 0, 0, NOTSRCCOPY);
    SetBkColor(dcSrc, OldColour);
    
    //BitBlt( dcScreen, 0, 0, width, height, dcMask, 0, 0, SRCCOPY );
    
    BitBlt(dcSrc, 0, 0, width, height, dcMask, 0, 0, SRCAND);
    BitBlt(dcMask, 0, 0, width, height, dcSrc, 0, 0, DSTINVERT);

    //BitBlt( dcScreen, width, 0, width, height, dcSrc, 0, 0, SRCCOPY );
	bFailed = FALSE;

failure:

    ReleaseDC( NULL, dcScreen );
	
	if(dcMask)
	{
		SelectObject(dcMask, hOldMask);
		DeleteDC(dcMask);
	}

	if(bFailed && hMaskBMP) {
		DeleteObject(hMaskBMP);
        hMaskBMP = 0;
    }

	return hMaskBMP;
}


/// Permanently discards the transparency mask, if one is associated with the
/// bitmap.
void
Bitmap::noTransparent()
{
    if ( _hMask ) {
        DeleteObject( _hMask );
        _hMask = 0;
    }
}

/// Creates a new transparency mask for the bitmap. All pixels of the given
/// colour will be made transparent.
void
Bitmap::makeTransparent( COLORREF colour )
{
    assert( _hBitmap );
    noTransparent();
    _hMask = CreateMask( _hDC, colour, width(), height() );
    assert( _hMask );
}

/**
 Returns the pixel at the given coordinate. This is very slow. For fast
 bitmap manipulations, you are better off using DibImage.
*/
COLORREF
Bitmap::getPixel(int x, int y)
{
    return ::GetPixel( _hDC, x, y );
}
    
