#include <Bitmap.h>
#include "ButtonControl.h"
#include "BaseWnd.h"
#include <assert.h>

/** 
 Construct a ButtonControl object.

 @param hwnd Handle of parent window.

 @param commandId Command ID to send to parent in WM_COMMAND message when 
        clicked.
 */
ButtonControl::ButtonControl(HWND hwnd, DWORD commandId) :
    FadingControl(hwnd),
    _textControl(hwnd),
    commandId(commandId)
{
    _bitmap = 0;
    _picturePos.x = _picturePos.y = 0;
    _insideMargin = 10;
    _backColour = RGB(127,127,127);
    _roundEllipse = 0;
    _textControl.setFont(Font(TEXT("Tahoma"), TwipsY(1440/2)));
    _textControl.setColour( RGB(255,255,0));
    _textControl.allowFading( true );
    _textControl.setWrap( true );
    _bulletColour = 0;
    _bulletHeight = 1;
    _useBullet = false;
    addChild( &_textControl );
}

ButtonControl::~ButtonControl()
{
    _bitmap = 0;
}

void 
ButtonControl::move( int x, int y, int cx, int cy )
{
    FadingControl::move( x, y, cx, cy );
    format();
}

/**
    Move to the given coordinates, and word wrap the text as well.
*/
int 
ButtonControl::moveFormat(int x, int y, int width )
{
    _rect.left = x;
    _rect.top = y;
    _rect.right = _rect.left + width;
    _rect.bottom = _rect.top;

    format();
    return _rect.bottom - _rect.top;
}

/**
 Layout the control and word-wrap the text.
 */
void
ButtonControl::format()
{
    int width = _rect.right - _rect.left;
    int height = _rect.bottom - _rect.top;
    int textX = _insideMargin;

    height = 0;

    if ( _bitmap ) {
        int bwidth = _bitmap->width();
        int bheight = _bitmap->height();

        _picturePos.x = _insideMargin;
        _picturePos.y = _insideMargin;

        textX = _picturePos.x + bwidth + 
            _textControl.getFont()->getCharWidth(TEXT('M'));
        height += bheight;
    } else if ( _useBullet ) {
        _bulletHeight = _textControl.getFont()->getLogHeight();
        if ( _bulletHeight < 0 ) {
            _bulletHeight = -_bulletHeight;
        }

        _picturePos.x = _insideMargin;
        _picturePos.y = _insideMargin; 
        textX = _picturePos.x + 
            _textControl.getFont()->getCharWidth(TEXT('M'))  +
            _bulletHeight;
        height += _bulletHeight;
    }

    int theight = _textControl.moveFormat( textX, _insideMargin,
            width - _insideMargin - textX );

    if ( theight > height ) {
        height = theight;
    }

    height += _insideMargin * 2;

    FadingControl::move( _rect.left, _rect.top, width, height );
}

SIZE
ButtonControl::calcSize()
{
    SIZE size;
    size = _textControl.calcSize();

    size.cx += _insideMargin * 2;
    size.cy += _insideMargin * 2;
   
    if ( _bitmap || _useBullet ) {
        size.cx += _textControl.getFont()->getCharWidth(TEXT('M'));
    }

    if ( _bitmap ) {
        size.cx += _bitmap->width();
        if ( _bitmap->height() + _insideMargin * 2 > size.cy ) {
            size.cy = _bitmap->height() + _insideMargin * 2;
        }
    }  else if ( _useBullet ) {
        size.cx += _bulletHeight;
        if ( _bulletHeight + _insideMargin * 2 > size.cy ) {
            size.cy = _bulletHeight + _insideMargin * 2;
        }
    }

    return size;
}

void
ButtonControl::onMouseEnter()
{
    FadingControl::onMouseEnter();
    invalidate();
    _textControl.onMouseEnter();
}

void 
ButtonControl::onMouseOut()
{
    FadingControl::onMouseOut();
    invalidate();
    _textControl.onMouseOut();
}

void
ButtonControl::draw(HDC hdc)
{

    Pen pen( PS_NULL, 0, 0 );
    Brush backBrush( getFadeColour( _backColour ) );
    Brush bulletBrush(_bulletColour);
    DeviceContext dc(hdc);
    dc.selectPen( pen );

    RECT rect = getWindowRect();

    if ( hover() || _allowFading ) {
        dc.selectBrush( backBrush );
        if ( _roundEllipse != 0 ) {
            RoundRect( dc, rect.left, rect.top,
                    rect.right, rect.bottom,
                    _roundEllipse, _roundEllipse );
        } else {
            Rectangle( dc, rect.left, rect.top,
                    rect.right, rect.bottom );
        }
    }

    POINT picturePos = _picturePos;
    if ( _parent ) {
        offset( &picturePos );
    }
    
    if ( _bitmap ) {
        _bitmap->bitBlt( dc, rect.left + picturePos.x,
                rect.top + picturePos.y,
                _bitmap->width(), _bitmap->height(),
                0, 0 );
    } else if ( _useBullet ) {
        dc.selectBrush(bulletBrush);
        Rectangle( dc, rect.left + picturePos.x,
                rect.top + picturePos.y,
                rect.left + picturePos.x + _bulletHeight,
                rect.top + picturePos.y + _bulletHeight );
    }

    Control::draw( hdc );
}

/**
 Set the text to be displayed on the button. The text may contain newlines.
 */
void
ButtonControl::setText( const TCHAR* text )
{
    _textControl.setText(text);
}

/**
 Sets the bitmap to be displayed to the left of the text, like a bullet. This will disable any other bullet setting.
 */
void
ButtonControl::setBitmap( Bitmap* bitmap )
{
    _bitmap = bitmap;
    _useBullet = false;
}

/**
 Sets the background colour of the button when the mouse is not over it.
 */
void
ButtonControl::setBackColour( COLORREF colour )
{
    _backColour = colour;
}

/**
 Sets the radius of the rounded corners of the button.
 */
void
ButtonControl::setRounding(int width)
{
    _roundEllipse = width;
}

/**
 Sets the colour of the bullet. This function will disable the bitmap and
 replace it with a bullet.
 */
void
ButtonControl::setBulletSquare( COLORREF colour )
{
    _useBullet = true;
    _bulletColour = colour;
    format();
}

/**
 Sets the time, in milliseconds, to fade between the fade colour and the back
 colour when the mouse enters or leaves the button.
 */
void
ButtonControl::setFadeTime( unsigned TimeMs )
{
    FadingControl::setFadeTime( TimeMs );
    _textControl.setFadeTime( TimeMs );
}

/**
 Returns the inner TextControl object.
 */
TextControl*
ButtonControl::getText()
{
    return &_textControl;
}

/**
 Sets the font of the inner TextControl object.
 */
void
ButtonControl::setFont( const Font& font )
{
    _textControl.setFont( font );
}

void 
ButtonControl::onLButtonDown( int x, int y, UINT keyFlags )
{
    FadingControl::onLButtonDown(x,y,keyFlags);
    if ( hover() && commandId != 0 ) {
        SendMessage( _hwnd, WM_COMMAND, MAKEWPARAM( commandId, BN_CLICKED ), (LPARAM)_hwnd );
    }

}
