#include "Cards.h"
#include <stdio.h>
#include <windowsx.h>
#include <assert.h>
#include "Fill.h"
#include "TagParser.h"
#include "DrawTimer.h"
#include "win95.h"
#include "debug.h"

using namespace std;

CardDocument::CardDocument() :
    _root(NULL),
    _listener(NULL)
{

}

CardDocument::~CardDocument()
{
    destruct();
}

void 
CardDocument::destruct()
{
    StyleMap::iterator iter = _styles.begin();
    for( ; iter != _styles.end(); ++iter ) {
        delete iter->second;
    }

    _styles.clear();
    _named.clear();

    delete _root;
}

void
CardDocument::clear()
{
    destruct();

    CardStyle* rootStyle = new CardStyle(NULL);
    _styles["root"] = rootStyle;
    _root = new CardElement( NULL );
    _root->_style = rootStyle;
    _root->_listener = this;
}

void
CardDocument::showCard( const std::string& card )
{
    for ( NameMap::iterator iter = _named.begin(); iter != _named.end(); 
            ++iter)
    {
        CardElement* cardptr = iter->second;
        if ( cardptr->_isCard ) {
            if ( iter->first == card ) {
                cardptr->_visible = true;
            } else {
                cardptr->_visible = false;
            }
        }
    }

    if ( _root && _root->_width > 0 ) {
        _root->format( ScreenDC(), _root->_x, _root->_y, _root->_width );
	}
}

void 
CardDocument::show( const std::string& name, bool visible )
{
    NameMap::iterator iter = _named.find(name);
    if ( iter != _named.end() ) {
        iter->second->_visible = visible;        
    }
}


void
CardDocument::setListener( CardEventListener* listener )
{
    _listener = listener;
}

void 
CardDocument::processMouseEvent( CardMouseEvent* event )
{
    if ( _root ) {
        _root->processMouseEvent( event );
    }

    while( !_eventQueue.empty() ) {
        if ( _listener ) {
            _listener->onCardEvent( _eventQueue.front() );
        }
        _eventQueue.pop_front();
    }
}

void 
CardDocument::onCardEvent( const std::string& event )
{
    _eventQueue.push_back( event );
}

void 
CardDocument::parseStyle( CardElement* parent, CTag* tag )
{
    // must have style name
    string name = tag->attributes["name"];

    CardStyle* style = 0;

    StyleMap::iterator siter = _styles.find(name);
    if ( siter == _styles.end() ) {

        string parentName = tag->attributes["parent"];
        CardStyle* parent = NULL;
        if ( !parentName.empty() ) {
            parent = _styles[parentName];
        }

        style = new CardStyle(parent);
        _styles[name] = style;
    } else {
        style = siter->second;
    }

    // children must be attributes -- name, value pairs.
    for( CTag::List::iterator iter = tag->ChildList.begin(); 
            iter != tag->ChildList.end(); ++iter ) 
    {
        CTag* attr = *iter;
        if ( attr->name != "attribute" ) {
            // error!
            continue;
        }

        style->set( attr->attributes["name"], attr->attributes["value"] );
    }
}

void
CardDocument::parseCommon( CardElement* element, CTag* tag )
{
    string name = tag->attributes["name"];
    string stylestr = tag->attributes["style"];
    string onclick = tag->attributes["onclick"];
    string visible = tag->attributes["visible"];
    string width = tag->attributes["width"];
    string height = tag->attributes["height"];

    string marginLeft = tag->attributes["marginleft"];
    string marginRight = tag->attributes["marginright"];
    string marginTop = tag->attributes["margintop"];
    string marginBottom = tag->attributes["marginbottom"];

    if ( !name.empty() ) {
        _named[name] = element;
    }

    if ( !stylestr.empty() ) {
        CardStyle* style = _styles[stylestr];
        if ( style ) {
            element->_style = style;
        }
    }

    if ( !onclick.empty() ) {
        element->onclick = onclick;
    }

    if ( visible == "false" || visible == "0" ) {
        element->_visible = false;
    }

    if ( !width.empty() ) {
        element->_requestedWidth = atoi(width.c_str());
    }
    if ( !height.empty() ) {
        element->_requestedHeight = atoi(height.c_str());
    }

    if ( !marginLeft.empty() ) {
        element->_marginLeft = atoi(marginLeft.c_str());
    }
    if ( !marginTop.empty() ) {
        element->_marginTop = atoi(marginTop.c_str());
    }
    if ( !marginRight.empty() ) {
        element->_marginRight = atoi(marginRight.c_str());
    }
    if ( !marginBottom.empty() ) {
        element->_marginBottom = atoi(marginBottom.c_str());
    }

    for( CTag::List::iterator iter = tag->ChildList.begin(); 
            iter != tag->ChildList.end(); ++iter ) 
    {
        parseTag( element, *iter );
    }
}


void 
CardDocument::parseCard( CardElement* parent,  CTag* tag )
{
    CardElement* element = new CardElement( parent );
    element->_isCard = true;
    parseCommon( element, tag );
}

void
CardDocument::parseDoc( CardElement* parent, CTag* tag )
{
    CardElement* element = new CardElement( parent );
    parseCommon( element, tag );
}

void 
CardDocument::parseP( CardElement* parent, CTag* tag )
{
    CardParagraph* element = new CardParagraph( parent );
    parseCommon( element, tag );
}

void 
CardDocument::parseText( CardElement* parent, CTag* tag )
{
    CardText* element = new CardText( parent );
    element->text = tag->text;
    parseCommon( element, tag );
}

void 
CardDocument::parseButton( CardElement* parent, CTag* tag )
{
    CardButton* element = new CardButton( parent );
    element->text = tag->text;
    parseCommon( element, tag );
}

void 
CardDocument::parseInput( CardElement* parent, CTag* tag )
{
    string type = tag->attributes["type"];

    CardControl* element = NULL;

    if ( type == "slider" ) {
        TrackBarControl tb;
        tb.create( NULL,0, 0, 0, 0, _hwnd, 0 );
        element = new CardControl( parent,tb.hwnd );
    } else if ( type == "combo" ) {
        ComboBox cb;
        cb.create( CBS_DROPDOWNLIST,0, 0, 0, 0, _hwnd, 0 );
        element = new CardControl( parent,cb.hwnd );
    } else if ( type == "progress" ) {
        ProgressBarControl cb;
        cb.create( 0,0, 0, 0, 0, _hwnd, 0 );
        element = new CardControl( parent,cb.hwnd );
    }

    if ( element ) {
        parseCommon( element, tag );
    }
}

void 
CardDocument::parseTag( CardElement* parent, CTag* tag )
{
    if ( tag->name == _T("style") ) {
        parseStyle( parent, tag );
    } else if ( tag->name == _T("doc") ) {
        parseDoc( parent, tag );
    } else if ( tag->name == _T("card") ) {
        parseCard( parent, tag );
    } else if ( tag->name == _T("p") ) {
        parseP( parent, tag );
    } else if ( tag->name == _T("text") ) {
        parseText( parent, tag );
    } else if ( tag->name == _T("button") ) {
        parseButton( parent, tag );
    } else if ( tag->name == _T("input") ) {
        parseInput( parent, tag );
    } else {
        CTag::List& tags = tag->ChildList;
        CTag::List::iterator iter = tags.begin();
        for( ; iter != tags.end(); ++iter ) {
            parseTag( parent, *iter );
        }
    }
}

bool 
CardDocument::loadFromString( HWND hwnd, const std::_tstring& text )
{
    _hwnd = hwnd;
    clear();

    DTD_TABLE_ENTRY dtd[] = {
        { "doc", "style", 0 },
        { "style", "attribute", 0 },
        { "doc", "card", 0 },
        { "card", "p", 0 },
        { "p", "text", 0 },
        { "p", "button", 0 },
        { "p", "input", 0 },
        { "text", "text", DTD_FLAG_CONTAINS_TEXT },
        { "button", "button", DTD_FLAG_CONTAINS_TEXT },
    };
    
    CTagParser parser;
    parser.SetTagRelationships( dtd, sizeof(dtd)/sizeof(*dtd) );
    parser.ParseText( text.c_str() );

    CTag::List& tags = parser.GetTagList();
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        parseTag(_root, *iter);
    }

    return true;
}

bool
CardDocument::loadFromFile( HWND hwnd, const std::_tstring& filename )
{
    FILE* file = fopen( filename.c_str(), "rt" );

    if ( file == 0 ) {
        return false;
    }
    
    _tstring text;
    _TCHAR buffer[1024];

    while( !feof( file ) ) {
        int read = fread( buffer, sizeof(*buffer), sizeof(buffer)/sizeof(*buffer), file );
        text.append( buffer, read );
    }

    fclose( file );

    return loadFromString( hwnd, text );
}

int 
CardDocument::format( DeviceContext dc, int x, int y, int width )
{
    if ( _root && width > 0 ) {
        return _root->format( dc, x, y, width );
	}
    return 0;
}

void 
CardDocument::draw( DeviceContext dc, int xoffset, int yoffset )
{
    if ( _root ) {
        _root->draw( dc, xoffset, yoffset );
    }
}

CardEventListener::~CardEventListener()
{
}


CardMouseEvent::CardMouseEvent( HWND hwnd, Type type, int x, int y, bool down ) :
    down(down),
    hwnd(hwnd),
    type(type),
    x(x),
    y(y),
    ox(0),
    oy(0)
{
}

void 
CardMouseEvent::offset( int ofx, int ofy )
{
    ox += ofx;
    oy += ofy;
}

bool 
CardMouseEvent::inside( int fx, int fy, int width, int height )
{
    return x+ox >= fx && x+ox < fx + width && y+oy >= fy && y+oy < fy + height;
}

void 
CardMouseEvent::invalidate( int fx, int fy, int width, int height )
{
    RECT rect = { fx-ox, fy-oy, fx-ox+width, fy-oy+height };
    InvalidateRect( hwnd, &rect, FALSE );
}

class CardWindowTimer : public TimerEvent
{
public:
    CardWindowTimer( CardWindow* parent) {
        _parent = parent;
        id = 0;
        _set = false;
    }

    unsigned id;
    bool _set;
    CardWindow* _parent;

    void set(unsigned ms) {
        if ( _set ) {
            DrawTimer::CancelEvent( this );
        }

        id = DrawTimer::RegisterEvent( this );
        _set = true;
    }

    void stop() {
        assert( _set );
        if ( _set ) {
            DrawTimer::CancelEvent( this );
        }
        _set = false;
    }

    ~CardWindowTimer() {
        DrawTimer::CancelEvent( this ); 
        _parent = 0;
    }

    virtual void fire( unsigned id ) {
        _parent->onTimer();
    }
};

CardWindow::CardWindow(HINSTANCE hInstance)
{
	_hInstance = hInstance;
	_pszClassName = TEXT("CardWindow");
	_pszTitle = TEXT("Card Window");

    _dwStyle |= WS_VISIBLE | WS_CHILD;  
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
    _doc.setListener( this );
    _timer = new CardWindowTimer( this );
    _sliding = false;
    _listener = NULL;
}


CardWindow::~CardWindow()
{
    delete _timer;
}

void
CardWindow::setListener( CardEventListener* listener )
{
    _listener = listener;
}

void 
CardWindow::onCardEvent( const std::string& event )
{
    if ( _listener ) {
        _listener->onCardEvent( event );
    }
}

bool 
CardWindow::loadFromFile( const std::_tstring& filename )
{
    bool ret =  _doc.loadFromFile( hwnd, filename );
    if ( ret ) {
        RECT rect = getClientRect();
        _doc.format( ScreenDC(), 0, 0, rect.right - rect.left ); 
        InvalidateRect( hwnd, NULL, TRUE );
    }

    return ret;
}

void 
CardWindow::show( const std::string& name, bool visible, bool redraw )
{
    _doc.show( name, visible );
    if ( redraw ) {
        format();
    }
}

void 
CardWindow::format()
{
    RECT rect = getClientRect();
    _doc.format( ScreenDC(), 0, 0, rect.right-rect.left );
    InvalidateRect( hwnd, NULL, FALSE );
}

void
CardWindow::create(HWND hParent)
{
    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                400, 600, hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

LRESULT 
CardWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}


void
CardWindow::onClose(HWND)
{

}

void 
CardWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{

}

BOOL 
CardWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{

    return TRUE;
}

void
CardWindow::onDestroy(HWND hwnd)
{

}


void 
CardWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
CardWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    CardMouseEvent event( hwnd, CardMouseEvent::DOWN, x, y, (keyFlags & MK_LBUTTON) ? true : false );
    _doc.processMouseEvent( &event );
}

void 
CardWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{
    CardMouseEvent event( hwnd, CardMouseEvent::UP, x, y, (keyFlags & MK_LBUTTON) ? true : false );
    _doc.processMouseEvent( &event );
}

void
CardWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    CardMouseEvent event( hwnd, CardMouseEvent::MOVE, x, y, (keyFlags & MK_LBUTTON) ? true : false );
    _doc.processMouseEvent( &event );
}

static void MakeVertex( TRIVERTEX* tri, LONG x, LONG y, COLORREF c )
{
    tri->x = x;
    tri->y = y;
    tri->Red = (COLOR16)((double)GetRValue(c)/255*0xff00);
    tri->Green = (COLOR16)((double)GetGValue(c)/255*0xff00);
    tri->Blue = (COLOR16)((double)GetBValue(c)/255*0xff00);
    tri->Alpha = 0;
}

static void MakeTriangle( GRADIENT_TRIANGLE* tri, ULONG t1, ULONG t2, ULONG t3 )
{
    tri->Vertex1 = t1;
    tri->Vertex2 = t2;
    tri->Vertex3 = t3;
}

void 
CardWindow::showCard( const std::string& name )
{
    _nextCard = name;
    slideAway();
}

void 
CardWindow::slideAway()
{
    RECT rect = getClientRect();
    _timer->set(10);
    _sliding = true;
    _slideFrom = 0;
    _slideTo = -rect.right;
    _slideAt = _slideFrom;
    _slideStart = GetTickCount();
}

void 
CardWindow::slideIn()
{
    RECT rect = getClientRect();
    _timer->set(10);
    _sliding = true;
    _slideFrom = -rect.right;
    _slideTo = 0;
    _slideAt = _slideFrom;
    _slideStart = GetTickCount();
}

void
CardWindow::onTimer()
{
    const double slideTime = 250;

    double percent = (double)(GetTickCount()-_slideStart)/slideTime;
    if ( percent >= 1.0 ) {
        _sliding = false;
        _timer->stop();
        if ( _slideTo < _slideFrom ) {
            _doc.showCard(_nextCard);
            slideIn();
        } else {
            RECT rect = getClientRect();
            InvalidateRect(hwnd, &rect, FALSE );
        }
        return;
    }

    _slideAt = (int)(_slideFrom + percent * (_slideTo-_slideFrom));

    RECT rect = getClientRect();
    InvalidateRect(hwnd, &rect, FALSE );
}

void 
CardWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, 
        bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    RECT rect = getClientRect();

    HDC backDC = _backBuffer.getDC();
    //HRGN region = CreateRectRgnIndirect( &PaintStruct.rcPaint);
    //SelectClipRgn( backDC, region );
    //DeleteObject( region );

    // insert painting code here.
    TRIVERTEX vert[4];
    GRADIENT_TRIANGLE tri[2]; 
    MakeVertex( &vert[0], rect.left, rect.top, RGB(0, 0, 0) );
    MakeVertex( &vert[1], rect.right, rect.top, RGB(32, 32, 32) );
    MakeVertex( &vert[2], rect.right, rect.bottom, RGB(0,0,0) );
    MakeVertex( &vert[3], rect.left, rect.bottom, RGB(128,128,200) );
    MakeTriangle( &tri[0], 0, 1, 2 );
    MakeTriangle( &tri[1], 0, 2, 3 );
    SafeGradientFill( backDC, vert, 4, tri, 2, GRADIENT_FILL_TRIANGLE );

    int offset = 0;
    if ( _sliding ) {
        offset = _slideAt;
    }

    _doc.draw( backDC, offset, 0 );

    RECT ur = PaintStruct.rcPaint;
    _backBuffer.bitBlt( hdc, ur.left, ur.top, ur.right - ur.left,
           ur.bottom - ur.top, ur.left, ur.top ); 


    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
CardWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    _backBuffer.createCompatible( hwnd, cx, cy );
    _doc.format( ScreenDC(), 0, 0, cx );
	InvalidateRect( hwnd, NULL, FALSE );
}


CardStyle::CardStyle( CardStyle* parent ) :
    _parent(parent)
{
}

CardStyle::~CardStyle()
{
}

bool 
CardStyle::lookup( const std::string& key, std::string* value, const std::string& def )
{
    Map::iterator iter = _map.find( key );
    if ( iter != _map.end() ) {
        *value = iter->second;
        return true;
    } else if ( _parent ) {
        return _parent->lookup( key, value, def );
    } else {
        *value = def;
        return false;
    }
}

bool 
CardStyle::lookup( const std::string& key, int* value, int def )
{
    std::string strval;
    *value = def;
    if ( lookup( key, &strval, "" ) ) {
        *value = atoi( strval.c_str() );
        return true;
    } else {
        return false;
    }
}

bool 
CardStyle::lookup( const std::string& key, bool* value, bool def )
{
    std::string strval;
    *value = def;
    if ( lookup( key, &strval, "" ) ) {
        if ( strval == "true" || strval == "1") {
            *value = true;
        } else {
            *value = false;
        }
        return true;
    } else {
        return false;
    }
}

bool 
CardStyle::lookupColour( const std::string& key, DWORD* value, DWORD def )
{
    std::string strval;
    *value = def;
    if ( lookup( key, &strval, "" ) ) {
        unsigned int r, g, b;
        if ( 3 == sscanf( strval.c_str(), "#%02x%02x%02x", &r, &g, &b ) ) {
            *value = RGB(r,g,b);
        } else if ( strval == "white" ) {
            *value = RGB(255,255,255);
        } else if ( strval == "yellow" ) {
            *value = RGB(255,255,0);
        } else if ( strval == "red" ) {
            *value = RGB(255,0,0);
        } else if ( strval == "green" ) {
            *value = RGB(0,255,0);
        } else if ( strval == "gray" ) {
            *value = RGB(128,128,128);
        }
        return true;

    } else {
        return false;
    }
}

void 
CardStyle::set( const std::string& key, const std::string& value )
{
    _map[key] = value;
}

class FadeTimer : public TimerEvent
{
public:
    FadeTimer( FadingCardElement* parent) {
        _parent = parent;
        id = 0;
        _set = false;
    }

    unsigned id;
    bool _set;
    FadingCardElement* _parent;

    void set(unsigned ms) {
        if ( _set ) {
            DrawTimer::CancelEvent( this );
        }

        id = DrawTimer::RegisterEvent( this );
        _set = true;
    }

    void stop() {
        assert( _set );
        if ( _set ) {
            DrawTimer::CancelEvent( this );
        }
        _set = false;
    }

    ~FadeTimer() {
        DrawTimer::CancelEvent( this ); 
        _parent = 0;
    }

    virtual void fire( unsigned id ) {
        _parent->onTimer();
    }
};

FadingCardElement::FadingCardElement( CardElement* parent) :
    CardElement(parent)
{
    _fadeTimeMs = 500;
    _fadeTimer = new FadeTimer( this );
    _fading = false;
    _intensity = 1.0;
    _fadedColour = RGB(128,128,128);
    _hwnd = NULL;
}

FadingCardElement::~FadingCardElement()
{
    delete _fadeTimer;
}

void
FadingCardElement::setFadeTime( unsigned TimeMs )
{
    _fadeTimeMs = TimeMs;
}

void 
FadingCardElement::startFading( HWND hwnd, COLORREF fromColour, COLORREF toColour )
{
    _hwnd = hwnd;
    _fromColour = fromColour;
    _toColour = toColour;
    if ( !_fading ) {
        _fadeTimer->set( 10 );
        _timeStarted = GetTickCount();
        _fading = true;
    } else {
        int elapsed = GetTickCount() - _timeStarted;
        _timeStarted = GetTickCount() + elapsed - _fadeTimeMs;
    }
}

bool 
FadingCardElement::isFading()
{
    return _fading;
}

void
FadingCardElement::invalidate()
{
    int x = 0;
    int y = 0;
    offset( &x, &y );
    RECT rect = {x, y, x+_width, y+_height};
    InvalidateRect( _hwnd, &rect, FALSE );
}

void
FadingCardElement::onTimer()
{
    invalidate();
    if ( GetTickCount() - _timeStarted > _fadeTimeMs ) {
        _fadeTimer->stop();
        _fading = false;
        return;
    }

    _intensity = (double)((unsigned)GetTickCount() - _timeStarted) /
        _fadeTimeMs;
}

COLORREF
FadingCardElement::getFadeColour()
{
    double r,g,b;
    r = (double) GetRValue( _toColour ) + 
        ((double)GetRValue( _fromColour) - GetRValue( _toColour )) * _intensity;  
    g = (double) GetGValue( _toColour ) + 
        ((double)GetGValue( _fromColour) - GetGValue( _toColour )) * _intensity;  
    b = (double) GetBValue( _toColour ) + 
        ((double)GetBValue( _fromColour) - GetBValue( _toColour )) * _intensity;  

    return RGB( (unsigned)r, (unsigned)g, (unsigned)b );
}

CardElement::CardElement( CardElement* parent ) :
    _parent(parent),
    _isinline(false),
    _visible(true),
    _style(NULL),
    _isCard(false),
    _requestedWidth(0),
    _requestedHeight(0)
{
    _x = _y = _width = _height = 0;
    _marginLeft = _marginTop = _marginRight = _marginBottom = 0;
    if ( _parent ) {
        _parent->children.push_back(this);
        _style = _parent->_style;
        _root = _parent->_root;
    } else {
        _root = this;
    }
    _listener = NULL;
}

CardElement::~CardElement()
{
    for( List::iterator iter = children.begin(); iter != children.end(); ++iter ) {
        delete *iter;
    }
}

CardElement*
CardElement::getParent()
{
    return _parent;
}

int 
CardElement::format( DeviceContext dc, int x, int y, int width )
{
    for( List::iterator iter = children.begin(); iter != children.end(); ++iter ) 
    {
        (*iter)->format( dc, x+(*iter)->_marginLeft, y+(*iter)->_marginTop-_y, 
            width - (*iter)->_marginRight - (*iter)->_marginLeft );
        if ( (*iter)->_visible ) {
            y += (*iter)->_height;
        }
    }

    return y;
}

/**
 * Used by child controls to calculate their relative positions. Convert a
 * child's local position to an absolute position in the window.
 *
 * @param x The coordinates passed in will be modified to be relative to the
 * parent controls.
 *
 * @param y
 */
void 
CardElement::offset( int* x, int* y)
{
    if ( _parent ) {
        _parent->offset(x, y);
    }

    *x += _x;
    *y += _y;
}

/**
 * Convert an absolute position in the window to a position relative to the
 * control's parents.
 */
void 
CardElement::unoffset( int* x, int* y)
{
    if ( _parent ) {
        _parent->unoffset(x, y);
    }

    *x -= _x;
    *y -= _y;
}

/**
 * Convert a relative rectangle to an absolute position on the window.
 */
void 
CardElement::offset( RECT* rect )
{
    if ( _parent ) {
        _parent->offset(rect);
    }

    /*
    rect->left += _rect.left;
    rect->top += _rect.top;
    rect->right += _rect.left;
    rect->bottom += _rect.top;
    */
}

void
CardElement::sendEvent( const std::string& event )
{
    if ( _root->_listener ) {
        _root->_listener->onCardEvent( event );
    }
}

void
CardElement::sendClickEvent()
{
    if ( !onclick.empty() ) {
        sendEvent( onclick );
    }
}

void 
CardElement::draw( DeviceContext dc, int offsetX, int offsetY )
{
    for( List::iterator iter = children.begin(); iter != children.end(); ++iter ) 
    {
        if ( (*iter)->_visible ) {
            (*iter)->draw( dc, _x+offsetX, _y+offsetY );
        }
    }
}

void 
CardElement::processMouseEvent( CardMouseEvent* event )
{
    event->offset( -_x, -_y );
    for( List::iterator iter = children.begin(); iter != children.end(); ++iter ) 
    {
        if ( (*iter)->_visible ) {
            (*iter)->processMouseEvent( event );
        }
    }
    event->offset( +_x, +_y );
}

void
CardElement::flow( DeviceContext dc, CardParagraph* paragraph, int width )
{
    format( dc, 0, 0, width );

    int lineNo = paragraph->getLineCount()-1;
    int widthLeft = width - paragraph->getLineWidth( lineNo );
    if ( _width > widthLeft && width != widthLeft) {
        paragraph->newLine();
        lineNo++;
    }

    int endX, endY;
    paragraph->getLineEnd( lineNo, &endX, &endY );
    _x = endX;
    _y = endY;

    paragraph->addToLine( lineNo, this );
}

LineSpan::LineSpan() :
    _x(0),
    _y(0),
    _width(0),
    _height(0)
{

}

LineSpan::LineSpan( int x, int y, int width, int height ) :
    _x(x),
    _y(y),
    _width(width),
    _height(height)
{

}

LineSpan::~LineSpan()
{
}

void 
LineSpan::processMouseEvent( CardMouseEvent* event )
{
}

class CardElementLineSpan : public LineSpan
{
public:
    CardElement* _parent;
    CardElementLineSpan( CardElement* parent ) :
        _parent(parent)
    {
        _x = _parent->_x;
        _y = _parent->_y;
        _width = _parent->_width;
        _height = _parent->_height;
    }

    virtual ~CardElementLineSpan()
    {
    }

    virtual void draw( DeviceContext dc, int offsetX, int offsetY )
    {
        _parent->draw( dc, offsetX, offsetY /*+ _height - _parent->_height*/ );
    }

    virtual void processMouseEvent( CardMouseEvent* event )
    {
        //event->offset( 0, -(_height-_parent->_height) );
        _parent->processMouseEvent( event );
        ///event->offset( 0, (_height-_parent->_height) );
    }
};


CardParagraph::CardParagraph( CardElement* parent ) :
    CardElement( parent )
{

}

CardParagraph::~CardParagraph()
{
    deleteLines();
}

void
CardParagraph::deleteLines()
{
    for( LineList::iterator iter = _lines.begin(); iter != _lines.end(); ++iter ) {
        for( std::list<LineSpan*>::iterator siter = (*iter)->spans.begin();
            siter != (*iter)->spans.end(); ++siter ) 
        {
            delete *siter;
        }
        delete *iter;
    }

    _lines.clear();
}

int
CardParagraph::format( DeviceContext dc, int x, int y, int width )
{
/*
    int margin;
    int marginTop;
    int marginLeft;
    int marginRight;
    _style->lookup("margin", &margin, 0 );
    _style->lookup("margin-left", &marginLeft, margin );
    _style->lookup("margin-right", &marginRight, margin );
    _style->lookup("margin-top", &marginTop, margin );

    width -= marginRight + marginLeft;
    printf("Margin: %d\n", marginTop );
*/
    _width = width;
    _x = x;
    _y = y;


    deleteLines();
    newLine();
    //_lines.back()->y += marginTop;
    //_lines.back()->x += marginLeft;

    for ( List::iterator iter = children.begin(); iter != children.end(); ++iter ) {
        (*iter)->flow( dc, this, width );
    }

    for ( LineList::iterator iter = _lines.begin(); iter != _lines.end(); ++iter ) {
        for( std::list<LineSpan*>::iterator siter = (*iter)->spans.begin();
            siter != (*iter)->spans.end(); ++siter ) 
        {
            (*siter)->_height = (*iter)->height;
        }
    }

    _height = _lines.back()->y + _lines.back()->height + _marginTop + _marginBottom;
    return _height;
}

void 
CardParagraph::draw( DeviceContext dc, int offsetX, int offsetY )
{
    for ( LineList::iterator iter = _lines.begin(); iter != _lines.end(); ++iter ) {
        for( std::list<LineSpan*>::iterator siter = (*iter)->spans.begin();
            siter != (*iter)->spans.end(); ++siter ) 
        {
            (*siter)->draw( dc, offsetX + _x, offsetY + _y );
        }
    }
}

void 
CardParagraph::processMouseEvent( CardMouseEvent* event )
{
    event->offset( -_x, -_y );

    for ( LineList::iterator iter = _lines.begin(); iter != _lines.end(); ++iter ) {
        for( std::list<LineSpan*>::iterator siter = (*iter)->spans.begin();
            siter != (*iter)->spans.end(); ++siter ) 
        {
            (*siter)->processMouseEvent( event );
        }
    }

    event->offset( +_x, +_y );
}

int 
CardParagraph::getLineCount()
{
    return _lines.size();
}

int 
CardParagraph::getLineWidth(int lineno)
{
    return _lines[lineno]->width;
}

void 
CardParagraph::getLineEnd( int lineno, int* x, int* y )
{
    *y = _lines[lineno]->y;
    *x = _lines[lineno]->x + _lines[lineno]->width;
}

void 
CardParagraph::addToLine( int lineno, CardElement* info )
{
    addToLine( lineno, new CardElementLineSpan( info ) );
}

void 
CardParagraph::addToLine( int lineno, LineSpan* info )
{
    if ( info->_height > _lines[lineno]->height ) {
        _lines[lineno]->height = info->_height;
    }

    _lines[lineno]->width += info->_width;
    _lines[lineno]->spans.push_back( info );
}

void 
CardParagraph::newLine()
{
    Line* line = new Line();
    line->x = 0;
    line->y = 0;
    line->width = 0;
    line->height = 0;

    if ( !_lines.empty() ) {
        line->y = _lines.back()->y + _lines.back()->height;
    }

    _lines.push_back( line );
}

class CardTextLineSpan : public LineSpan
{
public:
    CardText* _cardText;
    Font* _font;
    int start;
    int length;
    DWORD colour;
    bool _hover;
    bool _down;
    DWORD _downColour;
    DWORD _hoverColour;

    CardTextLineSpan( int x, int y, int width, int height, CardText* cardText,
            Font* font, DWORD colour ) :
        LineSpan( x, y, width, height ),
        _cardText( cardText ),
        _font( font ),
        start(0),
        length(0),
        colour(colour),
        _hover(false),
        _down(false)
    {
        _cardText->_style->lookupColour( "font.colour#hover", &_hoverColour, colour );
        _cardText->_style->lookupColour( "font.colour#down", &_downColour, _hoverColour );
    }

    virtual ~CardTextLineSpan()
    {

    }

    virtual void draw( DeviceContext dc, int offsetX, int offsetY )
    {
        dc.selectFont( *_font );
        SetBkMode( dc, TRANSPARENT );
        DWORD clr = colour;
        if ( _down ) {
            clr = _downColour;
        } else if ( _hover ) {
            clr = _hoverColour;
        }
        SetTextColor( dc, clr );
        int fheight = _font->getHeight();
        int descent = _font->getDescent();
        TextOut( dc, _x+offsetX, _y+offsetY+_height-fheight, &_cardText->text[start], length );
        if ( _font->getUnderline() ) {
            Pen pen( PS_SOLID, (int)((float)fheight/20+1), clr );
            dc.selectPen( pen );
            MoveToEx( dc, _x+offsetX,_y+offsetY+_height-descent, NULL );
            LineTo( dc, _x+offsetX+_width, _y+offsetY+_height-descent );
        }
    }

    virtual void processMouseEvent( CardMouseEvent* event )
    {
        bool over = event->inside( _x, _y, _width, _height );

        if ( over ^ _hover || over && (event->down ^ _down) ) {
            _hover = over;
            _down = event->down && over;

            event->invalidate( _x, _y, _width, _height );
        }

        if ( over && event->type == CardMouseEvent::UP ) {
            _cardText->sendClickEvent();
        }
    }
};

CardText::CardText( CardElement* parent ) :
    CardElement( parent ),
    _widths(0),
    _font(0)
{
    

}

CardText::~CardText()
{
    delete[] _widths;
    delete _font;
}

void
CardText::createFont()
{
    if ( _font ) {
        return;
    }

    string name;
    int underline;
    int height;
    _style->lookup( "font.name", &name, "Tahoma" );
    _style->lookup( "font.size", &height, 10 );
    _style->lookup( "font.underline", &underline, 0 );

    _font = new Font( name.c_str(), height );
    if ( underline ) {
        _font->setUnderline( true );
    }
}

void
CardText::calcWidths( DeviceContext dc )
{
    if ( _widths ) {
        return;
    }

    createFont();
    dc.selectFont( *_font );
    
    ABCFLOAT widths[256];
    GetCharABCWidthsFloat( dc, 0, 255, widths );
	int a = GetLastError();
    _widths = new float[256];
    for( int i = 0 ;i <= 255; i++ ) {
        _widths[i] = widths[i].abcfA + widths[i].abcfB + widths[i].abcfC;
    }
}

int 
CardText::split( int start, int width, float* total, bool* graceful )
{
    int space = -1;
    float totalAtSpace = 0.0;
    int len = text.length();
    *total = 0.0;
    for( int i = start; i < len; i++ ) {
        if ( text[i] == ' ' ) {
            space = i;
            totalAtSpace = *total;
        }

        *total = *total + _widths[text[i]];
        if ( *total > width ) {
            if ( space > -1 ) {
                *total = totalAtSpace;
                *graceful = true;
                return space-start+1;
            } else {
                *total -= _widths[text[i]];
                *graceful = false;
                return i-start;
            }
        }
    }

    *graceful = true;
    return len-start;
}

void
CardText::makeLine( CardParagraph* paragraph, int lineno, int start, int length, int width )
{
    int endX, endY;
    DWORD colour;
    paragraph->getLineEnd( lineno, &endX, &endY );
    _style->lookupColour( "font.colour", &colour, RGB(0,0,0));
    CardTextLineSpan* info = new CardTextLineSpan( endX, endY, width,
            _font->getHeight(), this, _font, colour );
    info->start = start;
    info->length = length;
    paragraph->addToLine( lineno, info );
}

void
CardText::flow( DeviceContext dc, CardParagraph* paragraph, int width )
{
    // start at last line in paragraph.
    int lineNo = paragraph->getLineCount()-1;

    // get char widths.
    calcWidths( dc );
    int start = 0;
    int widthLeft = width - paragraph->getLineWidth( lineNo );
    float total;
    bool graceful;

    // loop while there are characters.
    while( start < (int)text.length() ) {

        // find how many charaacters will fit into width remaining
        int len = split( start, widthLeft, &total, &graceful );

        // if word couldn't be split gracefully,
        if ( !graceful ) {
            // if not already something on the line, 
            if ( widthLeft == width ) {
                // perform ungraceful split and add to line.
                if ( len == 0 ) {
                    // can't even add one character to line.
                    len = 1;
                }
                makeLine( paragraph, lineNo, start, len, (int)total );
                start += len;
            }
        } else {
            // word break successful. Add to line.
            makeLine( paragraph, lineNo, start, len, (int)total );
            start += len;
            
            // if no more more characters,
            if ( start == text.length() ) {
                // break out of loop.
                break;
             }
        }

        // start a new line and continue.
        paragraph->newLine();
		widthLeft = width;
        lineNo++;
        assert(lineNo < 5000 );
    }
}


CardButton::CardButton( CardElement* parent ) :
    FadingCardElement( parent ),
    _font(0),
    _hover(false),
    _down(false)
{
    

}

CardButton::~CardButton()
{
    delete _font;
}

void
CardButton::createFont()
{
    if ( _font ) {
        return;
    }

    string name;
    int underline;
    int height;
    _style->lookup( "font.name", &name, "Tahoma" );
    _style->lookup( "font.size", &height, 10 );
    _style->lookup( "font.underline", &underline, 0 );
    _style->lookup( "font.colour", &underline, 0 );
    _style->lookupColour( "background-colour", &_backColour, RGB(200,200,200) );
    _style->lookupColour( "background-colour#hover", &_backColourHover,
            _backColour );
    _style->lookupColour( "background-colour#down", &_backColourDown,
            _backColourHover );

    _font = new Font( name.c_str(), height );
    if ( underline ) {
        _font->setUnderline( true );
    }
}

int 
CardButton::format( DeviceContext dc, int x, int y, int width )
{
    createFont();
    dc.selectFont( *_font );
    int mWidth = _font->getCharWidth('M');
    int mHeight = _font->getAscent();

    RECT rect = { 0,0,0,0};

    DrawText( dc, text.c_str(), text.length(), &rect, DT_CALCRECT | DT_SINGLELINE );

    _x = x;
    _y = y;
    _width = mWidth + rect.right + mWidth;
    _height = mHeight + _font->getHeight();
    _buttonPadLeft = mWidth;
    _buttonPadTop = mHeight/2;
    if ( _requestedWidth > 0 && _width < _requestedWidth ) {
        _width = _requestedWidth;
    }

    if ( _requestedHeight > 0 && _height < _requestedHeight ) {
        _height = _requestedHeight;
    }

    return _height;
}

void 
CardButton::draw( DeviceContext dc, int offsetX, int offsetY )
{
    RECT rect = {_x+offsetX, _y+offsetY, _x+offsetX+_width,
        _y+offsetY+_height};
    DWORD clr = _backColour;
    if ( isFading() ) {
        clr = getFadeColour();
    } else if ( _down ) {
        clr = _backColourDown;
    } else if ( _hover ) {
        clr = _backColourHover;
    }
    Brush brush( clr );
    Pen pen( PS_NULL, 1, 0 );
    dc.selectBrush( brush );
    dc.selectPen( pen );

    SetTextColor( dc, _textColour );
    SetBkMode( dc, TRANSPARENT );

    RoundRect( dc, rect.left, rect.top,
            rect.right, rect.bottom,
            8, 8 );

    dc.selectFont( *_font );

    DrawText( dc, text.c_str(), text.length(), &rect, DT_SINGLELINE |
            DT_CENTER | DT_VCENTER);
}

void 
CardButton::processMouseEvent( CardMouseEvent* event )
{
    bool over = event->inside( _x, _y, _width, _height );

    if ( over ^ _hover || over && (event->down ^ _down) ) {
        if ( over && !_hover ) {
            startFading( event->hwnd, _backColourHover, _backColour );
        } else if ( !over && _hover ) {
            startFading( event->hwnd, _backColour, _backColourHover );
        }

        _hover = over;
        _down = event->down && over;

        event->invalidate( _x, _y, _width, _height );
    }

    if ( over && event->type == CardMouseEvent::UP ) {
        sendClickEvent();
    }
}

CardControl::CardControl( CardElement* parent, HWND hControl ) :
    CardElement( parent ),
    _hControl( hControl )
{
    _requestedWidth = 100;
    _requestedHeight = 20;
    _overlappingHeight = 0;
}

CardControl::~CardControl()
{
}

int 
CardControl::format( DeviceContext dc, int x, int y, int width )
{
    _x = x;
    _y = y;
    _width = _requestedWidth;
    _height = _requestedHeight;
    return 0;
}

void 
CardControl::draw( DeviceContext dc, int offsetX, int offsetY )
{
    int x = _x + offsetX;
    int y = _y + offsetY;
    int height = _height;
    if ( _overlappingHeight > height ) {
        height = _overlappingHeight;
    }
    ::MoveWindow( _hControl, x, y, _width, height, FALSE );
}
