#include <assert.h>
#include "Control.h"
#include "BaseWnd.h"

static ControlHandler DefaultHandler;
const unsigned Control::DefaultFadeMs = 250;

/**
 * Contruct a control object.
 * @param hwnd Parent window on which the control will be drawn.
 */
Control::Control(HWND hwnd)
{
    _mouseHover = 0;
    _rect.left = _rect.top = 0;
    _rect.right = _rect.bottom = 0;
    _hwnd = hwnd;
    _handler = &DefaultHandler;
    _parent = 0;
}

Control::~Control()
{
    _hwnd = 0;
}

/**
 * Adds a child control. The positions of children controls are relative to
 * their parents, so they will slide along with their parents too.
 */
void
Control::addChild( Control* child )
{
    assert(!child->_parent);
    _children.addTail( child );
    child->_parent = this;
}

/**
 * Used by child controls to calculate their relative positions. Convert a
 * child's local position to an absolute position in the window.
 *
 * @param x The coordinates passed in will be modified to be relative to the
 * parent controls.
 *
 * @param y
 */
void 
Control::offset( int* x, int* y)
{
    if ( _parent ) {
        _parent->offset(x, y);
    }

    *x += _rect.left;
    *y += _rect.top;
}

/**
 * Convert an absolute position in the window to a position relative to the
 * control's parents.
 */
void 
Control::unoffset( int* x, int* y)
{
    if ( _parent ) {
        _parent->unoffset(x, y);
    }

    *x -= _rect.left;
    *y -= _rect.top;
}

/**
 * Convert a relative rectangle to an absolute position on the window.
 */
void 
Control::offset( RECT* rect )
{
    if ( _parent ) {
        _parent->offset(rect);
    }

    rect->left += _rect.left;
    rect->top += _rect.top;
    rect->right += _rect.left;
    rect->bottom += _rect.top;
}

/**
 * Convert a relative POINT to an absolute position on the window.
 */
void 
Control::offset( POINT* pt )
{
    if ( _parent ) {
        _parent->offset(pt);
    }

    pt->x += _rect.left;
    pt->y += _rect.right;
}

/**
 * Returns the control's rectangle, in absolute coordinates.
 */
RECT 
Control::getWindowRect()
{
    RECT rect = _rect;
    if ( _parent ) {
        _parent->offset( &rect );
    }

    return rect;
}

//#define ROUND 0

/**
 * Draws the control, and all children, onto the given DC. This should be
 * called in the parent window's onPaint() function.
 */
void
Control::draw( HDC hdc )
{
    RECT rect = getWindowRect();
    _fill.draw( hdc, rect.left, rect.top );
    Control* c = _children.first();
    while( c ) {
        c->draw( hdc );
        c = _children.next();
    }
    if ( 0 ) {
        Pen pen(PS_SOLID, 1, RGB(255,0,0));
        DeviceContext dc(hdc);
        dc.selectBrush((HBRUSH)GetStockObject(NULL_BRUSH));
        dc.selectPen(pen);
        Rectangle( dc, _rect.left, _rect.top, _rect.right, _rect.bottom );
    }
}

/**
 * Process an arbitrary windows message.
 */
void
Control::onMessage( unsigned, WPARAM, LPARAM )
{

}

/**
 * Sets the background fill of the control.
 *
 * @param fill A Fill object, which supports gradiants.
 */
void
Control::setBackgroundFill( const Fill& fill )
{
    _fill = fill;
    _fill.setSize( _rect.right - _rect.left,
            _rect.bottom - _rect.top );
}

/**
 * Moves the control to the given position and size, in absolute coordinates.
 */
void
Control::move( int x, int y, int cx, int cy )
{
    assert(_hwnd);
    RECT rect = getWindowRect();
    InvalidateRect( _hwnd, &rect, FALSE );
    _rect.left = x;
    _rect.top = y;
    _rect.right = _rect.left + cx;
    _rect.bottom = _rect.top + cy;
//    InvalidateRect( _hwnd, &_rect, FALSE );
    rect = getWindowRect();
    InvalidateRect( _hwnd, &rect, FALSE );
    _fill.setSize( cx, cy );
}

void
Control::onLButtonDown( int x, int y, UINT keyFlags )
{
    if ( _mouseHover ) {
        _handler->eventMouseClick( this );
    }

    Control* c = _children.first();
    while( c ) {
        c->onLButtonDown( x - _rect.left, y - _rect.right, keyFlags );
        c = _children.next();
    }
}

void
Control::onMouseMove( int x, int y, UINT keyFlags )
{
    if ( x >= _rect.left && x < _rect.right &&
           y >= _rect.top && y < _rect.bottom ) {
       if ( !_mouseHover ) {
          _mouseHover = true;
          onMouseEnter();
       }
    } else {
       if ( _mouseHover ) {
           _mouseHover = false;
          onMouseOut();
       }
    } 

   Control* c = _children.first();
   while( c ) {
       c->onMouseMove( x - _rect.left, y - _rect.top, keyFlags );
       c = _children.next();
   }
}

void
Control::onMouseEnter()
{
    _handler->eventMouseEnter( this );
}

void
Control::onMouseOut()
{
    _handler->eventMouseOut( this );
}

/**
 * Causes the control's rectangle to be invalidated, and eventually repainted
 * by Windows.
 */
void
Control::invalidate()
{
    assert(_hwnd);
    RECT rect = getWindowRect();
    InvalidateRect( _hwnd, &rect, FALSE );
}

/**
 * Determins if the mouse is currently hovering over the window.
 */
bool
Control::hover()
{
    return _mouseHover;
}

ControlHandler::ControlHandler()
{
}

ControlHandler::~ControlHandler()
{

}

void
ControlHandler::eventMouseEnter( Control* )
{

}

void
ControlHandler::eventMouseOut( Control* )
{

}

void
ControlHandler::eventMouseClick( Control* )
{

}

void
Control::setHandler( ControlHandler* handler )
{
    _handler = handler;
}



