//*******************************************************************/
//DibImage class
//By Steve Hanov (smhanov@uwaterloo.ca)
//April 2000
//
// DIBWrapper.
//******************************************************************/
//Adapted from:
//Petzold, Charles. PROGRAMMING WINDOWS. (Fifth Edition) Microsoft Press, 1999.

#include "windows.h"
#include <assert.h>
#include "DibImage.h"
#include <math.h>
#include <new>
#include "Stream.h"

#define USE_JPEG 
#define USE_PNG 

#ifdef USE_PNG
#include "png.h"
#endif

#ifdef USE_JPEG
extern "C" {
#include "jpeglib.h"
}
#endif

Canvas::Canvas()
{

}

Canvas::~Canvas()
{

}

/// returns the colour of a pixel.
unsigned
Canvas::getPixel(int x, int y)
{
    return 0;
}


/// Sets the colour of a pixel.
void
Canvas::setPixel( int x, int y, unsigned colour)
{
}

bool
SavePng( Canvas* canvas, Stream* stream );

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

DibImage::DibImage()
{
	_bCreated = false;
	memset(&_dib, 0, sizeof(_dib));
    _cacheSize = 0;
    _cache = 0;
}

DibImage::~DibImage()
{
    clearCache();
	Delete();
    _cache = 0;
}

BITMAPINFO* 
DibImage::getBitmapInfo()
{
    if ( !_bCreated ) {
        assert(0);
        return 0;
    }

    return _dib.pbmi;
}

unsigned char* 
DibImage::getRawBits()
{
    return _dib.pBits;
}


/**
 * Draw one DibImage ontop of another, with transparency. This object will be
 * the target image.
 *
 * @param x Position to draw on target image.
 *
 * @param y Position to draw on target image.
 *
 * @param pImage source image
 *
 * @param source Rectangle to draw from source image.
 *
 * @double transparency A number between 0.0 and 1.0, to specify the
 * transparency.
 */
bool DibImage::Overlay(unsigned int x, unsigned int y, DibImage* pImage, RECT* source, double transparency)
{
	if( !pImage || !pImage->_bCreated || !_bCreated || !source || transparency <= 0)
		return false;

	RECT clippedSource, sourceRect;
	pImage->GetRect(&sourceRect);
	IntersectRect(&clippedSource, &sourceRect, source);

	RECT clippedDest, destRect, dest;
	dest.left = x;
	dest.top = y;
	dest.right = dest.left + clippedSource.right - clippedSource.left;
	dest.bottom = dest.top + clippedSource.bottom - clippedSource.top;
	GetRect(&destRect);
	IntersectRect(&clippedDest, &destRect, &dest);

	int width = dest.right - dest.left;
	int height = dest.bottom - dest.top;

	if(transparency == 1.0)
	{
		//simple copy.
		for(int x = 0; x < width; x++)
			for(int y = 0; y < height; y++)
			{
				COLORREF colour;
				colour = pImage->getPixel(x + source->left, y + source->top);
				setPixel(x + dest.left, y + dest.top, colour);
			}
	}
	else
	{
		double inv = 1.0 - transparency;
		//we have to do math.
		for(int x = 0; x < width; x++)
			for(int y = 0; y < height; y++)
			{
				COLORREF srccolour, destcolour;
				srccolour = pImage->getPixel(x + source->left, y + source->top);
				destcolour = getPixel(x + dest.left, y + dest.top);
				int r = GetBValue(srccolour);
				r = (int)(transparency * r);
				destcolour = RGB((BYTE)(inv * GetRValue(destcolour)) + (BYTE)(transparency * GetRValue(srccolour)), (BYTE)(inv * GetGValue(destcolour)) + (BYTE)(transparency * GetGValue(srccolour)), (BYTE)(inv * GetBValue(destcolour)) + (BYTE)(transparency * GetBValue(srccolour)));
				setPixel(x + dest.left, y + dest.top, destcolour);
			}
	}

	return true;
}


/**
 * Obtain the width and height of the image, in RECT form.
 */
bool DibImage::GetRect(RECT* rect)
{
	if(!rect || !_bCreated)
		return false;

	rect->left = rect->top = 0;
	rect->right = _dib.pbmi->bmiHeader.biWidth;
	rect->bottom = _dib.pbmi->bmiHeader.biHeight;

	return true;
}

/**
 * Replace the contents of this DibImage with the given rectangle extracted
 * from a Device context.
 */
bool DibImage::GetFromDC(HDC sourceDC, RECT* source)
{
	Delete();

	if(!source)
		return false;

	unsigned int width = source->right - source->left;
	unsigned int height = source->bottom - source->top;

	if(!Create(width, height, 24, 0))
		return false;

	HDC destDC	= CreateCompatibleDC(sourceDC);
	HBITMAP hOld = (HBITMAP)SelectObject(destDC, _dib.hBitmap);

	BOOL bBlt = BitBlt(destDC, 0, 0, width, height, sourceDC, source->left, source->top, SRCCOPY);

	SelectObject(destDC, hOld);
	DeleteDC(destDC);

	return bBlt != 0;
}

/**
 * Replaces the contexts of the DibImage with the given bitmap. The screen DC
 * is used for the conversion.
 */
bool 
DibImage::createFromBitmap( HBITMAP hBitmap )
{
    BITMAP bmp;
    int cx, cy;
    if ( GetObject( (HGDIOBJ)hBitmap, sizeof(bmp), &bmp ) ) {
        cx = bmp.bmWidth;
        cy = bmp.bmHeight;
    } else {
        return false;
    }

    HDC dcScreen = GetDC(NULL);
    HDC hDC = CreateCompatibleDC( dcScreen );
    ReleaseDC( NULL, dcScreen );
    HBITMAP hOldBitmap = (HBITMAP)SelectObject( hDC, hBitmap );
    RECT rect = {0,0,cx,cy};
    BOOL success = GetFromDC( hDC, &rect );   
    SelectObject( hDC, hOldBitmap );
    DeleteDC( hDC );
    return success == TRUE;
}

/**
 * Creates an empty DibImage.
 *
 * @param cx Width
 * @param cy Height
 *
 * @param cBits Bit depth. Despite appearance of the code, only 24 bits work.
 *
 * @param cColours Must be 0
 */
bool DibImage::Create(int cx, int cy, int cBits, int cColors)
{
	BITMAPINFO *pbmi;
	DWORD dwInfoSize;
	int cEntries;

	if(cx <= 0 || cy <= 0 ||
		((cBits != 1) && (cBits != 4) && (cBits != 8) &&
		 (cBits != 16) && (cBits != 24) && (cBits != 32)))
	{
		return false;
	}

	if(cColors != 0)
		cEntries = cColors;
	else if(cBits <= 8)
		cEntries = 1 << cBits;
	else
		cEntries = cx * cy;

	dwInfoSize = sizeof(BITMAPINFOHEADER) + (cEntries - 1) * sizeof(RGBQUAD);

    //lint -e{433}
	if( NULL == (pbmi = (BITMAPINFO*)malloc(dwInfoSize)))
	{
		return false;
	}

    //lint -e{669}
	ZeroMemory(pbmi, dwInfoSize);

	pbmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pbmi->bmiHeader.biWidth = cx;
	pbmi->bmiHeader.biHeight = cy;
	pbmi->bmiHeader.biPlanes = 1;
	pbmi->bmiHeader.biBitCount = cBits;
	pbmi->bmiHeader.biCompression = BI_RGB;
	pbmi->bmiHeader.biSizeImage = 0;
	pbmi->bmiHeader.biXPelsPerMeter = 0;
	pbmi->bmiHeader.biClrUsed = cColors;
	pbmi->bmiHeader.biClrImportant = 0;
	
	_bCreated = CreateFromInfo(pbmi);

	return _bCreated;
}

/** 
 * Destroy the DibImage, and free all memory.
 */
bool DibImage::Delete()
{
	if(!_bCreated)
		return false;

	free(_dib.ppRow);
	DeleteObject(_dib.hBitmap);
	free(_dib.pbmi);
	memset(&_dib, 0, sizeof(_dib));

	_bCreated = false;
	
	return true;
}

bool DibImage::CreateFromInfo(BITMAPINFO * pbmi)
{
	Delete();

	BYTE* pBits;
	HBITMAP hBitmap;
	int i, iRowLength, cy, y;

	hBitmap = CreateDIBSection(NULL, pbmi, DIB_RGB_COLORS, (void**)&pBits, NULL, 0);

	if(hBitmap == NULL) {
        free( pbmi );
		return false;
    }

	_dib.hBitmap = hBitmap;
	_dib.pBits = pBits;

	GetObject(hBitmap, sizeof(DIBSECTION), &_dib.ds);

	if(_dib.ds.dsBmih.biCompression == BI_BITFIELDS)
	{
		for(i = 0; i < 3; i++)
		{
			_dib.iLShift[i] = MaskToLShift(_dib.ds.dsBitfields[i]);
			_dib.iRShift[i] = MaskToRShift(_dib.ds.dsBitfields[i]);
		}
	}
	else if(_dib.ds.dsBmih.biCompression == BI_RGB)
	{
		if(_dib.ds.dsBm.bmBitsPixel == 16)
		{
			_dib.ds.dsBitfields[0] = 0x00007C00;
			_dib.ds.dsBitfields[1] = 0x000003E0;
			_dib.ds.dsBitfields[2] = 0x0000001F;

			_dib.iRShift[0] = 10;
			_dib.iRShift[1] = 5;
			_dib.iRShift[2] = 0;
			
			_dib.iLShift[0] = 3;
			_dib.iLShift[1] = 3;
			_dib.iLShift[2] = 3;
		}
		else if(_dib.ds.dsBm.bmBitsPixel == 24 || _dib.ds.dsBm.bmBitsPixel == 32)
		{
			_dib.ds.dsBitfields[0] = 0x00FF0000;
			_dib.ds.dsBitfields[1] = 0x0000FF00;
			_dib.ds.dsBitfields[2] = 0x000000FF;

			_dib.iRShift[0] = 16;
			_dib.iRShift[1] = 8;
			_dib.iRShift[2] = 0;
			
			_dib.iLShift[0] = 0;
			_dib.iLShift[1] = 0;
			_dib.iLShift[2] = 0;
		}
	}

	cy = _dib.ds.dsBm.bmHeight;

	if(NULL == (_dib.ppRow = (unsigned char**)malloc(cy * sizeof(BYTE*))))
	{
		DeleteObject(hBitmap);
        free( pbmi );
		return false;
	}

	iRowLength = 4 * ((_dib.ds.dsBm.bmWidth * _dib.ds.dsBm.bmBitsPixel + 31) / 32);

	if(pbmi->bmiHeader.biHeight > 0)
	{
		//bottom up
		for(y = 0; y < cy; y++)
			_dib.ppRow[y] = pBits + (cy - y - 1) * iRowLength;
	}
	else
	{
		for(y = 0; y < cy; y++)
			_dib.ppRow[y] = pBits + y * iRowLength;
	}

	_dib.pbmi = pbmi;

	return true;
}

int DibImage::MaskToRShift(DWORD dwMask)
{
	int iShift;
	if(dwMask == 0)
		return 0;

	for(iShift = 0; !(dwMask & 1); iShift++)
		dwMask >>= 1;
	 
	return iShift;
}

int DibImage::MaskToLShift(DWORD dwMask)
{
	int iShift;

	if(dwMask == 0)
		return 0;

	while(!(dwMask & 1))
		dwMask >>= 1;

	for(iShift = 0; dwMask & 1; iShift++)
		dwMask >>= 1;

	return 8 - iShift;
}

/**
 * Display a section of the DibImage on the given device context, possibly
 * stretching it. Note this uses StretchDIBits to do the stretching which often
 * looks bad.
 *
 * @param dc Device context
 *
 * @param src Source rectangle
 *
 * @param dest Destination rectangle.
 */
bool
DibImage::stretch( HDC dc, RECT* src, RECT* dest )
{
    if ( !_bCreated ) {
        return false;
    }

    int iRet = StretchDIBits( dc, 
            dest->left, dest->top,
            dest->right - dest->left, dest->bottom - dest->top,
            src->left, height()-src->bottom,
            src->right - src->left, src->bottom - src->top,
            _dib.pBits, _dib.pbmi, DIB_RGB_COLORS, SRCCOPY );

    return iRet != GDI_ERROR;
}


/** 
 * Display the DibImage, without stretching, to the given device context.
 *
 * @param dc Device context.
 *
 * @param xdest destination position.
 *
 * @param yest Destination position
 *
 * @param cxClient Destination size
 *
 * @param cyClient Destination size
 *
 * @param xsrc Source position
 *
 * @param ysrc Source position.
 */
bool DibImage::display(HDC dc, int xdest, int ydest, int cxClient, int
        cyClient, int xsrc, int ysrc)
{
	if(!_bCreated)
		return false;

	int iReturn;

	iReturn = SetDIBitsToDevice(dc, xdest, ydest, cxClient, cyClient, xsrc,
            height() - ysrc - cyClient, 0, height(), _dib.pBits, _dib.pbmi, DIB_RGB_COLORS);

	return iReturn != 0;
}

/// Returns the width of the image.
int
DibImage::width()
{
    return _dib.ds.dsBm.bmWidth;
}

/// Returns the height of the image.
int 
DibImage::height()
{
    return _dib.ds.dsBm.bmHeight;
}

unsigned DibImage::getPixel(int x, int y)
{
	if(!_bCreated)
		return 0;

    assert( _dib.ds.dsBm.bmBitsPixel == 24 );
    return GetPixel(x,y);
}
	
void DibImage::setPixel(int x, int y, unsigned color)
{
	if(!_bCreated)
		return;

	RGBQUAD quad;
	quad.rgbBlue = GetBValue(color);
	quad.rgbRed = GetRValue(color);
	quad.rgbGreen = GetGValue(color);
	quad.rgbReserved = 0;

	int iBitCount;
	DWORD dwPixel;

	iBitCount = _dib.ds.dsBm.bmBitsPixel;

	if(iBitCount <= 8)
		return;
	else if(iBitCount == 24)
	{
		*(RGBTRIPLE*) &dwPixel = *(RGBTRIPLE*)&quad;
		dwPixel &= 0x00FFFFFF;
	}
	else if(iBitCount == 32 &&
		_dib.ds.dsBmih.biCompression == BI_RGB)
	{
		*(RGBQUAD*)& dwPixel = quad;
	}
	else
	{
		dwPixel = (((DWORD)quad.rgbRed >> _dib.iLShift[0])
			<< _dib.iRShift[0]);

		dwPixel |= (((DWORD)quad.rgbGreen >> _dib.iLShift[1])
			<< _dib.iRShift[1]);

		dwPixel |= (((DWORD)quad.rgbBlue >> _dib.iLShift[2])
			<< _dib.iRShift[2]);
	}

	SetPixel(x, y, dwPixel);
}
	
/// Sets the colour of a pixel in the image.	
bool DibImage::SetPixel(int x, int y, DWORD dwPixel)
{
	PBYTE pPixel;
    char* c = (char*)&dwPixel;

	pPixel = PixelPtr(x, y);
    if ( pPixel == 0 ) {
		return false;
    }

	switch(_dib.ds.dsBm.bmBitsPixel)
	{
	case 24:
        pPixel[0] = c[0];
        pPixel[1] = c[1];
        pPixel[2] = c[2];
		break;

	default:
		return false;
	}

	return true;
}

/// Returns the colour of a pixel in the image.
DWORD DibImage::GetPixel(int x, int y)
{
	PBYTE pPixel;

	pPixel = PixelPtr(x, y);
    if ( pPixel == 0 ) {
		return false;
    }

	//pPixel = _dib.ppRow[y] + (x * _dib.ds.dsBm.bmBitsPixel >> 3);
	//return 0x00FFFFFF & *(DWORD*)pPixel;

	switch(_dib.ds.dsBm.bmBitsPixel)
	{
	case 24:
        return RGB(pPixel[2], pPixel[1], pPixel[0]);
    default:
        assert(0);
	}

	return 0;
}

BYTE* 
DibImage::PixelPtr(int x, int y)
{
	if(x < 0 || x >= _dib.ds.dsBm.bmWidth || y < 0 || y >= _dib.ds.dsBm.bmHeight)
		return 0;

	return _dib.ppRow[y] + (x * (_dib.ds.dsBm.bmBitsPixel >> 3));
}

/**
 * Create a windows region object, which is the same size of the DibImage, and
 * consists of all the colours that are NOT similar to the given colour.
 *
 * @param cTransparentColour Colour of pixels not in the region.
 *
 * @param cTolerance Tolerance. Eg. RGB(1,1,1) will consider pixels that are 1
 * colour value different from cTransparentColour to still be transparent.
 *
 * @param matrix Transformation matrix that will be applied to the returned
 * region (eg. to shrink or expand it) See MSDN documentation.
 */
HRGN DibImage::createRegion(COLORREF cTransparentColor, COLORREF cTolerance,
        XFORM* matrix)
{
	HRGN hRgn = NULL;
					
    // For better performances, we will use the ExtCreateRegion() function to
    // create th region. This function take a RGNDATA structure on entry. We
    // will add rectangles b amount of ALLOC_UNIT number in this structure
#define ALLOC_UNIT	100
    DWORD maxRects = ALLOC_UNIT;
    HANDLE hData = GlobalAlloc(GMEM_MOVEABLE, 
            sizeof(RGNDATAHEADER) + (sizeof(RECT) * maxRects));
    RGNDATA *pData = (RGNDATA *)GlobalLock(hData);
    pData->rdh.dwSize = sizeof(RGNDATAHEADER);
    pData->rdh.iType = RDH_RECTANGLES;
    pData->rdh.nCount = pData->rdh.nRgnSize = 0;
    SetRect(&pData->rdh.rcBound, MAXLONG, MAXLONG, 0, 0);
    
    // Keep on hand highest and lowest values for the "transparent" pixel
    BYTE lr = max(0, 
            (signed)GetRValue(cTransparentColor) - GetRValue(cTolerance) );
    BYTE lg = max(0, 
            (signed)GetGValue(cTransparentColor) - GetGValue(cTolerance) );
    BYTE lb = max(0, 
            (signed)GetBValue(cTransparentColor) - GetBValue(cTolerance) );
    BYTE hr = min((unsigned)0xff, (unsigned)lr + GetRValue(cTolerance));
    BYTE hg = min((unsigned)0xff, (unsigned)lg + GetGValue(cTolerance));
    BYTE hb = min((unsigned)0xff, (unsigned)lb + GetBValue(cTolerance));

    int width = DibImage::width();
    int height = DibImage::height();
    
    // Scan each bitmap row from bottom to top (the bitmap is inverted
    // vertically
    for (int y = 0; y < height; y++)
    {
        // Scan each bitmap pixel from left to right
        for (int x = 0; x < width; x++)
        {
            // Search for a continuous range of "non transparent pixels"
            int x0 = x;
            COLORREF p = getPixel( x, y );
            while (x < width)
            {
                BYTE b = GetRValue(p);
                if (b >= lr && b <= hr)
                {
                    b = GetGValue(p);
                    if (b >= lg && b <= hg)
                    {
                        b = GetBValue(p);
                        if (b >= lb && b <= hb)
                            // This pixel is "transparent"
                            break;
                    }
                }
                x++;
                p = getPixel( x, y );
            }
            
            if (x > x0)
            {
                // Add the pixels (x0, y) to (x, y+1) as a new rectangle in
                // the region
                if (pData->rdh.nCount >= maxRects)
                {
                    GlobalUnlock(hData);
                    maxRects += ALLOC_UNIT;
                    hData = GlobalReAlloc(hData, sizeof(RGNDATAHEADER) + 
                            (sizeof(RECT) * maxRects), GMEM_MOVEABLE);
                    pData = (RGNDATA *)GlobalLock(hData);
                }
                //lint -e{826}
                RECT *pr = (RECT *)pData->Buffer;
                SetRect(&pr[pData->rdh.nCount], x0, y, x, y+1);
                if (x0 < pData->rdh.rcBound.left)
                    pData->rdh.rcBound.left = x0;
                if (y < pData->rdh.rcBound.top)
                    pData->rdh.rcBound.top = y;
                if (x > pData->rdh.rcBound.right)
                    pData->rdh.rcBound.right = x;
                if (y+1 > pData->rdh.rcBound.bottom)
                    pData->rdh.rcBound.bottom = y+1;
                pData->rdh.nCount++;
                
                // On Windows98, ExtCreateRegion() may fail if the number of
                // rectangles is to large (ie: > 4000). Therefore, we have to
                // create the region by multiple steps
                if (pData->rdh.nCount == 2000)
                {
                    HRGN h = ExtCreateRegion(matrix, sizeof(RGNDATAHEADER) + 
                            (sizeof(RECT) * maxRects), pData);
                    if (hRgn)
                    {
                        CombineRgn(hRgn, hRgn, h, RGN_OR);
                        DeleteObject(h);
                    }
                    else
                        hRgn = h;
                    pData->rdh.nCount = 0;
                    SetRect(&pData->rdh.rcBound, MAXLONG, MAXLONG, 0, 0);
                }
            }
        }
    }
        
    // Create or extend the region with the remaining rectangle
    HRGN h = ExtCreateRegion(matrix, sizeof(RGNDATAHEADER) + 
            (sizeof(RECT) * maxRects), pData);
    if (hRgn)
    {
        CombineRgn(hRgn, hRgn, h, RGN_OR);
        DeleteObject(h);
    }
    else
        hRgn = h;
    
    // Clean up
    GlobalUnlock(hData);
    GlobalFree(hData);
	
	return hRgn;
}

/**
 * Returns a copy of the image in the form of a bitmap image, using the given
 * device context for the conversion.
 */
HBITMAP
DibImage::createBitmap( HDC dc )
{
    // NOTE: This function doesn't work!
    assert( _bCreated );

    HBITMAP hBitmap = CreateDIBitmap( dc,
            &_dib.pbmi->bmiHeader,
            CBM_INIT,
            _dib.pBits,
            _dib.pbmi,
            DIB_RGB_COLORS
            );

    return hBitmap;
}

/**
 * Creates a copy of the DibImage that is scaled to the given width and height.
 * Nearest neighbour scaling is used, which does not create any new colours and
 * is therefore suitable for transparency masks.
 */
DibImage* 
DibImage::createScaled( int x, int y )
{
    assert( _bCreated );

    DibImage* dib = new DibImage();

    if ( x <= 0 || y <= 0 || !_bCreated ) {
        return dib;
    }

    dib->Create( x, y, 24, 0 );
    scaleBilinear( dib );
    // Note: Hotkey Jumpstart MUST use the nearest neighbour scaling, since
    // bilinear changes the transparent colours.
    //scaleNearestNeighbour( dib );
    return dib;
}

void
DibImage::scaleNearestNeighbour( DibImage* dib )
{
    double scaleX = (double)width() / dib->width();
    double scaleY = (double)height() / dib->height();
    int w = width();
    int h = height();
    
    for( int y = 0; y < dib->height(); y++ ) {
        int srcY = (int)((double)y * scaleY);

        for( int x = 0; x < dib->width(); x++ ) {
            int srcX = (int)((double)x * scaleX);

            dib->setPixel(x, y, getPixel(srcX, srcY)); 
        }
    }
}

static inline void addC( COLORREF* first, double d, COLORREF* second )
{
    unsigned char* f = (unsigned char*)first;
    unsigned char* s = (unsigned char*)second;
    f[0] = (int)((double)f[0] + d * s[0]);
    f[1] = (int)((double)f[1] + d * s[1]);
    f[2] = (int)((double)f[2] + d * s[2]);
} 

void
DibImage::scaleBilinear( DibImage* dib )
{
    double scaleX = (double)width() / dib->width();
    double scaleY = (double)height() / dib->height();
    int sheight = height();
    
    for( int y = 0; y < dib->height(); y++ ) {
        for( int x = 0; x < dib->width(); x++ ) {
            double sx = (double)x * scaleX;
            double dx = sx - floor( sx );
            int ix = (int)sx;
            double sy = (double)y * scaleY;
            double dy = sy - floor( sy );
            int iy = (int)sy;

            double f1, f2;
            int xdiff = 0;
            COLORREF c1, c2, c3, c4, xcol1, xcol2, ycol;

            // calculate X factors
            c1 = getPixel( ix, iy ); 
            if ( dx < 0.5 ) {
                f1 = 1.0 - dx;
                if ( ix > 0 ) {
                    xdiff = -1;
                    c2 = getPixel( ix - 1, iy );
                } else {
                    c2 = c1;
                }
            } else {
                f1 = dx;
                if ( ix < sheight - 1 ) {
                    xdiff = 1;
                    c2 = getPixel( ix + 1, iy );
                } else {
                    c2 = c1;
                }
            }

            if ( dy < 0.5 ) {
                f2 = 1.0 - dy;
                if ( iy > 0 ) {
                    c3 = getPixel( ix, iy - 1 );
                    c4 = getPixel( ix + xdiff, iy - 1 );
                } else {
                    c3 = getPixel( ix, iy );
                    c4 = getPixel( ix + xdiff, iy );
                }
            } else {
                f2 = dy;
                if ( iy < sheight - 1 ) { 
                    c3 = getPixel( ix, iy + 1 );
                    c4 = getPixel( ix + xdiff, iy + 1 );
                } else {
                    c3 = getPixel( ix, iy );
                    c4 = getPixel( ix + xdiff, iy );
                }
            }

            xcol1 = 0;
            addC( &xcol1, f1, &c1 );
            addC( &xcol1, 1.0 - f1, &c2 );

            xcol2 = 0;
            addC( &xcol2, f1, &c3 );
            addC( &xcol2, 1.0 - f1, &c4 );

            ycol = 0;
            addC( &ycol, f2, &xcol1 );
            addC( &ycol, 1.0 - f2, &xcol2 );
            
            dib->setPixel(x, y, ycol); 
        }
    }
}

void 
DibImage::freeCacheEntry(CacheEntry*& entry)
{
    if ( entry ) {
        delete entry->dib;
        delete entry;
        entry = 0;
    }
}

//lint -sem(freeCacheEntry,custodial(1))
void
DibImage::clearCache()
{
    if ( _cache == 0 ) {
        return;
    }
    
    for( int i = 0; i < _cacheSize; i++ ) {
        freeCacheEntry( _cache[i] );
    }
    free( _cache );
    _cache = 0;
    _cacheSize = 0;
}

/**
 * Draws the DibImage in to the device context, possible stretched or shrunk.
 * This function will perform the stretching using the nearest neighbour method. To save
 * resources, the stretched image is stored in a cache.
 *
 * @param dc Destination device context
 *
 * @param src Source rectangle.
 *
 * @param dest Destination rectangle, possibly different size than src.
 *
 * @param maxCache Maximum size of the cache. Set to 0 to disable caching.
 */ 
void
DibImage::cacheStretch( HDC dc, RECT* src, RECT* dest, int maxCache )
{
    // if maxCache is 0, then clear the cache and then do a stretch.
    if ( maxCache == 0 ) {
        clearCache();
        stretch( dc, src, dest );
    }

    if ( maxCache < 0 ) {
        assert(0);
        maxCache = 0;
    }

    // if the maxCache is larger than the cache size,
    if ( maxCache > _cacheSize ) {
        // realloc the cache. On failure through std::bad_alloc.
        _cache = (CacheEntry**)realloc( _cache, sizeof( *_cache ) * maxCache );
        if ( _cache == 0 ) {
            throw std::bad_alloc();
        }

        // set entry _cacheSize .. maxCache-1 to 0.
        for ( int i = _cacheSize; i < maxCache; i++ ) {
            _cache[i] = 0;
        }
        
        // set cache size to maxCache.
        _cacheSize = maxCache;

    } else if ( maxCache < _cacheSize && _cache ) {
        // the maxCache is smaller than the cache size.
        // clear entries from maxCache .. _cacheSize - 1.
        for ( int i = maxCache; i < _cacheSize - 1; i++ ) {
            freeCacheEntry( _cache[i] );
        }

        // set cache size to maxCache.
        _cacheSize = maxCache;
    }

    assert( _cacheSize == maxCache );
    if ( _cache == 0 ) {
        assert(0);
        return;
    }

    // if no stretch is required, just do the display.
    if ( src->right - src->left == dest->right - dest->left &&
            src->bottom - src->top == dest->bottom - dest->top )
    {
        display(dc, dest->left, dest->top, dest->right - dest->left,
                dest->bottom - dest->top, src->left, src->top );
        return;
    }
    
    int count = 0;
    int index = -1;

    double scaleX = (double)(dest->right-dest->left)/(src->right - src->left);
    double scaleY = (double)(dest->bottom-dest->top)/(src->bottom - src->top);
    unsigned int newX = (int)(scaleX*width());
    unsigned int newY = (int)(scaleY*height());

    // for each entry in the cache,
    for ( int i = 0; i < _cacheSize; i++ ) {
        // if it exists, and the src rects are equal and the size is equal,
        if ( _cache[i] ) {
          count++;
          if ( _cache[i]->size.cx == newX && _cache[i]->size.cy == newY ) 
          {
              // we've found it! keep going, though, to count the entries.
              index = i;
          }
        } else {
            // no more entries in the cache. break out.
            break;
        }
    }

    // if not found,
    if ( index == -1 ) {
        // create a new cache entry.
        CacheEntry* entry = new CacheEntry;
        // scale the image into the cache entry.
        memcpy( &entry->src, src, sizeof( RECT ) );
        entry->size.cx = newX;
        entry->size.cy = newY;
        entry->dib = createScaled( newX, newY );

        // if the number of images in the cache is maxCache,
        if ( count == maxCache ) {
            // delete the last image. Decrement count.
            freeCacheEntry( _cache[count-1] );
            count--;
        }

        assert( count < _cacheSize );
        //lint -e{661}
        _cache[count++] = entry;
        index = count - 1;
    }

    // if it's not the first image, 
    if ( index != 0 ) {
        // swap index with 0.
        //lint -e{661}
        CacheEntry* temp = _cache[index];
        //lint -e{661}
        _cache[index] = _cache[0];
        _cache[0] = temp; 
    }

    // we now have an image. Just display it.
    _cache[0]->dib->display(dc, dest->left, dest->top, 
            dest->right - dest->left, dest->bottom - dest->top, 
            (int)(src->left * scaleX),
            (int)(src->top * scaleY) );
}

static DibImage* LoadPng( const TCHAR* fileName );
static DibImage* LoadJpeg( const TCHAR* fileName );
static DibImage* LoadPng( Stream* stream );

/**
 * Load the image from the given file. The file extension is not used. First,
 * PNG is tried, and if that fails, JPEG is tried.
 */
DibImage* DibImage::Load( const TCHAR* fileName )
{
    DibImage* dib = 0;

    dib = ::LoadPng( fileName );

    if ( !dib ) {
        dib = LoadJpeg( fileName );
    }

    return dib;
}

#ifdef USE_PNG

void PNGAPI PngWrite( png_structp png_ptr, png_bytep bytes, png_size_t size )
{
    Stream* stream = static_cast<Stream*>(png_ptr->io_ptr);

    stream->write( bytes, 1, size );
}

void PNGAPI PngRead( png_structp png_ptr, png_bytep bytes, png_size_t size )
{
    Stream* stream = static_cast<Stream*>(png_ptr->io_ptr);
    stream->read( bytes, 1, size );
}

/**
 * Read a PNG file.  You may want to return an error code if the read
 * fails (depending upon the failure).  There are two "prototypes" given
 * here - one where we are given the filename, and we need to open the
 * file, and the other where we are given an open file (possibly with
 * some or all of the magic bytes read - see comments above).
 */
DibImage* 
DibImage::LoadPng(Stream* stream)  /* We need to open the file */
{
   png_structp png_ptr;
   png_infop info_ptr;
   png_uint_32 width, height;
   DibImage* image = 0;

   /* Create and initialize the png_struct with the desired error handler
    * functions.  If you want to use the default stderr and longjump method,
    * you can supply NULL for the last three parameters.  We also supply the
    * the compiler header file version, so that we know if the application
    * was compiled with a compatible version of the library.  REQUIRED
    */
   png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
      NULL, NULL, NULL);

   if (png_ptr == NULL)
   {
      return 0;
   }

   /* Allocate/initialize the memory for image information.  REQUIRED. */
   info_ptr = png_create_info_struct(png_ptr);
   if (info_ptr == NULL)
   {
      png_destroy_read_struct(&png_ptr, png_infopp_NULL, png_infopp_NULL);
      return 0;
   }

   /* Set error handling if you are using the setjmp/longjmp method (this is
    * the normal method of doing things with libpng).  REQUIRED unless you
    * set up your own error handlers in the png_create_read_struct() earlier.
    */

   if (setjmp(png_jmpbuf(png_ptr)))
   {
      /* Free all of the memory associated with the png_ptr and info_ptr */
      png_destroy_read_struct(&png_ptr, &info_ptr, png_infopp_NULL);
      /* If we get here, we had a problem reading the file */
      return (ERROR);
   }

   /* Set up the input control if you are using standard C streams */
   png_set_read_fn( png_ptr, stream, PngRead );

   png_read_png(png_ptr, info_ptr, 
       PNG_TRANSFORM_STRIP_16 | PNG_TRANSFORM_PACKING | PNG_TRANSFORM_EXPAND |
       PNG_TRANSFORM_STRIP_ALPHA, NULL);

   png_bytep* row_pointers = png_get_rows(png_ptr, info_ptr);
   height = png_get_image_height(png_ptr, info_ptr);
   width = png_get_image_width(png_ptr, info_ptr);
   image = new DibImage();
   image->Create(width, height, 24, 0);

   for ( unsigned y = 0; y < height; y++ ) {
       for( unsigned x = 0; x < width; x++ ) {
           unsigned c = 0;
           unsigned char* ch = (unsigned char*)&c;
           unsigned char* array = row_pointers[y];
           ch[0] = array[x * 3 + 0];
           ch[1] = array[x * 3 + 1];
           ch[2] = array[x * 3 + 2];
           image->setPixel(x, y, c);
       }
   }

   /* clean up after the read, and free any memory allocated - REQUIRED */
   png_destroy_read_struct(&png_ptr, &info_ptr, png_infopp_NULL);

   /* that's it */
   return image;
}

static DibImage* 
LoadPng(const TCHAR *file_name)  /* We need to open the file */
{
   png_structp png_ptr;
   png_infop info_ptr;
   //unsigned int sig_read = 0;
   png_uint_32 width, height;
   FILE *fp;
   DibImage* image = 0;

   if ((fp = _tfopen(file_name, TEXT("rb"))) == NULL)
      return 0;

   /* Create and initialize the png_struct with the desired error handler
    * functions.  If you want to use the default stderr and longjump method,
    * you can supply NULL for the last three parameters.  We also supply the
    * the compiler header file version, so that we know if the application
    * was compiled with a compatible version of the library.  REQUIRED
    */
   png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
      NULL, NULL, NULL);

   if (png_ptr == NULL)
   {
      fclose(fp);
      return 0;
   }

   /* Allocate/initialize the memory for image information.  REQUIRED. */
   info_ptr = png_create_info_struct(png_ptr);
   if (info_ptr == NULL)
   {
      fclose(fp);
      png_destroy_read_struct(&png_ptr, png_infopp_NULL, png_infopp_NULL);
      return 0;
   }

   /* Set error handling if you are using the setjmp/longjmp method (this is
    * the normal method of doing things with libpng).  REQUIRED unless you
    * set up your own error handlers in the png_create_read_struct() earlier.
    */

   if (setjmp(png_jmpbuf(png_ptr)))
   {
      /* Free all of the memory associated with the png_ptr and info_ptr */
      png_destroy_read_struct(&png_ptr, &info_ptr, png_infopp_NULL);
      fclose(fp);
      /* If we get here, we had a problem reading the file */
      return (ERROR);
   }

   /* Set up the input control if you are using standard C streams */
   png_init_io(png_ptr, fp);

   /* If we have already read some of the signature */
   //png_set_sig_bytes(png_ptr, sig_read);

   png_read_png(png_ptr, info_ptr, 
       PNG_TRANSFORM_STRIP_16 | PNG_TRANSFORM_PACKING | PNG_TRANSFORM_EXPAND |
       PNG_TRANSFORM_STRIP_ALPHA, NULL);

   png_bytep* row_pointers = png_get_rows(png_ptr, info_ptr);
   height = png_get_image_height(png_ptr, info_ptr);
   width = png_get_image_width(png_ptr, info_ptr);
   image = new DibImage();
   image->Create(width, height, 24, 0);

   for ( unsigned y = 0; y < height; y++ ) {
       for( unsigned x = 0; x < width; x++ ) {
           unsigned c = 0;
           unsigned char* ch = (unsigned char*)&c;
           unsigned char* array = row_pointers[y];
           ch[0] = array[x * 3 + 0];
           ch[1] = array[x * 3 + 1];
           ch[2] = array[x * 3 + 2];
           image->setPixel(x, y, c);
       }
   }

   /* clean up after the read, and free any memory allocated - REQUIRED */
   png_destroy_read_struct(&png_ptr, &info_ptr, png_infopp_NULL);

   /* close the file */
   fclose(fp);

   /* that's it */
   return image;
}

/**
 * Save the DibImage to the given Stream object.
 */ 
bool
SavePng( Canvas* canvas, Stream* stream )
{
    png_structp png_ptr = 0;
    png_infop png_info = 0;
    png_bytep row_ptr = 0;
    bool error = true;
	int x,y;

    row_ptr = (png_bytep)malloc(4 * canvas->width());
    if ( row_ptr == 0 ) {
        goto cleanup;
    }

    png_ptr = png_create_write_struct
        (PNG_LIBPNG_VER_STRING, (png_voidp)NULL, NULL, NULL);

    if ( png_ptr == 0 ) {
        goto cleanup;
    }

    png_info = png_create_info_struct(png_ptr);
    if ( png_info == 0 ) {
        goto cleanup;
    }

    if (setjmp(png_jmpbuf(png_ptr))) {
        goto cleanup;
    }

    //png_init_io( png_ptr, stream );
    png_set_write_fn( png_ptr, stream, PngWrite, NULL );

    png_set_IHDR(png_ptr, png_info, canvas->width(), canvas->height(),
        8, PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
        PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

    png_write_info(png_ptr, png_info);

    for ( y = 0; y < canvas->height(); y++ ) {
        for( x = 0; x < canvas->width(); x++ ) {
            COLORREF colour = canvas->getPixel(x, y);
            row_ptr[x * 3 + 0] = GetRValue(colour);
            row_ptr[x * 3 + 1] = GetGValue(colour);
            row_ptr[x * 3 + 2] = GetBValue(colour);
        }
        png_write_row(png_ptr, row_ptr);
    }

    png_write_end(png_ptr, png_info);

    error = false;
cleanup:
    if ( png_ptr ) { 
        png_destroy_write_struct(&png_ptr, &png_info);
    }    

    free( row_ptr );
    return !error;
}

/**
 * Save the DibImage to the given file.
 */ 
bool
SavePng( Canvas* canvas, const TCHAR* filename )
{
    png_structp png_ptr = 0;
    png_infop png_info = 0;
    png_bytep row_ptr = 0;
    bool error = true;
	int x,y;

    FILE* file = _tfopen(filename, TEXT("wb"));
    if ( file == 0 ) {
        return false;
    }

    row_ptr = (png_bytep)malloc(4 * canvas->width());
    if ( row_ptr == 0 ) {
        goto cleanup;
    }

    png_ptr = png_create_write_struct
        (PNG_LIBPNG_VER_STRING, (png_voidp)NULL, NULL, NULL);

    if ( png_ptr == 0 ) {
        goto cleanup;
    }

    png_info = png_create_info_struct(png_ptr);
    if ( png_info == 0 ) {
        goto cleanup;
    }

    if (setjmp(png_jmpbuf(png_ptr))) {
        goto cleanup;
    }

    png_init_io( png_ptr, file );

    png_set_IHDR(png_ptr, png_info, canvas->width(), canvas->height(),
        8, PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
        PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

    png_write_info(png_ptr, png_info);

    for ( y = 0; y < canvas->height(); y++ ) {
        for( x = 0; x < canvas->width(); x++ ) {
            COLORREF colour = canvas->getPixel(x, y);
            row_ptr[x * 3 + 0] = GetRValue(colour);
            row_ptr[x * 3 + 1] = GetGValue(colour);
            row_ptr[x * 3 + 2] = GetBValue(colour);
        }
        png_write_row(png_ptr, row_ptr);
    }

    png_write_end(png_ptr, png_info);

    error = false;
cleanup:
    fclose(file);
    if ( error ) {
        DeleteFile(filename);
    }

    if ( png_ptr ) { 
        png_destroy_write_struct(&png_ptr, &png_info);
    }    

    free( row_ptr );
    return !error;
}
#else
static DibImage* 
LoadPng(const TCHAR *file_name) {
    assert(0);
    return 0;
}

bool
SavePng( Canvas* canvas, const TCHAR* filename )
{
    assert(0);
    return false;
}

#endif // USE_PNG

#ifdef USE_JPEG

struct my_error_mgr {
    struct jpeg_error_mgr pub;	/* "public" fields */

    jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
    
  //lint -e{826}  
  my_error_ptr myerr = (my_error_ptr) cinfo->err;
  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  //(*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}

static DibImage* 
LoadJpeg( const TCHAR* fileName )
{
    DibImage* dib = 0;
    // cause we didn't redefine INT32 in jmorecfg.h
    assert( sizeof(int) == sizeof( long ));

    /* This struct contains the JPEG decompression parameters and pointers to
     * working space (which is allocated as needed by the JPEG library).
     */
    struct jpeg_decompress_struct cinfo;
    /* We use our private extension JPEG error handler.
     * Note that this struct must live as long as the main JPEG parameter
     * struct, to avoid dangling-pointer problems.
     */
    struct my_error_mgr jerr;
    /* More stuff */
    FILE * infile;		/* source file */
    unsigned char* buffer = 0;		/* Output row buffer */
    int row_stride;		/* physical row width in output buffer */

    /* In this example we want to open the input file before doing anything else,
     * so that the setjmp() error recovery below can assume the file is open.
     * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
     * requires it in order to read binary files.
     */

    if ((infile = _tfopen(fileName, TEXT("rb"))) == NULL) {
        //dbgprint(( DDIB, "Cannot open jpeg file." ));
        return 0;
    }

    /* Step 1: allocate and initialize JPEG decompression object */

    /* We set up the normal JPEG error routines, then override error_exit. */
    cinfo.err = jpeg_std_error(&jerr.pub);
    jerr.pub.error_exit = my_error_exit;
    /* Establish the setjmp return context for my_error_exit to use. */
    if (setjmp(jerr.setjmp_buffer)) {
        /* If we get here, the JPEG code has signaled an error.
         * We need to clean up the JPEG object, close the input file, and return.
         */
        jpeg_destroy_decompress(&cinfo);
        free( buffer );
        fclose(infile);
        delete dib;
        return 0;
    }
    /* Now we can initialize the JPEG decompression object. */
    jpeg_create_decompress(&cinfo);

    /* Step 2: specify data source (eg, a file) */

    jpeg_stdio_src(&cinfo, infile);

    /* Step 3: read file parameters with jpeg_read_header() */

    (void) jpeg_read_header(&cinfo, TRUE);
    /* We can ignore the return value from jpeg_read_header since
     *   (a) suspension is not possible with the stdio data source, and
     *   (b) we passed TRUE to reject a tables-only JPEG file as an error.
     * See libjpeg.doc for more info.
     */

    /* Step 4: set parameters for decompression */

    /* In this example, we don't need to change any of the defaults set by
     * jpeg_read_header(), so we do nothing here.
     */

    /* Step 5: Start decompressor */

    (void) jpeg_start_decompress(&cinfo);
    /* We can ignore the return value since suspension is not possible
     * with the stdio data source.
     */

    /* We may need to do some setup of our own at this point before reading
     * the data.  After jpeg_start_decompress() we have the correct scaled
     * output image dimensions available, as well as the output colormap
     * if we asked for color quantization.
     * In this example, we need to make an output work buffer of the right size.
     */ 
    /* JSAMPLEs per row in output buffer */
    row_stride = cinfo.output_width * cinfo.output_components;
    assert( cinfo.output_components == 3 || cinfo.output_components == 1 );

    /* Make a one-row-high sample array that will go away when done with image */
    buffer = (unsigned char*)malloc( row_stride );

    /* Step 6: while (scan lines remain to be read) */
    /*           jpeg_read_scanlines(...); */
    dib = new DibImage();
    dib->Create(cinfo.image_width, cinfo.image_height, 24, 0);

    /* Here we use the library's state variable cinfo.output_scanline as the
     * loop counter, so that we don't have to keep track ourselves.
     */
    while (cinfo.output_scanline < cinfo.output_height) {
        /* jpeg_read_scanlines expects an array of pointers to scanlines.
         * Here the array is only one element long, but you could ask for
         * more than one scanline at a time if that's more convenient.
         */

        (void) jpeg_read_scanlines(&cinfo, &buffer, 1);
        if ( cinfo.output_components == 3 ) {
            for( unsigned i = 0; i < cinfo.image_width; i++ ) {
               unsigned c = 0;
               unsigned char* ch = (unsigned char*)&c;
               ch[0] = buffer[i * 3 + 0];
               ch[1] = buffer[i * 3 + 1];
               ch[2] = buffer[i * 3 + 2];
               dib->setPixel(i, cinfo.output_scanline-1, c);
            }
        } else if ( cinfo.output_components == 1 ) {
            for( unsigned i = 0; i < cinfo.image_width; i++ ) {
               unsigned c = 0;
               unsigned char* ch = (unsigned char*)&c;
               ch[0] = ch[1] = ch[2] = buffer[i];
               dib->setPixel(i, cinfo.output_scanline-1, c);
            }
        }
    }

    free( buffer );

    /* Step 7: Finish decompression */

    (void) jpeg_finish_decompress(&cinfo);
    /* We can ignore the return value since suspension is not possible
     * with the stdio data source.
     */

    /* Step 8: Release JPEG decompression object */

    /* This is an important step since it will release a good deal of memory. */
    jpeg_destroy_decompress(&cinfo);

    /* After finish_decompress, we can close the input file.
     * Here we postpone it until after no more JPEG errors are possible,
     * so as to simplify the setjmp error logic above.  (Actually, I don't
     * think that jpeg_destroy can do an error exit, but why assume anything...)
     */
    fclose(infile);

    /* At this point you may want to check to see whether any corrupt-data
     * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
     */

    /* And we're done! */
    return dib;
}

bool
SaveJpeg( Canvas* canvas, const TCHAR* fileName )
{
	struct jpeg_compress_struct cinfo;
	struct my_error_mgr jerr;
    FILE* outfile;
    int x,y;
	JSAMPROW row_ptr[1];	/* pointer to a single row */
    row_ptr[0] = (JSAMPROW)malloc(3 * canvas->width());

    /* Step 1: allocate and initialize JPEG decompression object */

    /* We set up the normal JPEG error routines, then override error_exit. */

    if ((outfile = _tfopen(fileName, TEXT("wb"))) == NULL) {
        //dbgprint(( DDIB, "Cannot open jpeg file." ));
        return false;
    }

	cinfo.err = jpeg_std_error(&jerr.pub);
    jerr.pub.error_exit = my_error_exit;
	jpeg_create_compress(&cinfo);

    /* Establish the setjmp return context for my_error_exit to use. */
    if (setjmp(jerr.setjmp_buffer)) {
        /* If we get here, the JPEG code has signaled an error.
         * We need to clean up the JPEG object, close the input file, and return.
         */
        jpeg_destroy_compress(&cinfo);
        fclose(outfile);
		free( row_ptr[0] );
        return 0;
    }

	cinfo.image_width = canvas->width(); 	/* image width and height, in pixels */
	cinfo.image_height = canvas->height();
	cinfo.input_components = 3;	/* # of color components per pixel */
	cinfo.in_color_space = JCS_RGB; /* colorspace of input image */

	jpeg_set_defaults(&cinfo);
	/* Make optional parameter settings here */

    jpeg_stdio_dest(&cinfo, outfile);

	jpeg_start_compress(&cinfo, TRUE);


    for ( y = 0; y < canvas->height(); y++ ) {
        for( x = 0; x < canvas->width(); x++ ) {
            COLORREF colour = canvas->getPixel(x, y);
            row_ptr[0][x * 3 + 0] = GetRValue(colour);
            row_ptr[0][x * 3 + 1] = GetGValue(colour);
            row_ptr[0][x * 3 + 2] = GetBValue(colour);
        }
	    jpeg_write_scanlines(&cinfo, row_ptr, 1);
    }

	jpeg_finish_compress(&cinfo);
	jpeg_destroy_compress(&cinfo);
    fclose( outfile );
	free(row_ptr[0]);
    return true;

}

#else 
static DibImage* LoadJpeg( const char* fileName )
{
    assert(0);
    return 0;
}

#endif // USE_JPEG

