#include <assert.h>
#include "FlexList.h"
#include "ListItem.h"
#include <stdio.h>
#include <windowsx.h>
#include <tchar.h>
#include "debug.h" // Must be last

FlexListNotify::FlexListNotify()
{

}

FlexListNotify::~FlexListNotify()
{

}

/**
 * Called when an item is double clicked.
 */ 
void
FlexListNotify::onFlexDoubleClick(int index)
{

}

/** 
 * Called when the selection changes.
 */ 
void
FlexListNotify::onFlexSelChanged( int index )
{

}

/**
 * Called when an item is clicked.
 */ 
void
FlexListNotify::onFlexClicked()
{

}

/** Construct a FlexList control.
 * @param hInstance Module instance, passed to WinMain().
 */ 
FlexList::FlexList( HINSTANCE hInstance )
{
	_hInstance = hInstance;
	_pszClassName = _T("FlexList");
	_pszTitle = _T("");

    //_dwStyle |= WS_OVERLAPPED | WS_THICKFRAME | WS_SYSMENU | WS_CLIPCHILDREN; 
    _dwStyle |= WS_VISIBLE | WS_CHILD | WS_VSCROLL;
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
	//_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
	_WndClass.style |= CS_DBLCLKS;
    _dwExtendedStyle |= 0;
    _totalHeight = 0;
    _items_used = 0;
    _items_alloc = 1;
    _items = (ListItem**)malloc(sizeof(ListItem*));
    _curPos = 0;
    _offset = 0;
    _height = 0;
    _width = 0;
    _redraw = true;
    _focused = -1;
    _topItem = 0;
    _backColour = RGB(255,255,255);
    _notify = 0;

    _averageItemHeight = 14;
}

FlexList::~FlexList()
{
    clear();
    free( _items );
}

/**
 * Sets the notification object.
 */ 
void
FlexList::setNotify( FlexListNotify* notify )
{
    _notify = notify;
}

/** 
 * Sets the background colour of the list.
 */ 
void
FlexList::setBackColour(COLORREF clr)
{
    _backColour = clr;
}

/**
 * Clears the list.
 */ 
void
FlexList::clear()
{
    int i;
    for ( i = 0; i < _items_used; i++ ) {
        delete _items[i];
    }

    free( _items );

    _totalHeight = 0;
    _curPos = 0;
    _offset = 0;
    _items_used = 0;
    _items_alloc = 1;
    _items = (ListItem**)malloc(_items_alloc * sizeof(ListItem*));
    setFocused(-1);
}

/**
 * Sets which item has the focus.
 */ 
void
FlexList::setFocused(int focused)
{
    int oldFocused = _focused;
    if ( _focused >= 0 && _focused < _items_used ) {
        invalidate(_focused);
        _items[_focused]->select( false );
    }
	
	if ( focused < 0 || focused >= _items_used ) {
        _focused = -1;
        goto notify;
    }

    _focused = focused;
    _items[_focused]->select(true);
    invalidate(_focused);

notify:    
    if ( _focused != oldFocused ) {
        if ( _notify ) {
            _notify->onFlexSelChanged( _focused );
        }
    }
}

/**
 * Returns the index of the item which has the focus. -1 if none.
 */ 
int
FlexList::getFocused()
{
    return _focused;
}

/**
 * Creates the FlexList control.
 */ 
HWND 
FlexList::create( int x, int y, int cx, int cy, HWND hParent )
{
    return Window::Create( x, y, cx, cy, hParent, 0, _hInstance );
}

void 
FlexList::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC dc;
    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        dc = PaintStruct.hdc;
    } else {
        dc = (HDC)wParam;
    }

    // Paint the background.
    RECT rect = getClientRect();
    Brush brush( _backColour );
    FillRect( dc, &rect, brush );

    if ( _items_used > 0 ) {
        paintItems(dc);
    }
    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}

void
FlexList::paintItems(HDC dc )
{
    int i;
    SetTextColor( dc, RGB(0,0,0));

    // find the item that corresponds to where the scroll bar is.
    int index = 0;
    for ( i = 0; i < _items_used; i++ ) {
        if ( _items[i]->pos > _curPos ) {
            break;
        }
    }

    if ( i > 0 ) {
        index = i - 1;
    }

    // calculate where the top of the item would be above the window.
    int offset = _items[index]->pos - _curPos;
    _offset = offset;

    // draw the item.
    for ( i = index; i < _items_used; i++ ) {
        _items[i]->draw( dc, 0, offset );
        offset += _items[i]->height();        

        // keep drawing items until we get to the end of the list or the bottom of the window.
        if ( offset > _height ) {
            break;
        }
    }
}


void
FlexList::extend()
{
    if ( _items_used == _items_alloc ) {
        _items_alloc <<= 1;
        _items = (ListItem**)realloc( _items,
                _items_alloc * sizeof(ListItem*) );
    }
}

/**
 * Scrolls to the bottom of the flexlist control.
 */ 
void
FlexList::scrollToBottom()
{
    int newPos = _totalHeight - _height - 1;

    if ( newPos != _curPos ) {
        _curPos = _totalHeight - _height - 1;
        scrollTo( _curPos );
    }
}

/**
 * Inserts a new item into the FlexList control, at the given position.
 */ 
void
FlexList::insert( ListItem* item, int index )
{
    int i;
    if ( index > _items_used ) {
        index = _items_used;
    } else if ( index < 0 ) {
        index = 0;
    }

    extend();

    for ( i = _items_used; i > index; i-- ) {
        _items[i] = _items[i - 1];
    }
    _items_used++;

    _items[index] = item;

	reformat(index);

    assert(hwnd);
    InvalidateRect( hwnd, NULL, TRUE );
}

/**
 * Appends an item to the end of the FlexList control.
 */ 
void
FlexList::add( ListItem* item )
{
    extend();
    _items[_items_used++] = item;
    reformat(_items_used-1);
    assert(hwnd);
    InvalidateRect( hwnd, NULL, TRUE );
    if ( _focused == -1 ) {
        setFocused( _items_used-1);
    }
}

LRESULT 
FlexList::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
		HANDLE_MSG(hwnd, WM_LBUTTONDBLCLK, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        HANDLE_MSG(hwnd, WM_VSCROLL, onVScroll);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            if ( _redraw ) {
                onPaint(hwnd, wParam, lParam, false);
            }
            return 0;
        case WM_SETREDRAW:
            if ( wParam ) {
                _redraw = true;
                reformat(0);
                assert(hwnd);
                InvalidateRect( hwnd, NULL, FALSE );
            } else {
                _redraw = false;
            }
            return 0;

        case WM_MOUSEWHEEL: {
            int zDelta = GET_WHEEL_DELTA_WPARAM(wParam);
            double points = -(double)zDelta / 120 * 
                _averageItemHeight * 2;

            _curPos += (unsigned)points;
            if ( _curPos > _totalHeight - _height - 1 ) {
                _curPos = _totalHeight - _height - 1;
            } else if ( _curPos < 0 ) {
                _curPos = 0;
            }

            scrollTo( _curPos );
            return 0;
        }
    }
    
    *pbProcessed = FALSE;
    return 0;
}

void
FlexList::onVScroll( HWND hwnd, HWND hwndCtl, UINT code, int pos )
{
    if ( code == SB_THUMBTRACK || code == SB_THUMBPOSITION ) {
        SCROLLINFO info;
        memset( &info, 0, sizeof(info ));
        info.cbSize = sizeof( info );
        info.fMask = SIF_TRACKPOS;
        GetScrollInfo( hwnd, SB_VERT, &info );
        pos = info.nTrackPos;
    }
    
    switch( code ) {
        case SB_BOTTOM:
            _curPos = _totalHeight - _height - 1;
            break;
        case SB_LINEDOWN:
            _curPos = _curPos + 14;
            break;
        case SB_LINEUP:
             _curPos = _curPos - 14;
            break;
        case SB_PAGEDOWN:
            _curPos += _height;
            break;
        case SB_PAGEUP:
            _curPos -= _height;
            break;
        case SB_THUMBPOSITION:
        case SB_THUMBTRACK:
            _curPos = pos;
            break;
        case SB_TOP:
            _curPos = 0;
            break;
    }
            
    if ( _curPos > _totalHeight - _height - 1 ) {
        _curPos = _totalHeight - _height - 1;
    } else if ( _curPos < 0 ) {
        _curPos = 0;
    }

    scrollTo( _curPos );
}

void 
FlexList::scrollTo( int position )
{
    _curPos = position;
    SetScrollPos( hwnd, SB_VERT, position, TRUE );
    assert(hwnd);
    InvalidateRect( hwnd, NULL, TRUE );
}


void
FlexList::onClose(HWND)
{

}

void 
FlexList::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{

}

BOOL 
FlexList::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
#if 0
    Font* font = new Font("Arial", 25);
    for( int i = 0; i < 10; i++ ) {
        char buffer[100];
        sprintf(buffer, "Line item # %d The quick brown fox jumped over the laxy dog.", 
                i);
        LineItem* item = new LineItem;
        item->setFont(font);
        item->setText( buffer );
        add( item );
    }
#endif

    _width = lpCreateStruct->cx;
    _height = lpCreateStruct->cy;

    return TRUE;
}

/**
 * Returns the number of items in the FlexList.
 */ 
int
FlexList::getCount()
{
    return _items_used;
}

/**
 * Returns the item index below the given point on the window.
 * Returns -1 if the point is not above any item.
 */ 
int
FlexList::itemFromPoint(int x, int y)
{
    // find the item that corresponds to where the scroll bar is.
    int index = 0;
    int i;

    for ( i = 0; i < _items_used; i++ ) {
        if ( _items[i]->pos > _curPos ) {
            break;
        }
    }

    if ( i > 0 ) {
        index = i - 1;
    }

	if ( index >= _items_used ) {
		return -1;
	}

    // calculate where the top of the item would be above the window.
    int offset = _items[index]->pos - _curPos;

    for ( i = index; i < _items_used; i++ ) {
        offset += _items[i]->height();        
        if ( y < offset ) {
            return i;
        }

        if ( offset > _height ) {
            break;
        }
    }

    return -1;
}

/**
 * Sets the selected item.
 */ 
void
FlexList::setSelected(int sel)
{
    if ( sel >= _items_used || sel < 0 ) {
        //assert(false);
        return;
    }

    _items[sel]->select(true);
}

/**
 * Returns the index of the currently selected item.
 */ 
int
FlexList::getSelected( int sel )
{
    sel++;
    if ( sel < 0 || sel >= _items_used ) {
        return -1;
    }

    for( int i = 0; i < _items_used; i++ ) {
        if ( _items[i]->isSelected() ) {
            return i;
        }
    }

    return -1;
}

void
FlexList::onDestroy(HWND hwnd)
{
}


void 
FlexList::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{
    if ( _focused < 0 ) {
        return;
    }

    if ( vk == VK_DOWN && _focused + 1 < _items_used ) {
        setFocused( _focused + 1 );
    } else if ( vk == VK_UP && _focused > 0 ) {
        setFocused( _focused - 1 );
    } else if ( vk == VK_NEXT ) {
        int left = _height;
        int i;
        for ( i = _focused; i < _items_used - 1 && left > 0; i++ ) {
            left -= _items[i]->height();
        }
        setFocused( i );

    } else if ( vk == VK_PRIOR ) {
        int left = _height;
        int i;
        for ( i = _focused; i > 0 && left > 0; i-- ) {
            left -= _items[i]->height();
        }
        setFocused( i );

    } else if ( vk == VK_HOME ) {
        setFocused( 0 );
    } else if ( vk == VK_END ) {
        setFocused( _items_used - 1 );
    } else {
        return;
    }

    ensureVisible( _focused );
}

void 
FlexList::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{

    if ( _notify ) {
        _notify->onFlexClicked();
    }

    int item = itemFromPoint( x, y );
    if ( item == -1 ) {
        return;
    }

    setFocused(item);
    if ( fDoubleClick ) {
        if ( _notify ) {
            _notify->onFlexDoubleClick( item );
        }
    }
}

void 
FlexList::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
FlexList::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( keyFlags & MK_LBUTTON ) {
        onLButtonDown(hwnd, false, x, y, keyFlags);
    }
}

/**
 * Ensures that the given item is visible, scrolling if necessary.
 */ 
void
FlexList::ensureVisible( int index )
{
    if ( index < 0 || index >= _items_used ) {
        return;
    }

    if ( _items[index]->pos < _curPos ) {
        scrollTo(_items[index]->pos);
    } else if ( _items[index]->pos + _items[index]->height() > _curPos +
            _height ) 
    {
        scrollTo( _items[index]->pos - (_height - _items[index]->height()) );
    }
}

void
FlexList::reformat(int index)
{
    if ( !_redraw ) {
        return;
    }
    _totalHeight = 0;

    // for each line item, format it and add it to the total size.
    for( int i = 0; i < _items_used; i++ ) {
        _items[i]->pos = _totalHeight;
        _totalHeight += _items[i]->format(_width);
    }

    _averageItemHeight = (double)_totalHeight / _items_used;

    // set the scroll bar height based on visible area.
    SCROLLINFO info;
    memset( &info, 0, sizeof(info));
    info.cbSize = sizeof(info);
    info.fMask = SIF_RANGE | SIF_PAGE | SIF_DISABLENOSCROLL;
    info.nMin = 0;
    info.nMax = _totalHeight + 1;
    info.nPage = _height;

    if ( info.nMax < 0 ) {
        info.nMax = 0;
    }
    SetScrollInfo( hwnd, SB_VERT, &info, true );
}

void 
FlexList::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    _height = cy;
    _width = cx;
    reformat(0);
}

void
FlexList::invalidate( int index )
{
    if ( index >= _items_used || index < 0 ) {
        return;
    }

    invalidate( _items[index] );
}

void
FlexList::invalidate( ListItem* item )
{
	if ( !_redraw ) {
		return;
	}

    int top = _curPos;
    int bottom = _curPos + _height;

    if ( item->pos >= top || item->pos < bottom ||
         item->pos + item->height() >= top ||
            item->pos + item->height() < bottom ) 
    {
        RECT rect;
        SetRect( &rect, 0, item->pos - _curPos, _width, 
                item->pos + item->height() - _curPos );
        assert(hwnd);
        InvalidateRect( hwnd, &rect, FALSE );
    }
}

/**
 * Returns true if the item with the given index is currently visible.
 */ 
bool
FlexList::isItemVisible(int item)
{
    int top = _curPos;
    int bottom = _curPos + _height;

    assert( item < _items_used );
    assert( item > 0 );

    if ( _items[item]->pos >= top || _items[item]->pos < bottom ) {
        return true;
    }

    if ( _items[item]->pos + _items[item]->height() >= top ||
            _items[item]->pos + _items[item]->height() < bottom ) {
        return true;
    }

    return false;
}
