#include <assert.h>
#include "LineItem.h"
#include "FlexListView.h"
#include "debug.h"

const int ImageX = 16;
const int ImageY = 16;

LineItem::LineItem(FlexListViewInfo* info)
{
    _text = 0; 
    _width = 0;
    _height = 0;
    _info = info;
    _formatted = false;
    _image = -1;
}


LineItem::~LineItem()
{
    free( _text );
}

void
LineItem::setImage(int index )
{
    _image = index;
}

void 
LineItem::setText(const TCHAR* text)
{
    free( _text );
    _text = _tcsdup( text );
    _formatted = false;
}

int 
LineItem::width()
{
    assert( _formatted );
    return _width;
}

int 
LineItem::height()
{
    assert( _formatted );
    return _height;
}

int 
LineItem::format( int width )
{
    _height = _info->font->getHeight() + _info->font->getExternalLeading();
    assert( _height > 0 );
    if ( _height < ImageY ) {
        _height = ImageY;
    }

    _width = width;

    _formatted = true;

    return _height;
}

void
LineItem::draw( HDC hdc, int x, int y )
{
    if ( !_formatted ) {
        format( _width );
    }

    DeviceContext dc(hdc);
    SetBkMode( dc, TRANSPARENT );
    dc.selectFont( *_info->font);
    SetTextColor( dc, _selected ? _info->SelTextColour : _info->TextColour );

    RECT rect;
    SetRect( &rect, x + ImageX, y, x + _width, y + _height );

    if ( _image != -1 ) {
        // Leave at least one pixel between image and text.
        rect.left++;
    }

    if ( _selected ) {
        Brush brush(_info->SelBackColour);
        FillRect( hdc, &rect, brush );
    } else {
        Brush brush(_info->BackColour);
        FillRect( hdc, &rect, brush );
    }
    
    if ( 0 == DrawText( dc, _text, _tcslen( _text ), &rect,
           DT_VCENTER | DT_SINGLELINE | DT_WORD_ELLIPSIS) ) {
    }

    if ( _info->hImageList && _image != -1 ) {
        // centre vertically.
        // horizontally we already have exactly enough space.
        int iy = y + _height / 2 - ImageY / 2;
        
        ImageList_Draw( _info->hImageList,
                _image,
                hdc, 
                x,
                iy,
                ILD_NORMAL
                );
    }
}

