#include <assert.h>
#include <windowsx.h>
#include "ScrollWindow.h"

ScrollWindow::ScrollWindow(HINSTANCE hInstance)
{
	_hInstance = hInstance;
	_pszClassName = TEXT("ScrollWindow");
	_pszTitle = TEXT("Scroll Window");

    _dwStyle |= WS_HSCROLL | WS_VSCROLL; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = NULL;
	//_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
    _posX = 0;
    _posY = 0;
}

void 
ScrollWindow::format()
{
    RECT rect = getClientRect();
    // set the scroll bar height based on visible area.
    SCROLLINFO info;
    memset( &info, 0, sizeof(info));
    info.cbSize = sizeof(info);
    info.fMask = SIF_RANGE | SIF_PAGE | SIF_DISABLENOSCROLL;
    info.nMin = 0;
    info.nMax = _height;
    info.nPage = rect.bottom;

    _pageX = rect.right;
    _pageY = rect.bottom;

    SetScrollInfo( hwnd, SB_VERT, &info, true );

    info.nMin = 0;
    info.nMax = _width;
    info.nPage = rect.right;
    SetScrollInfo( hwnd, SB_HORZ, &info, true );

    bool change = false; 
    if ( rect.bottom >= _height ) {
        _posY = 0;
        change = true;
    }

    if ( rect.right >= _width ) {
        _posX = 0;
        change = true;
    }

    scrollTo( _posX, _posY );
}

/**
 * Returns the X scroll position.
 */
int 
ScrollWindow::getScrollX() {
    return _posX;
}

/**
 * Returns the Y scroll position.
 */
int 
ScrollWindow::getScrollY() {
    return _posY;
}


/**
 * Sets the scrolling information.
 * @param pixelsPerUnitX The number of pixels that one unit of scrolling
 * represents. Usually 1.
 * @param pixelsPerUnitY
 * @param noUnitsX The number of units that the whole scrollbar represents.
 * (eg. The size of the image minus the size of the viewport)
 */
void 
ScrollWindow::setScrollBars( int pixelsPerUnitX, int pixelsPerUnitY, int noUnitsX, int noUnitsY)
{
    _pixelsPerUnitX = pixelsPerUnitX;
    _pixelsPerUnitY = pixelsPerUnitY;
    _noUnitsX = noUnitsX;
    _noUnitsY = noUnitsY;
    _width = _noUnitsX * _pixelsPerUnitX;
    _height = _noUnitsY * _pixelsPerUnitY;
    format();
}


ScrollWindow::~ScrollWindow()
{

}

void
ScrollWindow::create()
{
    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                CW_USEDEFAULT, CW_USEDEFAULT, 0, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

LRESULT 
ScrollWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        HANDLE_MSG(hwnd, WM_VSCROLL, onVScroll);
        HANDLE_MSG(hwnd, WM_HSCROLL, onHScroll);
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}

/**
 * Override this function to draw using the offseted coordinates.
 @param dc The device context on which to draw. It is already initialized, no
 need to call BeginPaint().
 */
void 
ScrollWindow::onDraw( HDC dc )
{
    TextOut( dc, 100, 100, _T("BARK!"), 5);

}

void
ScrollWindow::onClose(HWND)
{

}

void 
ScrollWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{

}

BOOL 
ScrollWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{

    return TRUE;
}

void
ScrollWindow::onDestroy(HWND hwnd)
{

}


void 
ScrollWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
ScrollWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{

}

void
ScrollWindow::onScroll(int x, int y)
{

}

void 
ScrollWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
ScrollWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{

}


void 
ScrollWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, 
        bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    // Translate the dc.

    OffsetWindowOrgEx(hdc, _posX, _posY, NULL);
    SetBrushOrgEx(hdc, -_posX, -_posY, NULL);
    onDraw( hdc );

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
ScrollWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    format();
}

void
ScrollWindow::onHScroll( HWND hwnd, HWND hwndCtl, UINT code, int pos )
{
    int posX = _posX;

    if ( code == SB_THUMBTRACK || code == SB_THUMBPOSITION ) {
        SCROLLINFO info;
        memset( &info, 0, sizeof(info ));
        info.cbSize = sizeof( info );
        info.fMask = SIF_TRACKPOS;
        GetScrollInfo( hwnd, SB_HORZ, &info );
        pos = info.nTrackPos;
    }
    
    switch( code ) {
        case SB_BOTTOM:
            posX = _width - _pageX;
            break;
        case SB_LINEDOWN:
            posX = _posX + _pixelsPerUnitX;
            break;
        case SB_LINEUP:
             posX = _posX - _pixelsPerUnitX;
            break;
        case SB_PAGEDOWN:
            posX = _posX + _pageX;
            break;
        case SB_PAGEUP:
            posX = _posX - _pageX;
            break;
        case SB_THUMBPOSITION:
        case SB_THUMBTRACK:
            posX = pos;
            break;
        case SB_TOP:
            posX = 0;
            break;
    }
            
    if ( posX > _width - _pageX - 1 ) {
        posX = _width - _pageX - 1;
    } else if ( posX < 0 ) {
        posX = 0;
    }

    scrollTo( posX, _posY );

}

void
ScrollWindow::onVScroll( HWND hwnd, HWND hwndCtl, UINT code, int pos )
{
    int posY = _posY;

    if ( code == SB_THUMBTRACK || code == SB_THUMBPOSITION ) {
        SCROLLINFO info;
        memset( &info, 0, sizeof(info ));
        info.cbSize = sizeof( info );
        info.fMask = SIF_TRACKPOS;
        GetScrollInfo( hwnd, SB_VERT, &info );
        pos = info.nTrackPos;
    }
    
    switch( code ) {
        case SB_BOTTOM:
            posY = _height - _pageY;
            break;
        case SB_LINEDOWN:
            posY += _pixelsPerUnitY;
            break;
        case SB_LINEUP:
            posY -= _pixelsPerUnitY;
            break;
        case SB_PAGEDOWN:
            posY += _pageY;
            break;
        case SB_PAGEUP:
            posY -= _pageY;
            break;
        case SB_THUMBPOSITION:
        case SB_THUMBTRACK:
            posY = pos;
            break;
        case SB_TOP:
            posY = 0;
            break;
    }
            
    if ( posY > _height - _pageY - 1 ) {
        posY = _height - _pageY - 1;
    } else if ( posY < 0 ) {
        posY = 0;
    }

    scrollTo( _posX, posY );
}

/**
 * Immediately scroll to the given position.
 */
void 
ScrollWindow::scrollTo( int x, int y )
{
    int ydiff = y - _posY;
    int xdiff = x - _posX;
    _posX = x;
    _posY = y;
    SetScrollPos( hwnd, SB_VERT, _posY, TRUE );
    SetScrollPos( hwnd, SB_HORZ, _posX, TRUE );
    RECT rect;
    ::ScrollWindowEx( hwnd, -xdiff, -ydiff, NULL, NULL, NULL, &rect, SW_INVALIDATE );
    InvalidateRect( hwnd, &rect, FALSE );
    onScroll( x, y );
}
