#include <assert.h>
#include <windowsx.h>
#include "SplitWindow.h"

/**
 * Constructs a Splitter.
 *
 * @param horizontal If true, the splitter will split horizontally. If false,
 * the splitter will split veritcally and have a left and right side.
 */
Splitter::Splitter( bool horizontal )
{
    this->horizontal = horizontal;
    hFirst = hSecond = 0;
    first = second = 0;
    thickness = 4;
    pos = -1;
    SetRect( &rect, 0, 0, 0, 0 );
    _dragging = 0;
    _over = false;
    _immobile = false;

    int i;
    for ( i = 0; i < 2; i++ ) {
        _info[i].minWidth = _info[i].minHeight = _info[i].maxWidth = 
            _info[i].maxHeight = -1;
        _setHeight[0] = -1;
        _setWidth[0] = -1;
    }
}

Splitter::~Splitter()
{
    delete first;
    delete second;
}

void
Splitter::setThickness( int pixels )
{
    thickness = pixels;
}

/**
 * Sets the window handle for one of the sides of the splitter. It will be
 * automatically resized.
 */
void 
Splitter::setPane( bool first, HWND hChild )
{
    if ( first ) {
        hFirst = hChild;
    } else {
        hSecond = hChild;
    }
}

/**
 * Sets another splitter to be the contents of one of the panes.
 */
void 
Splitter::setPane( bool first, Splitter* splitter )
{
    if ( first ) {
        delete this->first;
        this->first = splitter;
    } else {
        delete this->second;
        this->second = splitter;
    }
}

/**
 * Sets the size information for one of the panes of this splitter, or of any
 * of its child splitters. All child splitters will be searched for the given
 * window.
 *
 * @param hwnd The contents of the window to be set.
 *
 * @param minWidth The minimum width. Use -1 if there is no minimum.
 *
 * @param minHeight The minimum height. Use -1 if there is no minimum.
 *
 * @param maxWidth The maximum width.
 *
 * @param maxHeight The maximum height.
 */
bool
Splitter::setSplitInfo( HWND hwnd, int minWidth, int minHeight, int maxWidth, int maxHeight )
{
    int i = -1;
    if ( hFirst == hwnd ) {
        i = 0;
    } else if ( hSecond == hwnd ) {
        i = 1;
    }

    if ( i >= 0 ) {
        _info[i].minWidth = minWidth;
        _info[i].minHeight = minHeight;
        _info[i].maxWidth = maxWidth;
        _info[i].maxHeight = maxHeight;
        if ( horizontal && minHeight > -1 && minHeight == maxHeight || 
                !horizontal && minWidth > -1 && minWidth == maxWidth ) 
        {
            _immobile = true;
        } else {
            _immobile = false;
        }
        return true;
    } else {
        if ( first && first->setSplitInfo( hwnd, minWidth, minHeight,
                    maxWidth, maxHeight ) ) {
            first->getSplitInfo( &_info[0] );
            return true;
        } else if ( second && second->setSplitInfo( hwnd, minWidth,
                    minHeight, maxWidth, maxHeight ) ) {
            second->getSplitInfo( &_info[1] );
            return true;
        }
    }

    return false;
}

void 
Splitter::merge( Splitter::SplitInfo* info1, Splitter::SplitInfo* info2 )
{
    if ( info2->minWidth > info1->minWidth ) {
        info1->minWidth = info2->minWidth;
    }
    if ( info2->minHeight > info1->minHeight ) {
        info1->minHeight = info2->minHeight;
    }
    if ( info2->maxWidth > info1->maxWidth ) {
        info1->maxWidth = info2->maxWidth;
    }
    if ( info2->maxHeight > info1->maxHeight ) {
        info1->maxHeight = info2->maxHeight;
    }
}

void
Splitter::getSplitInfo( Splitter::SplitInfo* info )
{
    SplitInfo fInfo = {-1, -1, -1, -1 };
    SplitInfo sInfo = {-1, -1, -1, -1 };

    if ( hFirst ) {
        fInfo = _info[0];
    } else if ( first ) {
        first->getSplitInfo( &fInfo );
    }

    if ( hSecond ) {
        sInfo = _info[1];
    } else if ( second ) {
        second->getSplitInfo( &sInfo );
    }

    merge( &fInfo, &sInfo );
    *info = fInfo;
}

void
Splitter::format( RECT* newRect )
{
    double oldLeft, oldRight, newLeft, newRight;

    if ( horizontal ) {
        oldLeft = rect.top;
        oldRight = rect.bottom;
        newLeft = newRect->top;
        newRight = newRect->bottom;
    } else {
        oldLeft = rect.left;
        oldRight = rect.right;
        newLeft = newRect->left;
        newRight = newRect->right;
    }
    
    if ( pos > -1 ) {
        double oldWidth = oldRight - oldLeft;
        double newWidth = newRight - newLeft;
        double percent = 0.0;

        if ( newWidth ) {
            percent = newWidth / oldWidth;
        }

        //pos = (int)((double)pos * percent);
    } else {
        if ( _info[0].minWidth > -1 ) {
            pos = _info[0].minWidth + thickness/2;
        } else if ( _info[1].minWidth > -1 ) {
            pos = (int)(newRight - newLeft - _info[1].minWidth - thickness/2);
        } else {
            pos = (int)((double)(newRight - newLeft) / 2);
        }
    }


    rect = *newRect;
    move();
}

bool
Splitter::setSize( HWND hwnd, int width, int height, RECT* needed )
{
    int swidth = width;
    int sheight = height;
    RECT srect = rect;
    swap( &srect );
    if ( horizontal ) {
        swidth = height;
        sheight = width;
    }
        
    if ( hFirst == hwnd ) {
        pos = swidth + thickness/2;
        *needed = srect;
        needed->bottom = needed->top + sheight;
        swap( needed );
        return true;
    } else if ( hSecond == hwnd ) {
        pos = srect.right - width - srect.left;
        *needed = srect;
        needed->bottom = needed->top + sheight;
        swap( needed );
        return true;
    } else if ( first && first->setSize( hwnd, width, height, needed ) ) {
        swap( needed );
        pos = needed->right - needed->left + thickness/2;
        swap( needed );
        move();
        return true;
    } else if ( second && second->setSize( hwnd, width, height, needed ) ) {
        swap( needed );
        pos = srect.right - (needed->right - needed->left) - srect.left - thickness/2;
        swap( needed );
        move();
        return true;
    }

    return false;

}

void
Splitter::swap( RECT* rect )
{
    if ( horizontal ) {
        int temp;
        temp = rect->left;
        rect->left = rect->top;
        rect->top = temp;
        temp = rect->right;
        rect->right = rect->bottom;
        rect->bottom = temp;
    }
}

RECT 
Splitter::barRect()
{
    RECT brect = rect;
    swap( &brect );
    brect.left = brect.left + pos - thickness / 2;
    brect.right = brect.left + thickness;
    swap( &brect );
    return brect;
}

void
Splitter::move()
{
    int i;
    RECT nrect = rect;
    swap(&nrect);
    for ( i = 1; i >= 0; i-- ) {
        SplitInfo info = _info[i];
        int width;
        if ( i == 0 ) {
            width = pos - thickness/2;
        } else {
            width = nrect.right - nrect.left - thickness / 2;
        }
        if ( horizontal ) {
            int temp = info.minWidth;
            info.minWidth = info.minHeight;
            info.minHeight = temp;
            temp = info.maxWidth;
            info.maxWidth = info.maxHeight;
            info.maxHeight = temp;
        }

        if ( i == 1 ) {
            if ( info.minWidth > -1 && width < info.minWidth ) {
                pos = (int)(nrect.right - info.minWidth - nrect.left);
            }
            if ( info.maxWidth > -1 && width > info.maxWidth ) {
                pos = (int)(nrect.right - info.maxWidth - nrect.left);
            }
        } else {
            if ( info.minWidth > -1 && width < info.minWidth ) {
                pos = info.minWidth;
            }
            if ( info.maxWidth > -1 && width > info.maxWidth ) {
                pos = info.maxWidth;
            }
        }
    }
    RECT fRect = rect;
    RECT sRect = rect;
    swap( &fRect );
    swap( &sRect );

    fRect.right = fRect.left + pos - thickness/2;
    sRect.left = sRect.left + pos + thickness/2;

    swap( &fRect );
    swap( &sRect );

    if ( hFirst ) {
        MoveWindow( hFirst, 
                fRect.left, fRect.top,
                fRect.right - fRect.left, fRect.bottom - fRect.top,
                TRUE );
    } else if ( first ) {
        first->format( &fRect );
    }

    if ( hSecond ) {
        MoveWindow( hSecond, 
                sRect.left, sRect.top,
                sRect.right - sRect.left, sRect.bottom - sRect.top,
                TRUE );
    } else if ( second ) {
        second->format( &sRect );
    }

}

void
Splitter::draw( HDC hdc )
{
    RECT brect = rect;
    swap( &brect );
    brect.left = brect.left + pos - thickness / 2;
    brect.right = brect.left + thickness;
    swap( &brect );
    if ( 1 || _immobile ) {
        FillRect( hdc, &brect, (HBRUSH)GetStockObject( LTGRAY_BRUSH ) );
    } else {
        DrawFrameControl( hdc, &brect, DFC_BUTTON, DFCS_BUTTONPUSH |
                (_dragging ? DFCS_PUSHED : 0 ));

    }
    if ( first ) {
        first->draw( hdc );
    }

    if ( second ) {
        second->draw( hdc );
    }
}

void 
Splitter::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    if ( _over ) {
        onMouseMove( hwnd, x, y, keyFlags );
        _dragStart.x = x;
        _dragStart.y = y;
        _dragging = true;
        _posStart = pos;
        SetCapture(hwnd);
    } else {
        if ( first ) {
            first->onLButtonDown( hwnd, fDoubleClick, x, y, keyFlags );
        }
        if ( second ) {
            second->onLButtonDown( hwnd, fDoubleClick, x, y, keyFlags );
        }
    }
}

void 
Splitter::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( _over ) {
        onMouseMove( hwnd, x, y, keyFlags );

        _dragging = false;
        RECT brect = barRect();
        InvalidateRect( hwnd, &brect, FALSE );
        ReleaseCapture();
    } else {
        if ( first ) {
            first->onLButtonUp( hwnd, x, y, keyFlags );
        }
        if ( second ) {
            second->onLButtonUp( hwnd, x, y, keyFlags );
        }
    }
}

void
Splitter::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    RECT brect = barRect();

    if ( !_immobile && PointInRect( &brect, x, y ) ) {
        SetCursor( LoadCursor( NULL, horizontal ? IDC_SIZENS : IDC_SIZEWE ) );
        _over = true;
    } else if ( !_dragging ) {
        _over = false;
    }

    if ( _dragging ) {
        RECT srect = rect;
        int distance = horizontal ? y - _dragStart.y : x - _dragStart.x;
        swap( &srect );
        pos = _posStart + distance;
        if ( pos > srect.right - thickness/2 ) {
            pos = srect.right - thickness/2;
        } else if ( pos < thickness/2 ) {
            pos = thickness/2;
        }
        move();
        InvalidateRect( hwnd, NULL, FALSE );
    } else {
        if ( first ) {
            first->onMouseMove( hwnd, x, y, keyFlags );
        }
        if ( second ) {
            second->onMouseMove( hwnd, x, y, keyFlags );
        }
    }
}

/**
 * Construct a SplitWindow object.
 */
SplitWindow::SplitWindow(HINSTANCE hInstance)
{
	_hInstance = hInstance;
	_pszClassName = _T("SplitWindow");
	_pszTitle = _T("insert title here");
    _mainSplitter = 0;

    _dwStyle |= WS_CLIPCHILDREN | WS_VISIBLE;  
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = NULL;
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
}


SplitWindow::~SplitWindow()
{
    delete _mainSplitter;
}

void
SplitWindow::create( HWND hParent )
{
    Window::Create(CW_USEDEFAULT, CW_USEDEFAULT,
            CW_USEDEFAULT, CW_USEDEFAULT, hParent, 0, _hInstance );
    assert(hwnd);
}

LRESULT 
SplitWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}

void 
SplitWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    if ( _mainSplitter ) {
        _mainSplitter->onLButtonDown( hwnd, fDoubleClick, x, y, keyFlags );
    }
}

void 
SplitWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( _mainSplitter ) {
        _mainSplitter->onLButtonUp( hwnd, x, y, keyFlags );
    }
}

void
SplitWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( _mainSplitter ) {
        _mainSplitter->onMouseMove( hwnd, x, y, keyFlags );
    }
}

/** 
 * Set the main splitter. A SplitWindow must have at least one splitter.
 */
void
SplitWindow::setMainSplitter( Splitter* splitter )
{
    delete _mainSplitter;
    _mainSplitter = splitter;

    if ( _mainSplitter ) {
        RECT rect = getClientRect();
        _mainSplitter->format( &rect );
    }
}

void
SplitWindow::drawSplitters(HDC hdc)
{
    if ( _mainSplitter ) {
        _mainSplitter->draw( hdc );
    }

}


void 
SplitWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    if ( _mainSplitter ) {
        _mainSplitter->draw(hdc);
    }

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
SplitWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    if ( _mainSplitter ) {
        RECT rect = getClientRect();
        _mainSplitter->format( &rect );
    }
}
