#ifndef STREAM_H
#define STREAM_H

#include <stdio.h>
#include <windows.h>
#include <tchar.h>

/** 
 * @defgroup Stream
 *
 * The Stream classes provide a generic, yet simple way to abstract files
 * without all that std::ostream crap.
 */

/**
 * @ingroup Stream
 * The Stream class is the base class for Stream. A stream represents a file,
 * and it supports the gets(), read(), write(), and eof() operations.
 */
class Stream
{   
public:
    Stream();
    virtual ~Stream();

    virtual _TCHAR* gets( _TCHAR*, int n );
    virtual unsigned read( void* buffer, int rec_size, int num_recs );
    virtual unsigned write( void* buffer, int rec_size, int num_recs );
    virtual bool eof();
};

/**
 * @ingroup Stream
 * Writes a null terminated string to a stream.
 */
bool write( Stream* stream, const char* str );

/**
 * @ingroup Stream
 * Writes a null terminated wide string to a stream.
 */
bool write( Stream* stream, const wchar_t* str );

/**
 * @ingroup Stream
 * Stream class for a file stored on disk.
 */
class FileStream : public Stream
{
public:
    FileStream();
    virtual ~FileStream();
    bool open( const _TCHAR* name, const _TCHAR* mode );
    void close();
    virtual _TCHAR* gets(_TCHAR*, int n);
    virtual unsigned read( void* buffer, int rec_size, int num_recs);
    virtual unsigned write( void* buffer, int rec_size, int num_recs);
    virtual bool eof();
protected:
    FILE* _handle;
};

/**
 * @ingroup Stream
 * Stream class for a zipped file stored on disk. Requires the zlib library to
 * be linked.
 */
class GzFileStream : public Stream
{
public:
    GzFileStream();
    virtual ~GzFileStream();

    bool open( const _TCHAR* name, const _TCHAR* mode );
    void close();
    virtual char* gets(char*, int n);
    virtual wchar_t* gets(wchar_t* val, int n);
    virtual unsigned read( void* buffer, int rec_size, int num_recs);
    virtual unsigned write( void* buffer, int rec_size, int num_recs);
    virtual bool eof();
protected:
    void* _handle;
};


/**
 * @ingroup Stream
 * Stream class for a temporary file. Automatically chooses a filename.
 */
class TempFileStream : public FileStream
{
public:
    TempFileStream( const _TCHAR* friendlyName);
    virtual ~TempFileStream();

    bool open(const _TCHAR* mode = _T("r+b"));
    const _TCHAR* name();
    const _TCHAR* file();

private:
    _TCHAR* _file;
    _TCHAR* _name;
};

/** 
 * @ingroup Stream
 * An archive represents a collection of streams.
 */
class Archive
{
public:
    Archive( Stream* stream );
    ~Archive();

    bool addFile( const _TCHAR* filename, const _TCHAR* friendlyName );
    TempFileStream* extract();

private:
    struct Header {
        _TCHAR name[256];
        unsigned size;
        unsigned checksum;
    };

    unsigned checksum( const char* data, unsigned bytes );
    Stream* _stream;
};

/**
 * @ingroup Stream
 * A record contains a collection of key/value pairs which can be written to
 * and read from a Stream all at once. For data integrity, a checksum is stored
 * with the record. The key are strings, and the values can be any of several
 * common data types.
 */
class Record
{
public:
    Record(int max_properties);
    ~Record();

    bool read( Stream* stream );
    bool write( Stream* stream );

    const _TCHAR* get( const _TCHAR* name );
    bool get( const _TCHAR* name, int& value );
    bool get( const _TCHAR* name, RECT& value );
    bool get( const _TCHAR* name, unsigned& value );
    bool get( const _TCHAR* name, _TCHAR*& value );
    bool get( const _TCHAR* name, POINT& value );
    bool get( const _TCHAR* name, unsigned long& value );
    bool get( const _TCHAR* name, long& value );
    bool get( const _TCHAR* name, bool& value );
    bool get( const _TCHAR* name, float& value );

    void set( const _TCHAR* name, const _TCHAR* value );
    void set( const _TCHAR* name, const int& value );
    void set( const _TCHAR* name, const RECT& value );
    void set( const _TCHAR* name, const unsigned& value );
    void set( const _TCHAR* name, const unsigned long& value);
    void set( const _TCHAR* name, const POINT& value);
    void set( const _TCHAR* name, const long& value );
    void set( const _TCHAR* name, const bool& value );
    void set( const _TCHAR* name, const float& value );

private:
    struct Property {
        _TCHAR* name;
        _TCHAR* value;
    }* _props;

    Property* getProp( const _TCHAR name );
    void newProp( const _TCHAR* name, const _TCHAR* value );
    void clear();

    int _max;
    int _used;
};

class ZData;
class ZStream
{   
public:
    ZStream( Stream* source );
    virtual ~ZStream();

    virtual unsigned read( void* buffer, int rec_size, int num_recs );
    virtual unsigned write( void* buffer, int rec_size, int num_recs );
    virtual bool eof();
private:
    ZData* _zdata;
    Stream* _source;
    bool _writing;
    bool _reading;
};

#endif // STREAM_H
