#include "TextControl.h"
#include "BaseWnd.h"
#include <assert.h>
#include <stdio.h>

TextControl::TextControl(HWND hwnd) :
    FadingControl(hwnd)
{
    _text = _tcsdup(TEXT(""));
    _font = new Font( TEXT("Tahoma"), 100 );
    _shadow = false;
    _colour = RGB( 200, 100, 0 );
    _halign = HAlign_Left;
    _wrap = false;
    format();
}

TextControl::~TextControl()
{
    delete _font;
    free( _text );
}

void
TextControl::setText(const TCHAR* text )
{
    free( _text );
    _text = _tcsdup( text );
    format();
}

void 
TextControl::setHorizontalAlignment( HAlign_e a )
{
    _halign = a;
    format();
}

void
TextControl::setFont( const Font& font )
{
    delete _font;
    _font = new Font( font );
    format();
}

void
TextControl::setShadow( bool on )
{
    _shadow = on;
}

void
TextControl::setColour( COLORREF colour )
{
    _colour = colour;
}

void
TextControl::move( int x, int y, int cx, int cy )
{
    bool doformat = false;

    if ( _rect.right - _rect.left - 1 != cx || 
           _rect.bottom - _rect.top - 1 != cy ) {
        doformat = true;
    } 

    FadingControl::move( x, y, cx, cy );
    if ( doformat ) {
        format();
    }
}

SIZE
TextControl::calcSize( int width )
{
    if ( width == 0 ) {
        width = _rect.right - _rect.left;
    }

    RECT rect;
    rect.left = 0;
    rect.top = 0;
    rect.right = rect.left + width;
    rect.bottom = rect.top;

    HDC dc = GetDC(_hwnd);
    HFONT hOldFont = (HFONT)SelectObject( dc, *_font );
    assert( _tcslen(_text) > 0 );

    if ( _wrap ) {
        assert( width != 0 );
        int height = DrawText( dc, _text, _tcslen( _text ), &rect, 
                DT_CALCRECT | DT_WORDBREAK );
        if ( height == 0 ) {
            assert(0);
            int a = GetLastError();
        }

        rect.bottom = rect.top + height;
    } else {
        int height = DrawText( dc, _text, _tcslen( _text ), &rect, 
            DT_CALCRECT | DT_SINGLELINE );
        if ( height == 0 ) {
            assert(0);
        }

    }

    SelectObject( dc, hOldFont );
    ReleaseDC(_hwnd, dc);

    SIZE ret;
    ret.cx = rect.right - rect.left;
    ret.cy = rect.bottom - rect.top;
    return ret;
}

int
TextControl::moveFormat( int x, int y, int width )
{
    if ( !_wrap ) {
        int height = _font->getHeight();
        if ( height < 0 ) {
            height = -height;
        }
        FadingControl::move( x, y, width, height );
        format();
        return height;
    }


    SIZE size = calcSize(width);

    _rect.left = x;
    _rect.top = y;
    _rect.right = _rect.left + size.cx;
    _rect.bottom = _rect.top + size.cy;

    return _rect.bottom - _rect.top;
}

void
TextControl::draw( HDC hdc )
{
    Control::draw( hdc );
    
    if ( _shadow ) {
        DeviceContext dc(hdc);
        dc.selectFont( *_font);
        SetBkMode( dc, TRANSPARENT );
        SetTextColor( dc, RGB( 50, 50, 50 ) );
        int offset = 1;

        if ( _font->getHeight() < 24 ) {
            offset = 1;
        }

        RECT rect = getWindowRect();

        if ( !_wrap ) {
            POINT pos = _pos;
            if ( _parent ) {
                _parent->offset( &pos );
            }
            if ( !TextOut( dc, rect.left + offset + pos.x, 
                    rect.top + offset + pos.y,
                    _text, _tcslen( _text ) ) ) {
                assert(0);
            }
        } else {
            OffsetRect( &rect, offset, offset );
            if ( 0 == DrawText( dc, _text, _tcslen( _text ), &rect,
                   DT_WORDBREAK ) ) {
               assert(0);
            } 
        }
    }
    
    drawFaded( hdc );
}

void 
TextControl::drawFaded( HDC hdc )
{
    DeviceContext dc(hdc);
    SetBkMode( dc, TRANSPARENT );
    dc.selectFont( *_font);
    SetTextColor( dc, getFadeColour( _colour ) );
    POINT pos = _pos;
    RECT rect = getWindowRect();
    if ( _parent ) {
        _parent->offset( &pos );
    }
    if ( !_wrap ) {
        if( 0 == TextOut( dc, rect.left + pos.x, rect.top + pos.y, 
                _text, _tcslen( _text ) ) ) {
            assert(0);
        }

    } else {
        if ( 0 == DrawText( dc, _text, _tcslen( _text ), &rect,
               DT_WORDBREAK ) ) {
            assert(0);
        }
    }
}

void
TextControl::format()
{
    HDC dc = GetDC( _hwnd );
    HFONT hOldFont = (HFONT)SelectObject( dc, (HFONT)*_font );

    SIZE size;

    if ( !GetTextExtentPoint32( dc, _text, _tcslen( _text ), &size ) ) {
        assert( 0 );
        int a = GetLastError();
        size.cx = 1;
        size.cy = 1;
    }

    if ( _halign == HAlign_Left ) {
       _pos.x = _pos.y = 0;
    } else if ( _halign == HAlign_Centre ) {
       _pos.y = 0;
       _pos.x = (_rect.right - _rect.left) / 2 - size.cx / 2;
    }

    SelectObject( dc, hOldFont );
    ReleaseDC( _hwnd, dc );
}

Font*
TextControl::getFont()
{
    return _font;
}

void 
TextControl::setWrap( bool on )
{
    _wrap = on;
    
}
