#include <assert.h>
#include <errno.h>
#include "Thread.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "debug.h"

#ifdef LINUX
#include <pthread.h>
#include <unistd.h>
#include <sys/time.h>
#include <signal.h>
#endif

#ifdef _WIN32
#include <windows.h>
#include <process.h>
#endif

Thread::Thread() :
  _started(false)
{
  //trace << "Thread::Thread()" << endl;
  _debugInfo = this;
#ifdef _WIN32
  _doneEvent = CreateEvent(NULL, TRUE, TRUE, NULL);
  _hThread = NULL;
#endif
}

Thread::~Thread()
{
    join();

#ifdef _WIN32
    CloseHandle(_doneEvent);
#endif

  _debugInfo = 0;
}

void 
Thread::start()
{
  //trace << "Thread::start()" << endl;
    join();
  _started = true;

#ifdef _WIN32
  ResetEvent(_doneEvent);
  _hThread = (HANDLE)_beginthreadex(0, 0, (unsigned int (__stdcall*)(void*))Thread::main, 
      (void*)this, 0, (unsigned int*)&_dwThreadId);
#endif

#ifdef LINUX
  pthread_create(&_pthread, 0,  
			      (void*(*)(void*))Thread::main, 
			      (void*)this);
#endif
}

void
Thread::elevatePriority()
{
    SetThreadPriority( _hThread, THREAD_PRIORITY_ABOVE_NORMAL ); 
}

/**
 * Returns the thread id of the current thread.
 */
unsigned int 
Thread::myID()
{
#ifdef LINUX
	return pthread_self();
#endif

#ifdef _WIN32
    return GetCurrentThreadId();
#endif
}

unsigned int
Thread::getID()
{
#ifdef LINUX
	return _pthread;
#endif

#ifdef _WIN32
    return _dwThreadId;
#endif
}

/**
 * Waits for the thread to end.
 */
void
Thread::join()
{
  if ( _started ) {
#ifdef LINUX
    pthread_join( _pthread, 0 );
#endif
  }

#ifdef _WIN32
    WaitForSingleObject(_doneEvent, INFINITE);
#endif
}

void 
Thread::main( void* arg)
{
  Thread* thread = (Thread*)arg;
  thread->assertValid();
  thread->run();
#ifdef LINUX
  pthread_exit(0);
#endif

#ifdef _WIN32
    CloseHandle(thread->_hThread);
    thread->_hThread = 0;
    SetEvent(thread->_doneEvent);
#endif
}

/**
 * Cause the thread to sleep for the given number of milliseconds.
 */
void Thread::sleep(int ms)
{
#ifdef LINUX
  usleep(ms * 1000);
#endif

#ifdef _WIN32
    Sleep(ms);
#endif
}

void Thread::assertValid() 
{
  assert(_debugInfo == this );
}


void Thread::run()
{
  assertValid();
  //trace << "Bare thread run." << endl;
}

Mutex::Mutex()
{
    _owner = 0;
#ifdef LINUX
  pthread_mutex_init(&_mutex, 0);
#endif

#ifdef _WIN32
    InitializeCriticalSection(&_cs);
#endif
}

Mutex::~Mutex()
{
#ifdef LINUX
  pthread_mutex_destroy(&_mutex);
#endif

#ifdef _WIN32
    DeleteCriticalSection(&_cs);
#endif
}

void
Mutex::lock(const char* file, unsigned line)
{
    if ( _owner == Thread::myID() ) {
        assert(0 /* locking mutex twice in a row */ );
    }

#ifdef LINUX
    pthread_mutex_lock(&_mutex);
#endif

#ifdef _WIN32
    EnterCriticalSection(&_cs);
#endif
    
    dbgprint(( DTHREAD, "%s:%d: Thread %d locked mutex %x", 
                file, line, Thread::myID(), this ));

    _owner = Thread::myID();
}

void
Mutex::unlock(const char* file, unsigned line)
{
    assert( _owner == Thread::myID() );
    _owner = 0;
#ifdef LINUX
  pthread_mutex_unlock(&_mutex);
#endif

#ifdef _WIN32
    dbgprint(( DTHREAD, "%s:%d: Thread %d unlocking mutex %x", 
                file, line, GetCurrentThreadId(), this ));
    LeaveCriticalSection(&_cs);
#endif
}


Condition::Condition()
{
#ifdef LINUX
  pthread_cond_init(&_cond, 0);
#endif
#ifdef _WIN32
  _hEvent = CreateEvent( NULL, FALSE, FALSE, NULL );
#endif
}

Condition::~Condition()
{
#ifdef LINUX
  pthread_cond_destroy(&_cond);
#endif
#ifdef _WIN32
  CloseHandle( _hEvent );
#endif
}

bool
Condition::wait(Mutex& mutex, const char* file, unsigned line)
{

#ifdef LINUX
  int ret = pthread_cond_wait(&_cond, &mutex._mutex);
  if ( ret != 0 ) {
      printf("Error: %s\n", strerror(ret) );
  }
  return ret == 0;
#endif
#ifdef _WIN32
  mutex.unlock(file, line);
  bool ret = WAIT_OBJECT_0 == WaitForSingleObject( _hEvent, INFINITE );
  mutex.lock(file, line);
  return ret;
#endif
}

bool
Condition::wait(Mutex& mutex, int ms)
{
#ifdef LINUX
  struct timespec ts;
  struct timeval tv;
  gettimeofday(&tv, 0);
  ts.tv_sec = tv.tv_sec + ms / 1000;
  ts.tv_nsec = tv.tv_usec * 1000 + ( ms % 1000 ) * 1000000;

  // trace << (unsigned long)ts.tv_sec << ":" << (unsigned long)ts.tv_nsec <<
  // endl;
 
  int err = pthread_cond_timedwait(&_cond, &mutex._mutex, &ts);

  if ( err == ETIMEDOUT ) {
    //trace << "pthread timed out." << endl;
  } else if ( err == EINTR ) {
    //trace << "pthread interrupted." << endl;
  }

  return err == 0;
#endif
#ifdef _WIN32
  mutex.unlock(DBG);
  bool ret = WAIT_OBJECT_0 == WaitForSingleObject( _hEvent, ms );
  mutex.lock(DBG);
  return ret;
#endif
}

void
Condition::signal()
{
#ifdef LINUX
  pthread_cond_signal(&_cond);
#endif
#ifdef _WIN32
  SetEvent( _hEvent );
#endif
}

Semaphore::Semaphore( unsigned int value )
{
#ifdef LINUX
    if ( 0 != sem_init( &_sem, 0, value ) ) {
        perror("sem_init");
        abort();
    }
#endif
}

Semaphore::~Semaphore()
{
#ifdef LINUX
    sem_destroy(&_sem);
#endif
}

void
Semaphore::post()
{
#ifdef LINUX
    sem_post(&_sem);
#endif
}

void
Semaphore::wait()
{
#ifdef LINUX
    sem_wait(&_sem);
#endif
}

bool Thread::UnitTest()
{
  class TestThread : public Thread
  {
  public:
    char c;
    TestThread(char c) : c(c) {};

    void run()
    {
      for(int i = 0; i < 80; i++) {
          printf("%c", c);
	Thread::sleep(10);
      }
    }
  };

    TestThread A('a');
    TestThread B('b');
    
    A.start();
    B.start();

    A.join();
    B.join();

    return true;
}

