#ifndef THREAD_H
#define THREAD_H

//#define _WIN32

#ifdef LINUX
#include <pthread.h>
#include <semaphore.h>
#endif

#ifdef _WIN32
#include <windows.h>
#endif

#ifndef DBGDATA
#define DBGDATA     __FILE__, __LINE__
#endif

/**
 * The Thread class provides a threading framework that works on both linux and
 * win32. On Linux, you must link with the pthreads library.
 *
 * To use the Thread class, you will usually create a subclass and override the
 * run() method to perform a task. You may also have to modify the destructor
 * to cause the thread to quit, and then call join() to ensure that it has done
 * so.

 Here is an example of how to use a thread.
 \code
  class TestThread : public Thread
  {
  public:
    char c;
    TestThread(char c) : c(c) {};

    void run()
    {
      for(int i = 0; i < 80; i++) {
          printf("%c", c);
          Thread::sleep(10);
      }
    }
  };

    // This example should print 'abababab' etc to the screen.
    TestThread A('a');
    TestThread B('b');
    
    A.start();
    B.start();

    A.join();
    B.join();
 \endcode
 */
class Thread
{
 public:
  Thread();
  virtual ~Thread();

  virtual void start();

  static unsigned int myID();
  unsigned int getID();

  void elevatePriority();

  virtual void run();

  void join();
  static void sleep(int ms);

  static bool UnitTest();

  void assertValid();

 private:
  static void main( void* arg );
#ifdef LINUX
  pthread_t _pthread;
#endif

#ifdef _WIN32
    DWORD _dwThreadId;
    HANDLE _doneEvent;
    HANDLE _hThread;
#endif

  bool _started;
  Thread* _debugInfo;
};

class Condition;

/**
 * a Mutex provides mutual exclusion. For debugging purposes, the lock() and
 * unlock() functions require a file and line number. For this purpose you can
 * use the DBG macro which has been provided:
 *
 * \code
 * Mutex mutex();
 *
 * mutex.lock(DBG);
 * mutex.unlock(DBG);
 * \endcode
 */
class Mutex
{
 public:
  Mutex();
  ~Mutex();

  void lock(const char* file, unsigned line);
  void unlock(const char* file, unsigned line);

  /**
   * The Mutex::Scope object will lock a mutex and automatically unlock it when
   * it goes out of scope.
   */
  class Scope {
  public:
    Scope(Mutex& pm) : m(&pm) { m->lock(__FILE__, __LINE__); }
    ~Scope() { try { m->unlock(__FILE__, __LINE__); m = 0; } catch(...) {} }
  private:
    Mutex* m;
  };

 private:
#ifdef LINUX
  pthread_mutex_t _mutex;
#endif

  unsigned _owner;
#ifdef _WIN32
    CRITICAL_SECTION _cs;
#endif

  friend class Condition;
};

/**
 * Waiting on a condition will cause your thread to block until the condition
 * is signaled by another thread.
 */
class Condition
{
 public:
  Condition();
  ~Condition();

  bool wait(Mutex& mutex, const char* file, unsigned line);
  bool wait(Mutex& mutex, int ms);
  void signal();
 private:
#ifdef LINUX
  pthread_cond_t _cond;
#endif
#ifdef _WIN32
  HANDLE _hEvent;
#endif
};

class Semaphore
{
    public:
    Semaphore( unsigned int value );
    ~Semaphore();

    void wait();
    void post();

    private:
#ifdef LINUX
    sem_t _sem;
#endif
};

#endif //THREAD_H
