#ifndef UNDOREDO_H
#define UNDOREDO_H

#include <tchar.h>

class UndoRecord;
class UndoClient;
class UndoStack;

/**
 * @defgroup UndoRedo
 * The Undo/Redo framework makes it easy to provide multi-level undo/redo
 * functionality, simply by overriding a few classes.
 */

/**
 * @ingroup UndoRedo
 *
 * Override this class to receive notifications of what to undo/redo. The
 * undoRedo() function is passed an UndoRecord, and must perform whatever
 * action is requested using the data in the record.
 *
 */
class UndoClient
{
public:
    UndoClient();
    virtual ~UndoClient();

    virtual void undoRedo( UndoRecord* record, bool redo ) = 0;
};

/**
 * @ingroup UndoRedo
 *
 * Override this class for each type of Undo/Redo action that you want to
 * perform. The base class contains only a description of the action (for
 * displaying, for example, in a menu). The rest of the information must be
 * recorded in your subclasses.
 */
class UndoRecord
{
public:
    UndoRecord( const _TCHAR* description );
    virtual ~UndoRecord();

    bool done;

private:
    friend class UndoStack;
    _TCHAR* description;
    UndoRecord* next;
    UndoRecord* prev;
};

/**
 * @ingroup UndoRedo
 * 
 * This class implements a multilevel undo stack.
 *
 * To add Undo/Redo functionality to your program:
 *
 * -# Derive a subclass of UndoClient. This will usually be your document
 *  class. Implement the undoRedo method. The only way to modify the document
 *  or make changes to it will be by creating an UndoRedo record and calling
 *  UndoStack::action() on it.
 *
 *  -# Derive a subclass of UndoRecord. The base class only keeps track of the
 *  name of the action ("draw rectangle" for example) and whether it is done or
 *  not. Your subclass will have to contain all the information to either to or
 *  undo the action. You might create a whole backup of the document, if this
 *  is feasible, or instead you might just keep a record of how the document
 *  changed. If the action is drawing a rectangle, you would probably keep
 *  track of its size and position. A nice, object-oriented way to do things is
 *  to put a method in each possible UndoRecord subclass called "do(Document*)" and have
 *  so it modifies the document, doing whatever the class is supposed to do.
 *
 *  -# When you want to undo something, call UndoStack::undo() or redo(). This
 *  will cause your client's UndoClient::undoRedo() to be called with a given
 *  UndoRecord.
 *
 */
class UndoStack
{
public:
    UndoStack( UndoClient* client, unsigned maxRecords );
    ~UndoStack();

    void clear();
    void clearRedo();

    bool canUndo();
    bool canRedo();

    void action( UndoRecord* record, bool callClient );
    UndoRecord* getUndoRecord();

    void undo();
    void redo();
    
    const _TCHAR* undoDescription();
    const _TCHAR* redoDescription();

private:
    void destroyRecords( UndoRecord*& record );
    unsigned _maxRecords;
    unsigned _numRecords;
    UndoRecord* _firstRecord;
    UndoRecord* _undoRecord;
    UndoClient* _client;
};


#endif // UNDOREDO_H

