#include <stdio.h>
#include "BaseWnd.h"
#include "XmlApplication.h"
#include "TagParser.h"
#include <windowsx.h>
#include <commctrl.h>
#include <assert.h>
#include "SplitWindow.h"
#include "SplitterWindow.h"
using std::_tstring;

/****************************************************************************
 Default XML Frame Window
 ****************************************************************************/ 
class DefaultXmlFrameWindow : public Window
{
public:
    DefaultXmlFrameWindow(HINSTANCE, XmlApplication* app, const _TCHAR* icon);
    ~DefaultXmlFrameWindow();
    void reformat();

    ButtonBar* buttonBar;
    WindowFunctions* client;
    std::_tstring onCloseEvent;

private:
    LRESULT 
    WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
            LPARAM lParam, PBOOL pbProcessed);

    void onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify);
    BOOL onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct);
    void onSize(HWND hwnd, UINT state, int cx, int cy);
    void onClose(HWND);
    XmlApplication* _app;
    StatusBar _statusBar;
};

DefaultXmlFrameWindow::DefaultXmlFrameWindow(HINSTANCE hInstance,
    XmlApplication* app, const _TCHAR* icon)
{
    _app = app;
	_hInstance = hInstance;
	_pszClassName = TEXT("DefaultXmlFrameWindow");
	_pszTitle = TEXT("DefaultXmlFrameWindow Window");

    _dwStyle |= WS_THICKFRAME | WS_SYSMENU | WS_CLIPCHILDREN |
        WS_VISIBLE | WS_CAPTION | WS_MAXIMIZEBOX | WS_MINIMIZEBOX | 
        WS_CLIPSIBLINGS;
 
    _WndClass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    _WndClass.hCursor = (HCURSOR)LoadCursor(NULL, IDC_ARROW);
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _WndClass.hIcon = LoadIcon( hInstance, icon );
    _dwExtendedStyle |= 0;

    buttonBar = NULL;
    client = NULL;
}

DefaultXmlFrameWindow::~DefaultXmlFrameWindow()
{
    delete buttonBar;
    delete client;
}

LRESULT 
DefaultXmlFrameWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        //HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        //HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        //HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        //HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        //HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
    }
    
    *pbProcessed = FALSE;
    return 0;
}

void
DefaultXmlFrameWindow::reformat()
{
    RECT rect;
    GetClientRect( hwnd, &rect );
    onSize( hwnd, 0, rect.right-rect.left, rect.bottom-rect.top );
}

void 
DefaultXmlFrameWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    RECT clientRect = { 0, 0, cx, cy };

    if ( _statusBar.hwnd ) {
        RECT statusRect;
        GetWindowRect( _statusBar.hwnd, &statusRect );
        statusRect.top = cy - (statusRect.bottom-statusRect.top);
        statusRect.left = 0;
        statusRect.right = cy;
        _statusBar.moveWindow( &statusRect, true );
        clientRect.bottom = statusRect.top;
    }

    if ( buttonBar ) {
        RECT rect = buttonBar->getClientRect();
        if ( buttonBar->isHorizontal() ) {
            buttonBar->moveWindow( 0, 0, cx, rect.bottom, true );
            clientRect.top = rect.bottom;
        } else {
            buttonBar->moveWindow( 0, 0, rect.right, clientRect.bottom-clientRect.top, true );
            clientRect.left = rect.right;
        }
    }


    if ( client ) {
        client->moveWindow( &clientRect, false );
    }
}

void
DefaultXmlFrameWindow::onClose(HWND)
{
    if ( !onCloseEvent.empty() ) {
        _app->sendEvent( onCloseEvent );
    }
}

void 
DefaultXmlFrameWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    _app->onCommandId( id );
}

BOOL 
DefaultXmlFrameWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    _statusBar.create( SBARS_SIZEGRIP, _T(""), hwnd, 0 );
    return TRUE;
}

/****************************************************************************
 XmlAppFactory
 ****************************************************************************/ 
XmlAppFactory::XmlAppFactory()
{
}

XmlAppFactory::~XmlAppFactory()
{
}

WindowFunctions*
XmlAppFactory::createWindow( HINSTANCE hInstance, HWND hParent, const _TCHAR* type )
{
    if ( _tcscmp( type, _T("edit") ) == 0 ) {
        EditControl* edit = new EditControl();
        edit->create( NULL, 0, 0, 100, 100, hParent, 1 );
        return edit;
    } else if ( _tcscmp( type, _T("combo") ) == 0 ) {
        ComboBox* combo = new ComboBox();
        combo->create( CBS_DROPDOWNLIST, 0, 0, 320, 240, hParent, 0 );
        return combo;
    }
    return NULL;
}


XmlAppListener::~XmlAppListener()
{
}

/****************************************************************************
 XmlApplication
 ****************************************************************************/ 
XmlApplication::XmlApplication( HINSTANCE hInstance, XmlAppListener* listener, 
    XmlAppFactory* factory ) :
    _hInstance( hInstance ),
    _factory(factory),
    _listener( listener),
    _delFactory( false )
{
    _nextId = 1000;
    if ( _factory == NULL ) {
        _factory = new XmlAppFactory();
        _delFactory = true;
    }
}

XmlApplication::~XmlApplication()
{
    for( WindowList::iterator iter = _windows.begin();
         iter != _windows.end();
         ++iter )
    {
        delete *iter;
    }

    for( std::list<Menu*>::iterator iter = _menus.begin();
         iter != _menus.end();
         ++iter )
    {
        delete *iter;
    }

    if ( _delFactory ) {
        delete _factory;
    }
}

void 
XmlApplication::parseTag( CTag* tag )
{
    if ( tag->name == _T("framewindow") ) {
        parseFrameWindow( tag );
    } else {
        parseChildren( tag );
    }
}

void
XmlApplication::parseChildren( CTag* tag )
{
    CTag::List& tags = tag->ChildList;
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        parseTag( *iter );
    }
}

bool 
XmlApplication::parseSize( const _TCHAR* str, int* width, int* height )
{
    return 2 == _tscanf( str, L"%dx%d", width, height ); 
}

bool
XmlApplication::parseColour( const _TCHAR* name, unsigned* colour )
{
    unsigned red = 0;
    unsigned green = 0;
    unsigned blue = 0;

    if ( _tcscmp( _T("black"), name) == 0 )  {
        red = green = blue = 0;
    } else if ( _tcscmp( _T("white"), name ) == 0 ) {
        red = green = blue = 255;
    } else if ( 3 != sscanf( name, _T("#%02x%02x%02x"), &red, &green, &blue ) ) {
        return false;        
    }

    *colour = RGB( red, green, blue );
    return false;
}

Menu*
XmlApplication::parseMenu( CTag* tag, bool popup )
{
    Menu* menu = new Menu( popup );
    _menus.push_back( menu );

    CTag::List& tags = tag->ChildList;
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        CTag* tag = *iter;
        if ( tag->name == _T("menu") ) {
            Menu* subMenu = parseMenu( tag, true );
            menu->append( tag->attributes["text"].c_str(), *subMenu );
        } else if ( tag->name == _T("menubar") ) {
            menu->appendBar();
        } else if ( tag->name == _T("menuitem") ) {
            _tstring textStr = tag->attributes["text"];
            _tstring onclick = tag->attributes["onclick"];
            menu->append( textStr.c_str(), mapId( onclick.c_str() ) );
        }
    }
    return menu;
}

unsigned 
XmlApplication::mapId( const std::_tstring& event )
{
    if ( event.empty() ) {
        return 0;
    }
    unsigned id = _nextId++;
    _eventIdMap[id] = event;
    return id;
}

void
XmlApplication::onCommandId( unsigned id )
{
    IntStringMap::iterator iter = _eventIdMap.find( id );
    if ( iter != _eventIdMap.end() ) {
        sendEvent( iter->second );
    }
}

void
XmlApplication::sendEvent( const _tstring& event )
{
    if ( _listener ) {
        _listener->onXmlAppEvent( event );
    }
}

void
XmlApplication::error( CTag* tag, const _TCHAR* msg )
{
    MessageBox( NULL, msg, _T("XmlApplication"), MB_OK );
}

WindowFunctions*
XmlApplication::parseWindow( HWND hParent, CTag* tag )
{
    _tstring type = tag->attributes[_T("type")];

    if ( type.empty() ) {
        error( tag, _T("Error: Window must have type."));
        return NULL;
    }

    WindowFunctions* window = _factory->createWindow( _hInstance, hParent, type.c_str() );
    _windows.push_back( window );
    return window;
}

/*
WindowFunctions* 
XmlApplication::parsePane( HWND hParent, CTag* tag )
{
    CTag::List& tags = tag->ChildList;
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        CTag* tag = *iter;
        if ( tag->name == _T("pane") ) {
            WindowFunctions* pane = parseWindow( hParent, tag );
            return pane;
        } else if ( tag->name == _T("splitter") ) {
            WindowFunctions* pane = parseSplitter( hParent, tag );
            return pane;
        }
    }
    return NULL;
}
*/

WindowFunctions*
XmlApplication::parseSplitter( HWND hParent, CTag* tag )
{
    SplitterWindow* splitWindow = new SplitterWindow(_hInstance);
    _windows.push_back( splitWindow );
    int pane1Size = _tstoi( tag->attributes[_T("pane1")].c_str() ); 
    int pane2Size = _tstoi( tag->attributes[_T("pane2")].c_str() ); 
    double gravity = _tstof( tag->attributes[_T("gravity")].c_str() );
    bool vertical = !tag->attributes[_T("vertical")].empty();
    WindowFunctions* panes[2];

    panes[0] = panes[1] = NULL;

    splitWindow->create( hParent );

    CTag::List& tags = tag->ChildList;
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        CTag* tag = *iter;
        WindowFunctions* pane = NULL;
        if ( tag->name == _T("splitter") ) {
            pane = parseSplitter( splitWindow->hwnd, tag );
        } else if ( tag->name == _T("window") ) {
            pane = parseWindow( splitWindow->hwnd, tag );
        }

        if ( pane ) {
            if ( !panes[0] ) {
                panes[0] = pane;
            } else if ( !panes[1] ) {
                panes[1] = pane;
            }
        }
    }

    splitWindow->split( panes[0]->hwnd, panes[1]->hwnd, vertical );
    splitWindow->setPaneSizes( pane1Size, pane2Size, gravity );

    return splitWindow;
}

void
XmlApplication::parseFrameWindow( CTag* tag )
{
    _tstring classStr = tag->attributes["class"];
    _tstring sizeStr = tag->attributes["size"];
    _tstring titleStr = tag->attributes["title"];
    _tstring nameStr = tag->attributes["name"];
    _tstring iconStr = tag->attributes["icon"];
    int x = CW_USEDEFAULT;
    int y = CW_USEDEFAULT;
    int width = CW_USEDEFAULT;
    int height = CW_USEDEFAULT;
    parseSize( sizeStr.c_str(), &width, &height );

    DefaultXmlFrameWindow* window = new DefaultXmlFrameWindow( _hInstance, this, iconStr.c_str() );
    window->onCloseEvent = tag->attributes[_T("onclose")];
    window->Create( x, y, width, height, NULL, NULL, _hInstance );
    window->setWindowText( titleStr.c_str() );

    _windows.push_back( window );

    CTag::List& tags = tag->ChildList;
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        CTag* tag = *iter;
        if ( tag->name == _T("menu") ) {
            Menu* menu = parseMenu( tag, false );
            SetMenu( window->hwnd, menu->hMenu );
        } else if ( tag->name == _T("buttonbar") ) {
            ButtonBar* buttonBar = parseButtonBar( window->hwnd, tag );
            delete window->buttonBar;
            window->buttonBar = buttonBar;
            buttonBar->setCommandTarget( window->hwnd );
        } else if ( tag->name == _T("window") ) {
            WindowFunctions* client = parseWindow( window->hwnd, tag );
            window->client = client;
        } else if ( tag->name == _T("splitter") ) {
            WindowFunctions* client = parseSplitter( window->hwnd, tag );
            window->client = client;
        }
    }

    window->reformat();
}

ButtonBar* 
XmlApplication::parseButtonBar( HWND hParent, CTag* tag )
{
    int width = _tstoi( tag->attributes["width"].c_str() ); 
    int height = _tstoi( tag->attributes["height"].c_str() ); 
    unsigned bgcolour = RGB(0x80,0x80,0x80);
    parseColour( tag->attributes["bgcolour"].c_str(), &bgcolour );
    bool horizontal = true;
    _tstring backImage = tag->attributes["backimage"];
    bool backImageRightAlign = tag->attributes["backimagealign"] == _T("right");

    if ( height <= 0 && width <= 0 ) {
        height = 32;
        width = 33;
    } else if ( height <= 0 ) {
        height = width + 1;
    } else {
        width = height + 1;
    }

    ButtonBar* buttonBar = new ButtonBar( _hInstance );
    buttonBar->create( hParent );
    buttonBar->setBackColour( bgcolour );
    if ( !backImage.empty() ) {
        buttonBar->setBackImage( backImage.c_str(), backImageRightAlign);
    }

    CTag::List& tags = tag->ChildList;
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        CTag* tag = *iter;
        if ( tag->name == _T("button") ) {
            _tstring out = tag->attributes["out"];
            _tstring over = tag->attributes["over"];
            _tstring down = tag->attributes["down"];
            _tstring onclick = tag->attributes["onclick"];

            unsigned id = mapId(onclick);
            buttonBar->addButton( id, out.c_str(), over.c_str(), down.c_str(),
                NULL );
        } else if ( tag->name == _T("window") ) {
            WindowFunctions* window = parseWindow( buttonBar->hwnd, tag );
            buttonBar->addButton( window->hwnd );
        }
    }

    buttonBar->sizeWindow( width, height );
    return buttonBar;
}


bool 
XmlApplication::loadFromString( const std::_tstring& text )
{
    //clear();

    DTD_TABLE_ENTRY dtd[] = {
        { "application", "framewindow", 0 },
        { "framewindow", "menu", 0 },
        { "menu", "menubar", 0 },
        { "menu", "menu", 0 },
        { "framewindow", "buttonbar", 0 },
        { "buttonbar", "button", 0 },
        { "buttonbar", "window", 0 },
        { "framewindow", "window", 0 },
        { "framewindow", "splitter", 0 },
        { "splitter", "splitter", 0 },
        { "splitter", "window", 0 }
    };
    
    CTagParser parser;
    parser.SetTagRelationships( dtd, sizeof(dtd)/sizeof(*dtd) );
    parser.ParseText( text.c_str() );

    CTag::List& tags = parser.GetTagList();
    CTag::List::iterator iter = tags.begin();
    for( ; iter != tags.end(); ++iter ) {
        parseTag(*iter);
    }

    return true;
}

bool
XmlApplication::loadFromFile( const std::_tstring& filename )
{
    FILE* file = fopen( filename.c_str(), "rt" );

    if ( file == 0 ) {
        return false;
    }
    
    _tstring text;
    _TCHAR buffer[1024];

    while( !feof( file ) ) {
        int read = fread( buffer, sizeof(*buffer), sizeof(buffer)/sizeof(*buffer), file );
        text.append( buffer, read );
    }

    fclose( file );

    return loadFromString( text );
}
