#include "win95.h"
#include <shlobj.h>
#include <limits.h>

static DWORD WINAPI
Probe_GetLongPathName(LPCTSTR ptszShort, LPTSTR ptszLong, DWORD ctchBuf);

static DWORD (CALLBACK *pGetLongPathName)(LPCTSTR, LPTSTR, DWORD) 
    = Probe_GetLongPathName;

BOOL WINAPI
SafeGetLongPathName(LPCTSTR ptszShort, LPTSTR ptszLong, DWORD ctchBuf)
{
    return pGetLongPathName(ptszShort, ptszLong, ctchBuf);
}

static DWORD WINAPI
Emulate_GetLongPathName(LPCTSTR ptszShort, LPTSTR ptszLong, DWORD ctchBuf)
{
    LPSHELLFOLDER psfDesk;
    HRESULT hr;
    LPITEMIDLIST pidl;
    TCHAR tsz[MAX_PATH];            /* Scratch TCHAR buffer */
    DWORD dwRc;
    LPMALLOC pMalloc;

    /*
     *  The file had better exist.  GetFileAttributes() will
     *  not only tell us, but it'll even call SetLastError()
     *  for us.
     */
    if (GetFileAttributes(ptszShort) == 0xFFFFFFFF) {
        return 0;
    }

    /*
     *  First convert from relative path to absolute path.
     *  This uses the scratch TCHAR buffer.
     */
    dwRc = GetFullPathName(ptszShort, MAX_PATH, tsz, NULL);
    if (dwRc == 0) {
        /*
         *  Failed; GFPN already did SetLastError().
         */
    } else if (dwRc >= MAX_PATH) {
        /*
         *  Resulting path would be too long.
         */
        SetLastError(ERROR_BUFFER_OVERFLOW);
        dwRc = 0;
    } else {
        /*
         *  Just right.
         */
        hr = SHGetDesktopFolder(&psfDesk);
        if (SUCCEEDED(hr)) {
            ULONG cwchEaten;

#ifdef UNICODE
#ifdef __cplusplus
            hr = psfDesk->ParseDisplayName(NULL, NULL, tsz,
                                       &cwchEaten, &pidl, NULL);
#else
            hr = psfDesk->lpVtbl->ParseDisplayName(psfDesk, NULL, NULL, tsz,
                                       &cwchEaten, &pidl, NULL);
#endif
#else
            WCHAR wsz[MAX_PATH];        /* Scratch WCHAR buffer */

            /*
             *  ParseDisplayName requires UNICODE, so we use
             *  the scratch WCHAR buffer during the conversion.
             */
            dwRc = MultiByteToWideChar(
                        AreFileApisANSI() ? CP_ACP : CP_OEMCP,
                        0, tsz, -1, wsz, MAX_PATH);
            if (dwRc == 0) {
                /*
                 *  Couldn't convert to UNICODE.  MB2WC uses
                 *  ERROR_INSUFFICIENT_BUFFER, which we convert
                 *  to ERROR_BUFFER_OVERFLOW.  Any other error
                 *  we leave alone.
                 */
                if (GetLastError() == ERROR_INSUFFICIENT_BUFFER) {
                    SetLastError(ERROR_BUFFER_OVERFLOW);
                }
                dwRc = 0;
            } else {
#ifdef __cplusplus
                hr = psfDesk->ParseDisplayName(NULL, NULL, wsz,
                                           &cwchEaten, &pidl, NULL);
#else
                hr = psfDesk->lpVtbl->ParseDisplayName(psfDesk, NULL, NULL,
                                           wsz, &cwchEaten, &pidl, NULL);
#endif
#endif

                if (FAILED(hr)) {
                    /*
                     *  Weird.  Convert the result back to a Win32
                     *  error code if we can.  Otherwise, use the
                     *  generic "duh" error code ERROR_INVALID_DATA.
                     */
                    if (HRESULT_FACILITY(hr) == FACILITY_WIN32) {
                        SetLastError(HRESULT_CODE(hr));
                    } else {
                        SetLastError(ERROR_INVALID_DATA);
                    }
                    dwRc = 0;
                } else {
                    /*
                     *  Convert the pidl back to a filename in the
                     *  TCHAR scratch buffer.
                     */
                    dwRc = SHGetPathFromIDList(pidl, tsz);
                    if (dwRc == 0 && tsz[0]) {
                        /*
                         *  Bizarre failure.
                         */
                        SetLastError(ERROR_INVALID_DATA);
                    } else {
                        /*
                         *  Copy the result back to the user's buffer.
                         */
                        dwRc = lstrlen(tsz);
                        if (dwRc + 1 > ctchBuf) {
                            /*
                             *  On buffer overflow, return necessary
                             *  size including terminating null (+1).
                             */
                            SetLastError(ERROR_INSUFFICIENT_BUFFER);
                            dwRc = dwRc + 1;
                        } else {
                            /*
                             *  On buffer okay, return actual size not
                             *  including terminating null.
                             */
                            lstrcpyn(ptszLong, tsz, ctchBuf);
                        }
                    }

                    /*
                     *  Free the pidl.
                     */
                    if (SUCCEEDED(SHGetMalloc(&pMalloc))) {
#ifdef __cplusplus
                        pMalloc->Free(pidl);
                        pMalloc->Release();
#else
                        pMalloc->lpVtbl->Free(pMalloc, pidl);
                        pMalloc->lpVtbl->Release(pMalloc);
#endif
                    }
                }
#ifndef UNICODE
            }
#endif
            /*
             *  Release the desktop folder now that we no longer
             *  need it.
             */
#ifdef __cplusplus
            psfDesk->Release();
#else
            psfDesk->lpVtbl->Release(psfDesk);
#endif
        }
    }
    return dwRc;
}

/*
 * The stub that probes to decide which version to use.
 */
static DWORD WINAPI
Probe_GetLongPathName(LPCTSTR ptszShort, LPTSTR ptszLong, DWORD ctchBuf)
{
    HINSTANCE hinst;
    FARPROC fp;
    DWORD dwRc;
    DWORD (CALLBACK *RealGetLongPathName)(LPCTSTR, LPTSTR, DWORD);

    hinst = GetModuleHandle(TEXT("KERNEL32"));
#ifdef UNICODE
    fp = GetProcAddress(hinst, "GetLongPathNameW");
#else
    fp = GetProcAddress(hinst, "GetLongPathNameA");
#endif

    if (fp) {
        *(FARPROC *)&RealGetLongPathName = fp;
        dwRc = RealGetLongPathName(ptszShort, ptszLong, ctchBuf);
        if (dwRc || GetLastError() != ERROR_CALL_NOT_IMPLEMENTED) {
            pGetLongPathName = RealGetLongPathName;
        } else {
            pGetLongPathName = Emulate_GetLongPathName;
            dwRc = pGetLongPathName(ptszShort, ptszLong, ctchBuf);
        }
    } else {
        pGetLongPathName = Emulate_GetLongPathName;
        dwRc = pGetLongPathName(ptszShort, ptszLong, ctchBuf);
    }

    return dwRc;

}


/******************************************************************************
    SafeGradientFill()
  ****************************************************************************/

static BOOL WINAPI Probe_GradientFill( 
        IN HDC hdc, 
        IN PTRIVERTEX pVertex, 
        IN ULONG NumVertex, 
        IN PVOID pMesh, 
        IN ULONG dwNumMesh, 
        IN ULONG dwMode);

typedef BOOL (WINAPI *GradientFillFunction)
    (IN HDC, IN PTRIVERTEX, IN ULONG, IN PVOID, IN ULONG, IN ULONG);
static GradientFillFunction pGradientFill = Probe_GradientFill;

BOOL SafeGradientFill( 
        IN HDC hdc, 
        IN PTRIVERTEX pVertex, 
        IN ULONG NumVertex, 
        IN PVOID pMesh, 
        IN ULONG dwNumMesh, 
        IN ULONG dwMode)
{
    return pGradientFill( hdc, pVertex, NumVertex, pMesh, dwNumMesh, dwMode );
}

static BOOL WINAPI Emulate_GradientFill( 
        IN HDC hdc, 
        IN PTRIVERTEX pVertex, 
        IN ULONG NumVertex, 
        IN PVOID pMesh, 
        IN ULONG dwNumMesh, 
        IN ULONG dwMode)
{
    // average together all the verticies and just do a rectangle over the
    // bounds.
    
    if ( NumVertex == 0 ) {
        return TRUE;
    }
   
    int left = INT_MAX;
    int right = INT_MIN;
    int top = INT_MAX;
    int bottom = INT_MIN;

    int Red = 0;
    int Green = 0;
    int Blue = 0;

    for( unsigned i = 0; i < NumVertex; i++ ) {
        if ( pVertex[i].x < left ) {
            left = pVertex[i].x;
        }

        if ( pVertex[i].x > right ) {
            right = pVertex[i].x;
        }

        if ( pVertex[i].y < top ) {
            top = pVertex[i].y;
        }

        if ( pVertex[i].y > bottom ) {
            bottom = pVertex[i].y;
        }

        Red += (int)((double)pVertex[i].Red / 0xff00 * 0xff);
        Green += (int)((double)pVertex[i].Green / 0xff00 * 0xff);
        Blue += (int)((double)pVertex[i].Blue / 0xff00 * 0xff);
    }

    Red = (int)((double)Red / NumVertex);
    Green = (int)((double)Green / NumVertex);
    Blue = (int)((double)Blue / NumVertex);

    RECT rect;
    rect.left = left;
    rect.right = right;
    rect.bottom = bottom;
    rect.top = top;

    HBRUSH brush = CreateSolidBrush( RGB(Red, Green, Blue) );
    FillRect( hdc, &rect, brush );
    DeleteObject( brush );

    return TRUE;
}

static BOOL WINAPI Probe_GradientFill( 
        IN HDC hdc, 
        IN PTRIVERTEX pVertex, 
        IN ULONG NumVertex, 
        IN PVOID pMesh, 
        IN ULONG dwNumMesh, 
        IN ULONG dwMode)
{
    HMODULE hMsimg = LoadLibrary(TEXT("MSIMG32.DLL"));
    if ( hMsimg ) {
        pGradientFill = (GradientFillFunction)GetProcAddress(hMsimg,
            "GradientFill");
    } else {
        pGradientFill = Emulate_GradientFill;
    }

    return pGradientFill( hdc, pVertex, NumVertex, pMesh, dwNumMesh, dwMode );
}

static BOOL WINAPI Probe_AlphaBlend(
  HDC hdcDest,                 // handle to destination DC
  int nXOriginDest,            // x-coord of upper-left corner
  int nYOriginDest,            // y-coord of upper-left corner
  int nWidthDest,              // destination width
  int nHeightDest,             // destination height
  HDC hdcSrc,                  // handle to source DC
  int nXOriginSrc,             // x-coord of upper-left corner
  int nYOriginSrc,             // y-coord of upper-left corner
  int nWidthSrc,               // source width
  int nHeightSrc,              // source height
  BLENDFUNCTION blendFunction  // alpha-blending function
);

typedef BOOL (WINAPI* AlphaBlendFunction)(
  HDC hdcDest,                 // handle to destination DC
  int nXOriginDest,            // x-coord of upper-left corner
  int nYOriginDest,            // y-coord of upper-left corner
  int nWidthDest,              // destination width
  int nHeightDest,             // destination height
  HDC hdcSrc,                  // handle to source DC
  int nXOriginSrc,             // x-coord of upper-left corner
  int nYOriginSrc,             // y-coord of upper-left corner
  int nWidthSrc,               // source width
  int nHeightSrc,              // source height
  BLENDFUNCTION blendFunction  // alpha-blending function
);

static AlphaBlendFunction pAlphaBlend = Probe_AlphaBlend;

BOOL SafeAlphaBlend( 
  HDC hdcDest,                 // handle to destination DC
  int nXOriginDest,            // x-coord of upper-left corner
  int nYOriginDest,            // y-coord of upper-left corner
  int nWidthDest,              // destination width
  int nHeightDest,             // destination height
  HDC hdcSrc,                  // handle to source DC
  int nXOriginSrc,             // x-coord of upper-left corner
  int nYOriginSrc,             // y-coord of upper-left corner
  int nWidthSrc,               // source width
  int nHeightSrc,              // source height
  BLENDFUNCTION blendFunction  // alpha-blending function
  )
{
    return pAlphaBlend( hdcDest, nXOriginDest, nYOriginDest, nWidthDest, nHeightDest, hdcSrc,
        nXOriginSrc, nYOriginSrc, nWidthSrc, nHeightSrc, blendFunction);
}

static BOOL WINAPI Emulate_AlphaBlend( 
  HDC hdcDest,                 // handle to destination DC
  int nXOriginDest,            // x-coord of upper-left corner
  int nYOriginDest,            // y-coord of upper-left corner
  int nWidthDest,              // destination width
  int nHeightDest,             // destination height
  HDC hdcSrc,                  // handle to source DC
  int nXOriginSrc,             // x-coord of upper-left corner
  int nYOriginSrc,             // y-coord of upper-left corner
  int nWidthSrc,               // source width
  int nHeightSrc,              // source height
  BLENDFUNCTION blendFunction  // alpha-blending function
  )
{
    return FALSE;
}

static BOOL WINAPI Probe_AlphaBlend( 
  HDC hdcDest,                 // handle to destination DC
  int nXOriginDest,            // x-coord of upper-left corner
  int nYOriginDest,            // y-coord of upper-left corner
  int nWidthDest,              // destination width
  int nHeightDest,             // destination height
  HDC hdcSrc,                  // handle to source DC
  int nXOriginSrc,             // x-coord of upper-left corner
  int nYOriginSrc,             // y-coord of upper-left corner
  int nWidthSrc,               // source width
  int nHeightSrc,              // source height
  BLENDFUNCTION blendFunction  // alpha-blending function
  )
{
    HMODULE hMsimg = LoadLibrary(TEXT("MSIMG32.DLL"));
    if ( hMsimg ) {
        pAlphaBlend = (AlphaBlendFunction)GetProcAddress(hMsimg,
            "AlphaBlend");
    } else {
        pAlphaBlend = Emulate_AlphaBlend;
    }

    return pAlphaBlend( hdcDest, nXOriginDest, nYOriginDest, nWidthDest, nHeightDest, hdcSrc,
        nXOriginSrc, nYOriginSrc, nWidthSrc, nHeightSrc, blendFunction);
}
