// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include <windows.h>
#include <windowsx.h>
#include "DrawTool.h"
#include "ScalogramViewWindow.h"
#include "WaveletDoc.h"
#include "dbg.h" // must be last

DragHandler::DragHandler()
{
    hwnd = 0;
    _dragStart.x = 0;
    _dragStart.y = 0;
    _dragStop.x = 0;
    _dragStop.y = 0;
	_dragging = false;
}

DragHandler::~DragHandler()
{

}
 
/**
 * The window containing the drawing tool should call this to process its mouse
 * messages. Of course, it is free to do subsequent processing of the messages.
 */
LRESULT
DragHandler::processMessage(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    this->hwnd = hwnd;
    switch(msg) {
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONDBLCLK, onLButtonDown);
        HANDLE_MSG(hwnd, WM_RBUTTONDOWN, onRButtonDown);
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_KEYUP, onKey);

        case WM_LBUTTONUP: {
            POINT pt;
            pt.x = GET_X_LPARAM(lParam);
            pt.y = GET_Y_LPARAM(lParam);
            if ( _dragging ) {
                _dragging = false;
                _dragStop = pt;
                onDragStop( pt.x, pt.y );
            }
            break;
        }

        case WM_MOUSEMOVE:
            POINT pt;
            pt.x = GET_X_LPARAM(lParam);
            pt.y = GET_Y_LPARAM(lParam);
            if ( (wParam & MK_LBUTTON) == 0 && _dragging ) {
                _dragging = 0;
                onDragAborted();
            } else {
                onMouseMove( hwnd, pt.x, pt.y, wParam );
            }
            break;
    }

    return 0;
}

void 
DragHandler::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{

}

void 
DragHandler::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags)
{
    POINT pt;
    pt.x = x;
    pt.y = y;
    if ( !_dragging ) {
        _dragStart = pt;
        onDragStart( pt.x, pt.y );
        _dragging = true;
    }
}

void 
DragHandler::onRButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags)
{
}

bool 
DragHandler::dragging()
{
    return _dragging;
}

POINT 
DragHandler::getDragStartPoint()
{
    return _dragStart;
}

POINT 
DragHandler::getDragReleasePoint()
{
    return _dragStop;
}


void 
DragHandler::onDragStart( int x, int y )
{

}

void 
DragHandler::onDragStop( int x, int y )
{

}

void
DragHandler::onDragAborted()
{

}

void 
DragHandler::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{

}

DrawTool::DrawTool( ScalogramViewWindow* drawWindow ) :
    _drawWindow(drawWindow)
{
    
}

DrawTool::~DrawTool()
{

}

bool
DrawTool::active()
{
    return false;
}

void 
DrawTool::onDragAborted()
{
}

void 
DrawTool::onPaint(DeviceContext dc)
{
}

DrawBoxTool::DrawBoxTool( ScalogramViewWindow* drawWindow ) :
    DrawTool(drawWindow),
    _drawn(false)
{
    
}

DrawBoxTool::~DrawBoxTool()
{

}

void 
DrawBoxTool::onDragStart( int x, int y )
{
}

void
DrawBoxTool::onDragStop( int x, int y )
{
    POINT start = _dragStart;
    POINT end = _dragStop;
    RECT rect = {start.x, start.y, end.x, end.y };
    ValidateRect( rect );
    HDC dc = GetDC(hwnd);
    drawRect( dc, false );
    ReleaseDC(hwnd, dc);
    _drawWindow->onDrawRect( &rect );
}

void 
DrawBoxTool::onDragAborted()
{
    HDC dc = GetDC(hwnd);
    drawRect( dc, false );
    ReleaseDC( hwnd, dc );
}

void 
DrawBoxTool::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( !active() ) {
        return;
    }

    POINT start = _dragStart;
    RECT rect = {start.x, start.y, x, y};
    ValidateRect( rect );
    HDC dc = GetDC(hwnd);
    drawRect( dc, false );
    _rect = rect;
    drawRect( dc, true );
    ReleaseDC(hwnd, dc);
}

void 
DrawBoxTool::drawRect( DeviceContext dc, bool draw )
{
    if ( draw ^ _drawn ) {
        int mode = SetROP2( dc, R2_NOT );
        Pen pen(PS_SOLID, 4, RGB(64, 64, 128 ) );
        dc.selectPen( pen );
        dc.selectBrush( (HBRUSH)GetStockObject(NULL_BRUSH) );
        Rectangle( dc, _rect.left, _rect.top, _rect.right, _rect.bottom );

        SetROP2( dc, mode );

        _drawn = draw;
    }
}

void 
DrawBoxTool::onPaint(DeviceContext dc)
{
    if ( !active() ) {
        return;
    }

    if ( _drawn ) {
        _drawn = false;
        drawRect( dc, true );
    }
}

bool
DrawBoxTool::active()
{
    return dragging();
}


DrawSelectionTool::DrawSelectionTool( ScalogramViewWindow* drawWindow ) :
    DrawTool(drawWindow),
    _drawn(false)
{
    
}

DrawSelectionTool::~DrawSelectionTool()
{

}

void 
DrawSelectionTool::onDragStart( int x, int y )
{
}

void
DrawSelectionTool::onDragStop( int x, int y )
{
    POINT start = _dragStart;
    POINT end = _dragStop;
    RECT rect = {start.x, start.y, end.x, end.y };
    ValidateRect( rect );
    HDC dc = GetDC(hwnd);
    drawRect( dc, false );
    ReleaseDC(hwnd, dc);
    _drawWindow->onDrawRect( &rect );
}

void 
DrawSelectionTool::onDragAborted()
{
    HDC dc = GetDC(hwnd);
    drawRect( dc, false );
    ReleaseDC( hwnd, dc );
}

void 
DrawSelectionTool::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( !active() ) {
        return;
    }

    POINT start = _dragStart;
    RECT rect = {start.x, start.y, x, y};
    ValidateRect( rect );
    HDC dc = GetDC(hwnd);
    drawRect( dc, false );
    _rect = rect;
    drawRect( dc, true );
    ReleaseDC(hwnd, dc);
}

void 
DrawSelectionTool::drawRect( DeviceContext dc, bool draw )
{
    if ( draw ^ _drawn ) {
        int mode = SetROP2( dc, R2_NOT );
        Pen pen(PS_DASH, 1, 0 );
        dc.selectPen( pen );
        dc.selectBrush( (HBRUSH)GetStockObject(NULL_BRUSH) );
        Rectangle( dc, _rect.left, _rect.top, _rect.right, _rect.bottom );

        SetROP2( dc, mode );

        _drawn = draw;
    }
}

void 
DrawSelectionTool::onPaint(DeviceContext dc)
{
    if ( !active() ) {
        return;
    }

    if ( _drawn ) {
        _drawn = false;
        drawRect( dc, true );
    }
}

bool
DrawSelectionTool::active()
{
    return dragging();
}
