// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include <assert.h>
#include <windowsx.h>
#include "MainViewWindow.h"
#include "resource.h"
#include "MainWindow.h"
#include "WaveletOptions.h"
#include "Waveform.h"
#include "WaveletApplication.h"

#include "dbg.h" // must be last

MainViewWindow::MainViewWindow(HINSTANCE hInstance) :
    SplitWindow(hInstance),
    _waveView(hInstance),
    _waveletView(hInstance),
    _toolbar( hInstance ),
    _cards( hInstance )
{
	_pszClassName = TEXT("MainViewWindow");
	_pszTitle = TEXT("MainView Window");

    _dwStyle |= WS_CHILD | WS_VISIBLE; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = NULL;
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;

    _doc = NULL;

    _main = 0;

    _dispatcher->registerEvent( this, IDC_OPEN_CNF );
    _dispatcher->registerEvent( this, IDC_OPEN_FAILED_CNF );
    _dispatcher->registerEvent( this, IDC_PLAY_REQ );
    _dispatcher->registerEvent( this, IDC_STOP_REQ );
    _dispatcher->registerEvent( this, IDC_PLAYTOOL_REQ );
    _dispatcher->registerEvent( this, IDC_SELTOOL_REQ );
}


MainViewWindow::~MainViewWindow()
{
}

bool
MainViewWindow::hasDoc()
{
    return _doc;
}

void 
MainViewWindow::onCardEvent( const std::string& event )
{
    // This takes an event specified in the sidebar.xml file, and
    // dispatches it to the rest of the system.
    if ( event == _T("open") ) {
        _dispatcher->dispatch( IDC_OPEN_REQ );
    } else if ( event == _T("webpage") ) {
        _dispatcher->dispatch( IDC_WEBPAGE_REQ );
    } else if ( event == _T("8000") ) {
        resample(8000);
    } else if ( event == _T("11025") ) {
        resample(11025);
    } else if ( event == _T("22050") ) {
        resample(22050);
    } else if ( event == _T("noresample") ) {
        resample(_waveform->sampleRate);
    }
}

void
MainViewWindow::resample( unsigned sampleRate )
{
    _waveform->resample( sampleRate );
    continueOpen();
}

void
MainViewWindow::continueOpen()
{
    unsigned mb = SoundImage::calcMemorySizeMB( _waveform->sampleRate,
            _waveform->size);

    SoundImage* image = NULL;

    if ( mb > 1024 ) {
alloc_fail:
        TCHAR buffer[1024];
        TCHAR dbuffer[MAX_PATH];
        _stprintf(buffer, _T("The file will require %d MB of disk space to compute. Do you want to continue?"), mb);
        _stprintf(dbuffer, _T("%s.wavelet"), _waveform->name.c_str() );
        
        // more than XXX megabytes required.
        if ( IDOK == MessageBox(hwnd, buffer, _application->name(), MB_OKCANCEL ) ) {
            // create disk sound image.       
            DiskSoundImage* dimage = new DiskSoundImage();
            if ( dimage->create( dbuffer ) && dimage->alloc( _waveform->size,
                        _waveform->sampleRate ) ) {
                image = dimage;
            } else {
                delete dimage;
            }
        } else {
            _cards.showCard("welcome");
            return;
        }
    } else {
        // create memory sound image.
        image = new MemorySoundImage();
        if ( !image->alloc( _waveform->size, _waveform->sampleRate ) ) {
            delete image;
            image = NULL;
            goto alloc_fail;
        }
    }

    if ( image ) {
        _doc = new WaveletDoc( _waveform, image );
        _waveform = NULL;
        _waveView.setDoc( _doc );
        _waveletView.setDoc( _doc );
        _dispatcher->dispatch( IDC_OPEN_CNF );
    } else {
        _dispatcher->dispatch( IDC_OPEN_FAILED_CNF );
    }

    printf("CAlculated %d MB\n", mb);

}


void
MainViewWindow::onEvent( unsigned event )
{
    switch( event ) {
        case IDC_OPEN_CNF:
            // user wanted to open a file.
            _dispatcher->dispatch( IDC_PLAYTOOL_REQ );
            break;
        case IDC_OPEN_FAILED_CNF:
            // opening a file failed.
            _cards.showCard( "openfailed" );
            break;
        case IDC_PLAYTOOL_REQ:
            // User clicked play tool.
            if ( _doc ) {
                if ( !_options->release ) {
                    _toolbar.enable( IDC_PLAYTOOL_REQ, false );
                    _toolbar.enable( IDC_SELTOOL_REQ, true );
                }
                _cards.showCard( "opened" );
            }
            break;
        case IDC_SELTOOL_REQ:    
            if ( _doc ) {
                if ( !_options->release ) {
                    _toolbar.enable( IDC_PLAYTOOL_REQ, true );
                    _toolbar.enable( IDC_SELTOOL_REQ, false );
                }
                _cards.showCard( "seltool" );
            }
            break;
    }
}

void
MainViewWindow::create(HWND hParent, MainWindow* main)
{
    _main = main;
    _hParent = hParent;

    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                CW_USEDEFAULT, CW_USEDEFAULT, _hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

LRESULT 
MainViewWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        //HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        //HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        //HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        //HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        //HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        //HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        //HANDLE_MSG(hwnd, WM_SIZE, onSize);
    }
    
    *pbProcessed = FALSE;
    return SplitWindow::WindowProc(hwnd,msg,wParam,lParam,pbProcessed);
}

void 
MainViewWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    _dispatcher->dispatch( id );
}

BOOL 
MainViewWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    // Create the toolbar, help window, the scalogram viewer, the waveform
    // viewer.
    _waveView.create(hwnd);
    _waveletView.create(hwnd);
    _toolbar.create(hwnd);
    _toolbar.setBackColour( RGB(220,220,220 ));
    _cards.create(hwnd);
    _cards.setListener( this );
    _cards.loadFromFile( _T("sidebar.xml") );

    _toolbar.addButton( IDC_PLAY_REQ, IDB_PLAY_OUT, IDB_PLAY_OVER, IDB_PLAY_DOWN, IDB_PLAY_OUT );
    _toolbar.addButton( IDC_STOP_REQ, IDB_STOP_OUT, IDB_STOP_OVER, IDB_STOP_DOWN, IDB_STOP_OUT );

    if ( !_options->release ) {
        _toolbar.addButton( IDC_PLAYTOOL_REQ, IDB_PLAYTOOL_OUT, IDB_PLAYTOOL_OVER,
                IDB_PLAYTOOL_DOWN, IDB_PLAYTOOL_DOWN );
        _toolbar.addButton( IDC_SELTOOL_REQ, IDB_SEL_OUT, IDB_SEL_OVER,
                IDB_SEL_DOWN, IDB_SEL_DOWN );
    }

    _waveView.setScalogramView( &_waveletView );

    // Set up splitters. The splitters that I wrote don't work too well, so
    // don't allow them to move.

    /***********************************************************
      sp1 || sp2 |       sp3 (vertical)
      ********************************************************/
    RECT needed;

    Splitter* sp3 = new Splitter( true );
    sp3->setPane( false, _waveView.hwnd );
    sp3->setPane( true, _waveletView.hwnd );
    sp3->setSplitInfo( _waveView.hwnd, -1, 100, -1, 100 );
    sp3->setSize( _waveletView.hwnd, 200, 200, &needed );


    Splitter* sp2 = new Splitter( false );
    sp2->setPane( true, _cards.hwnd );
    sp2->setSplitInfo( _cards.hwnd, 150, -1, 150, -1 );
    sp3->setSize( _cards.hwnd, 150, 150, &needed );
    sp2->setPane( false, sp3 );
    sp2->setThickness( 2 );

    Splitter* sp1 = new Splitter( false );
    sp1->setPane( true, _toolbar.hwnd );
    sp1->setPane( false, sp2 );
    sp1->setSplitInfo( _toolbar.hwnd, 32, 32, 32, 32 );
    sp1->setThickness( 0 );
    setMainSplitter( sp1 );

    return TRUE;
}

bool
MainViewWindow::open( const char* filename )
{
    /*
    WaveletDoc* doc = WaveletDoc::open( filename );
    if ( doc ) {
        onStop();
        _doc = doc;
        _doc->wave->resample(8000);
        _waveView.setDoc( _doc );
        _waveletView.setDoc( _doc );
        _dispatcher->dispatch( IDC_OPEN_CNF );
        return true;
    } else {
        _dispatcher->dispatch( IDC_OPEN_FAILED_CNF );
    }
    */
    _waveform = Waveform::load(filename);
    if ( _waveform ) {
        if ( _waveform->sampleRate > 8000 ) {
            _cards.showCard( "ask_resample" );
            _cards.show( "11025", _waveform->sampleRate > 11025 ); 
            _cards.show( "22050", _waveform->sampleRate > 22050 ); 
            _cards.format();
        } else {
            continueOpen();
        }
    } else {
        _dispatcher->dispatch( IDC_OPEN_FAILED_CNF );
    }


    return false;
}

void 
MainViewWindow::onViewScalogram()
{

}

void 
MainViewWindow::onViewInverse()
{

}

