// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include <assert.h>
#include <windowsx.h>
#include "WaveletApplication.h"
#include "MainWindow.h"
#include "ColourMap.h"
#include "WaveletOptions.h"
#include "resource.h"
#include "dbg.h" // must be last

// These represent the items that can be in the menu.
enum Commands {
    IDC_LINEAR,
    IDC_LOGARITHMIC,
    IDC_LOGARITHMIC_REAL,
    IDC_LOGARITHMIC_COMPLEX,
    IDC_PHASE,
    IDC_AUTO_ENHANCE,

    IDC_FIRST_COLOUR,

    IDC_LAST_COLOUR = IDC_FIRST_COLOUR+10
};


MainWindow::MainWindow(HINSTANCE hInstance) :
    _mainMenu(false),
    _mainView(hInstance)
{
    // Boilerplate window creation stuff.
	_hInstance = hInstance;
	_pszClassName = TEXT("MainWindow");
	_pszTitle = _application->fullName();

    _dwStyle |= WS_THICKFRAME | WS_SYSMENU | WS_CLIPCHILDREN |
        WS_VISIBLE | WS_CAPTION | WS_MAXIMIZEBOX | WS_MINIMIZEBOX | 
        WS_CLIPSIBLINGS;
 
    _WndClass.hbrBackground = (HBRUSH)BLACK_BRUSH;
    _WndClass.hCursor = (HCURSOR)LoadCursor(NULL, IDC_ARROW);
	//_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _WndClass.hIcon = LoadIcon( hInstance, MAKEINTRESOURCE(IDI_ICON) );
    _dwExtendedStyle |= 0;


    // Create the menu bar.
    _mainMenu.append( TEXT("&File"), _fileMenu );
    _mainMenu.append( TEXT("&Eile"), _editMenu );
    _mainMenu.append( TEXT("&View"), _viewMenu );
    _mainMenu.append( TEXT("&Help"), _helpMenu );

    _fileMenu.append( TEXT("&Open..."), IDC_OPEN_REQ );
    _fileMenu.appendBar();
    _fileMenu.append( TEXT("&Play"), IDC_PLAY_REQ );
    _fileMenu.append( TEXT("&Stop"), IDC_STOP_REQ );
    _fileMenu.appendBar();
    _fileMenu.append( TEXT("E&xit"), IDC_EXIT_REQ );

    _editMenu.append( _T("&Copy view to clipboard"),
            IDC_COPY_VIEW_TO_CLIPBOARD_REQ );

    _viewMenu.append( TEXT("&Linear Magnitude"), IDC_LINEAR );
    _viewMenu.append( TEXT("Lo&garithmic Magnitude"), IDC_LOGARITHMIC );
    _viewMenu.append( TEXT("Lo&garithmic Real Magnitude"), IDC_LOGARITHMIC_REAL );
    //_viewMenu.append( TEXT("Lo&garithmic Complex Magnitude"), IDC_LOGARITHMIC_COMPLEX );
    _viewMenu.append( TEXT("&Phase"), IDC_PHASE );
    _viewMenu.appendBar();
    _viewMenu.append( TEXT("&Auto enhance"), IDC_AUTO_ENHANCE_TOGGLE_REQ );
    _viewMenu.appendBar();

    _helpMenu.append( TEXT( "&Web site" ), IDC_WEBPAGE_REQ );

    // Add to the view menu the colour map names.
    for( unsigned i = 0; i < ColourMap::GetNumMaps(); i++ ) {
        _viewMenu.append( ColourMap::GetMapName( i ), IDC_FIRST_COLOUR + i );
    }

    _dispatcher->registerEvent( this, IDC_OPEN_REQ );
    _dispatcher->registerEvent( this, IDC_WEBPAGE_REQ );
}

MainWindow::~MainWindow()
{

}

void
MainWindow::create()
{
    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                CW_USEDEFAULT, CW_USEDEFAULT, 0, _mainMenu.hMenu, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

int
MainWindow::onInitMenuPopup(HWND hwnd, HMENU hMenu, UINT item, 
        BOOL fSystemMenu)
{
    // Whenever the View menu is displayed, make sure the appropriate options
    // are checked.
    if ( hMenu == _viewMenu.hMenu ) {
        _viewMenu.setCheck(IDC_LINEAR,
                _mainView._waveletView.getType() == SoundImage::LINEAR );
        _viewMenu.setCheck(IDC_LOGARITHMIC,
                _mainView._waveletView.getType() == SoundImage::LOGARITHMIC );
        _viewMenu.setCheck(IDC_LOGARITHMIC_REAL,
                _mainView._waveletView.getType() == SoundImage::LOGARITHMIC_REAL );
        _viewMenu.setCheck(IDC_LOGARITHMIC_COMPLEX,
                _mainView._waveletView.getType() == SoundImage::LOGARITHMIC_COMPLEX );
        _viewMenu.setCheck(IDC_PHASE,
                _mainView._waveletView.getType() == SoundImage::PHASE );
        _viewMenu.setCheck(IDC_AUTO_ENHANCE_TOGGLE_REQ,
                _options->autoEnhance );
        _editMenu.enable(IDC_COPY_VIEW_TO_CLIPBOARD_REQ,
                _mainView.hasDoc() );
    }

    return 0;
}

LRESULT 
MainWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        HANDLE_MSG(hwnd, WM_INITMENUPOPUP, onInitMenuPopup);
    }
    
    *pbProcessed = FALSE;
    return 0;
}


void
MainWindow::onClose(HWND)
{
    PostQuitMessage(0);

}

void 
MainWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    // This is called when user clikced on menu.
    switch(id) {
        // all events:
        case IDC_WEBPAGE_REQ:
        case IDC_AUTO_ENHANCE_TOGGLE_REQ:
        case IDC_PLAY_REQ:
        case IDC_STOP_REQ:
        case IDC_COPY_VIEW_TO_CLIPBOARD_REQ:
            _dispatcher->dispatch( id );
            break;

        case IDC_EXIT_REQ:
            onClose(hwnd);
            break;

        case IDC_OPEN_REQ:
            onOpen();
            break;

        case IDC_LINEAR:
            _mainView._waveletView.setType( SoundImage::LINEAR );
            break;

        case IDC_LOGARITHMIC:
            _mainView._waveletView.setType( SoundImage::LOGARITHMIC );
            break;

        case IDC_LOGARITHMIC_REAL:
            _mainView._waveletView.setType( SoundImage::LOGARITHMIC_REAL );
            break;

        case IDC_LOGARITHMIC_COMPLEX:
            _mainView._waveletView.setType( SoundImage::LOGARITHMIC_COMPLEX );
            break;

        case IDC_PHASE:
            _mainView._waveletView.setType( SoundImage::PHASE );
            break;

    }

    if ( id >= IDC_FIRST_COLOUR && id <= IDC_LAST_COLOUR ) {
        _mainView._waveletView.setColourMap( id - IDC_FIRST_COLOUR );
    }
}

void
MainWindow::onOpen()
{
    TCHAR buffer[MAX_PATH*4];
    FileSelectDialog dialog(hwnd, TEXT("Open sound file"));
    dialog.addFilter(TEXT("Supported formats (*.wav)"), TEXT("*.wav"));
    dialog.addFilter(TEXT("All files (*.*)"), TEXT("*.*"));
    buffer[0] = 0;
    if ( dialog.getOpenFileName(buffer, sizeof(buffer)/sizeof(*buffer) ) ) {
        _mainView.open(buffer);
    }
}

void
MainWindow::onEvent( unsigned event )
{
    switch ( event ) {
        case IDC_OPEN_REQ:
            onOpen();
            break;
        case IDC_WEBPAGE_REQ:
            _application->openWebPage();
            break;
    }
}

BOOL 
MainWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{
    _mainView.create(hwnd, this);

    return TRUE;
}


void 
MainWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    RECT rect = getClientRect();
    _mainView.moveWindow( &rect, false );
}
