// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include "PlayTask.h"
#include "SoundDevice.h"
#include "Waveform.h"
#include "SoundImage.h"
#include <math.h>
#include "dbg.h" // must be last

PlayTask::PlayTask( TaskObserver* observer ) :
    Task<PlayTaskParams>::Task( observer )
{

}

PlayTask::~PlayTask()
{

}

void
PlayTask::run( PlayTaskParams params )
{
    unsigned row = 0;
    Waveform wave;
    wave.alloc( params.length );
    wave.sampleRate = params.image->sampleRate;
    wave.clear();
    SoundImageReader* reader = params.image->createReader();

    // Add up the rows of the wavelet transform to reconstruct the sound.

    float df = pow(params.image->upperScale/params.image->lowerScale, 1 /
            (params.image->upperScale-params.image->lowerScale));
    for ( float period = params.image->lowerScale; period <= params.image->upperScale; period *= df, row += 1 ) {

        if ( row < params.rowStart || row >= params.rowStart + params.numRows )
        {
            continue;
        }

        if ( _stopRequested  ) {
            break;
        }

        float* data = reader->getRow( row, params.offset, params.length );
        //float scale = (float)(1.0/pow((float)period, (float)1.5));

        for ( unsigned i = 0; i < params.length; i++ ) {
            wave.samples[i] += data[i*2] / pow( (float)period, (float)1.5 );
        }
    }

    delete reader;

    if ( _stopRequested ) {
        return;
    }

    // Maximize the volume.
    wave.normalize();

    // increase thread priority for audio pump.
    elevatePriority();

    // Play the produced waveform.
    SoundDevice::Initialize( wave.sampleRate, 1 );
    SoundDevice* dev = SoundDevice::instance();

    dev->beginPlay();

    unsigned size = dev->samplesPerBlock();
    unsigned numSamplesPlayed = 0;

    while(!_stopRequested && numSamplesPlayed != wave.size ) {
        short* buffer = (short*)dev->allocateBlock();
        unsigned i = 0;

        for ( i = 0; i < (unsigned)size && numSamplesPlayed < wave.size; i++, numSamplesPlayed++ ) {
            buffer[i] = (short)(wave.samples[numSamplesPlayed] * 32767); 
        }

        for ( ; i < size; i++ ) {
            buffer[i] = 0;
        }

        dev->write( (unsigned char*)buffer );
    }

    dev->endPlay();

    SoundDevice::shutdown();
}
