// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#ifndef Task_H
#define Task_H

#include "Thread.h"

class AbstractTask;

/**
  * Derive from this class to obtain progress information from a task.
  */
class TaskObserver 
{
public:
    virtual void onTaskNotify( AbstractTask* )=0;
    virtual ~TaskObserver();
};

class AbstractTask : public Thread
{
public:
	AbstractTask( TaskObserver* observer );
	virtual ~AbstractTask();

    double getProgress();
    bool done();
    void stop();

protected:
    void setProgress(double);
    void setDone();
    volatile bool _stopRequested;
    bool _done;
    double _progress;

    /**
     * Called when the controlling thread has requested stop. Warning: Executes in the context of
     * the controlling thread.
     */
    virtual void onStop();

private:	
    TaskObserver* _observer;
};

/** 
  * Derive from this class to implement a task that can be stopped, started,
  * and accepts the given parameters.
  */
template<class PARAMS>
class Task : public AbstractTask
{
private:
    PARAMS _params;
    virtual void run() {
        run( _params );
        _progress = 1.0;
        setDone();
    }

public:
	Task( TaskObserver* observer ) :
        AbstractTask( observer ) {

    }
    void start( PARAMS params ) {
        stop();
        _params = params;
        _done = false;
        _stopRequested = false;
        _progress = 0.0;
        Thread::start();
    }

protected:
    /**
      * The run method must continuously check _stopRequested, and immediately
      * cease processing if it is set to true. It should also occasionally call
      * setProgress() to report progress information to its observer.
      */
    virtual void run(PARAMS params)=0;
};

#endif // Task_H
