// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include <assert.h>
#include <windowsx.h>
#include "WaveViewWindow.h"
#include "WaveletDoc.h"
#include "ScalogramViewWindow.h"
#include <algorithm>
#include "dbg.h" // must be last

WaveViewWindow::WaveViewWindow(HINSTANCE hInstance) :
    ViewWindow(hInstance)
{
	_pszClassName = TEXT("WaveViewWindow");
	_pszTitle = TEXT("WaveView Window");

    _dwStyle |= WS_CHILD | WS_VISIBLE; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = (HCURSOR)LoadCursor(NULL, IDC_ARROW);
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
	_doc = 0;

    _dragging = false;
    _moving = false;
    _selStart = 0;
    _selEnd = 0;

    _scalogramView= 0;
}

WaveViewWindow::~WaveViewWindow()
{

}

void
WaveViewWindow::setScalogramView( ScalogramViewWindow* view )
{
    _scalogramView = view;
}

void
WaveViewWindow::create(HWND hParent)
{
    _hParent = hParent;

    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                CW_USEDEFAULT, CW_USEDEFAULT, _hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
}

LRESULT 
WaveViewWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}


void
WaveViewWindow::onClose(HWND)
{

}

void 
WaveViewWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{

}

BOOL 
WaveViewWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{

    return TRUE;
}

void
WaveViewWindow::onDestroy(HWND hwnd)
{

}


void 
WaveViewWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
WaveViewWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    int start, end;
    getSelExtentsPixels( &start, &end );
    RECT rect = getClientRect();
    rect.left = start;
    rect.right = end;
    onMouseMove( hwnd, x, y, keyFlags );
    _dragging = true;
    _dragStart.x = x;
    _dragStart.y = y;
    SetCapture( hwnd );
    if ( x >= start && x < end && y < 20 ) {
        _moving = true;
    }
    onMouseMove( hwnd, x, y, keyFlags );

    InvalidateRect( hwnd, &rect, FALSE );

}

void 
WaveViewWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{
    onMouseMove( hwnd, x, y, keyFlags );
    _dragging = false;
    _moving = false;
    ReleaseCapture();

    if ( _selEnd - _selStart < 10 ) {
        clearSel();
    }

    if ( _scalogramView ) {
        _scalogramView->setViewExtents( _selStart, _selEnd );
        _scalogramView->enhance();
    }
}

/**
  * Returns the selection extents in pixels.
  */
void
WaveViewWindow::getSelExtentsPixels( int* start, int* end )
{
    if ( _doc == NULL ) {
        *start = 0;
        *end = 0;
    }
    RECT rect = getClientRect();
    int width = rect.right - rect.left;

    *start = (int)((float)_selStart / _doc->wave->size * width);
    *end = (int)((float)_selEnd / _doc->wave->size * width);
}

void 
WaveViewWindow::onWaveformChanged( Waveform* )
{
    InvalidateRect( hwnd, NULL, FALSE );
}

void
WaveViewWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( _doc ) {
        int left, right, width;
        RECT rect = getClientRect();
        width = rect.right - rect.left;
        getSelExtentsPixels( &left, &right );

        // Change the cursor to indicate what the user can do.
        if ( x >= left && x < right && y <= 20 ) {
            SetCursor( LoadCursor( NULL, IDC_SIZEWE ) );
        } else {
            SetCursor( LoadCursor( NULL, IDC_CROSS ) );
        }

        // NOTE Because of the slowness of drawing it is vital to only
        // invalidate the smallest possible regions. Otherwise the user
        // will see flickering.

        // If we are moving the selection without changing its size,
        if ( _moving ) {
            int diff = x - _dragStart.x;
            int diffSamples = (int)((float)diff / width * _doc->wave->size);
            if ( diff != 0 ) {
                
                _selStart += diffSamples;
                _selEnd += diffSamples;
                if ( _selStart < 0 ) {
                    _selEnd -= _selStart;
                    _selStart -= _selStart;
                }
                if ( _selEnd > (int)_doc->wave->size ) {
                    _selStart -= _selEnd - (_doc->wave->size);
                    _selEnd -= _selEnd - (_doc->wave->size);
                }
                
                if ( _scalogramView ) {
                    _scalogramView->setViewExtents( _selStart, _selEnd );
                }
                RECT irect = rect;
                irect.left = _cpp_min(left, left+diff);
                irect.right = _cpp_max(right, right+diff)+2;
                InvalidateRect( hwnd, &irect, FALSE );
                _dragStart.x = x;
            }
        // If we are changing the size of the selection,    
        } else if ( _dragging ) {
            
            RECT inv = rect;

            inv.left = left;
            inv.right = right;

            _selStart = (int)((float)_dragStart.x / width * _doc->wave->size);
            _selEnd = (int)((float)x / width * _doc->wave->size);

            if ( _selStart > _selEnd ) {
                unsigned temp = _selStart;
                _selStart = _selEnd;
                _selEnd = temp;
            }

            int newLeft = (int)((float)_selStart / _doc->wave->size * width);
            int newRight = (int)((float)_selEnd / _doc->wave->size * width);

            if ( newLeft < inv.left ) {
                inv.left = newLeft;
            }

            if ( newRight > inv.right ) {
                inv.right = newRight;
            }

            inv.right += 4;
            inv.left -= 4;

            InvalidateRect( hwnd, &inv, FALSE );
        }
    }
}

/**
  * Draw a waveform to the screen. Compresses or stretches the waveform to fit.
  *
  * \param hdc Device context on which to draw.
  * 
  * \param scale Scale factor at which to draw.
  *
  * \param samples Pointer to waveform data.
  * 
  * \param numSamples Number of samples to render.
  *
  * \param offset Offset at which to render. samples[offset] will be placed at
  * rect->left.
  *
  * \parmas rect Area in which to draw.
  */
void
DrawWave( HDC hdc, double scale, double* samples, unsigned numSamples, unsigned offset, RECT* rect )
{
    unsigned i;
    int size = rect->bottom - rect->top;
    int middle = rect->top + size / 2;
    unsigned wstart;
    unsigned wend;

    Pen pen(PS_SOLID, 1, RGB(128,128,192) );
    DeviceContext dc(hdc);
    dc.selectPen( pen );

    // calculate start and end sample.

    wstart = offset;
    if  ( wstart >= numSamples ) {
        wstart = numSamples - 1;
    }

    wend = wstart + numSamples;
    if ( wend > numSamples ) {
        wend = numSamples;
    }

    // if the scale is >= 1.0,
    if ( scale >= 1.0 ) {

        // Move to start.
        MoveToEx( dc, rect->left, middle, NULL);


        // for each sample,
        for( i = wstart; i < wend; i++ ) { 
            int x;
            int y;

            // calculate y coordinate.
            y = middle + (int)(samples[i] * size/2);

            // calculate x coordinate on screen.
            x = rect->left + (int)(scale * (i - wstart));
            LineTo( dc, x, y );
        }
    } else {
        // compressed scale

        int lastX = rect->left;
        double maximum = samples[wstart];
        double minimum = samples[wstart];

        // for each sample,
        for( i = wstart; i < wend; i++ ) { 
            int x;
            
            // update maximum and minimum.
            if ( samples[i] > maximum ) {
                maximum = samples[i];
            }
            if ( samples[i] < minimum ) {
                minimum = samples[i];
            }

            // calculate x coordinate.
            x = rect->left + (int)(scale * (i - wstart));

            // if x coordinate is different from previous,
            if ( x != lastX ) {
                // Draw a line from the max to the min.
                MoveToEx( dc, x, 
                    middle + (int)(maximum * size/2),
                    NULL);
                LineTo( dc, x, 
                    middle + (int)(minimum * size/2) - 1 );

                maximum = samples[i];
                minimum = samples[i];
                lastX = x;
            }
        }
    }
}

/**
  * Clear the selection, setting it back to the entire waveform.
  */
void
WaveViewWindow::clearSel()
{
    if ( _doc ) {
        _selStart = 0;
        _selEnd = _doc->wave->size;
        InvalidateRect( hwnd, NULL, FALSE );
    }
}

void 
WaveViewWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, 
        bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    DeviceContext dc(hdc);

    // Paint window black.
    RECT rect = getClientRect();
    int width = rect.right - rect.left;
    FillRect( hdc, &rect, Brush(0) );
    Pen bluePen( PS_SOLID, 2, RGB(64,64,97) );
    Brush grayBrush( RGB(32,32,64));

	if ( _doc ) {
        if ( _doc->wave->size > 0 ) {
            RECT selRect = rect;

            // calculate where the selection is and draw it in a different
            // background colour.
            selRect.left = (int)((float)_selStart / _doc->wave->size * width);
            selRect.right = (int)((float)_selEnd / _doc->wave->size * width);

            dc.selectPen( bluePen );
            FillRect( dc, &selRect, grayBrush );

            RECT scrollRect = selRect;
            scrollRect.bottom = 20;
            DrawFrameControl( dc, &scrollRect, DFC_BUTTON, DFCS_BUTTONPUSH );
        }

        rect.top += 20;

        DrawWave( hdc,
              (double)(rect.right - rect.left) / _doc->wave->size,
              _doc->wave->samples,
              _doc->wave->size,
              0,
              &rect);
	}

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
WaveViewWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{

}

void 
WaveViewWindow::setDoc( WaveletDocRef doc )
{
    ViewWindow::setDoc( doc );

    _doc->wave->setObserver( this );

    clearSel();
}
