// -----------------------------------------------------------------------------
// Copyright 2008 Steve Hanov. All rights reserved.
//
// For permission to use, please contact steve.hanov@gmail.com. Permission will
// usually be granted without charge.
// -----------------------------------------------------------------------------
#include "SoundDevice.h"
#include <windows.h>
#include <assert.h>
#include <mmsystem.h>
#include "Thread.h"
#include <tchar.h>
#include "dbg.h" // must be last.

static unsigned int SamplesPerSec = 44100;
static unsigned int Channels = 2;
static unsigned int NumPrePostedBuffers = 3;

class WinSoundDevice : public SoundDevice
{
public:
	WinSoundDevice();
	~WinSoundDevice();

	void open();
	void winClose();
	
	virtual void beginRecord( AudioSink* pAudioSink );
	virtual bool isRecording();
	virtual void endRecord();

	virtual unsigned char* allocateBlock();
	virtual void releaseBlock( unsigned char* );
	virtual void read( unsigned char* );
	virtual void write( unsigned char* );
	virtual void beginPlay();
	virtual void endPlay();

	void fail(MMRESULT);

	unsigned int sampleRate();
	unsigned int samplesPerBlock();
    unsigned int numChannels();

	SoundDevice::Format format();

	HWAVEIN _hRecDevice;
	AudioSink* _audioSink;

	int _sampleRate;
	HWAVEOUT _hDevice;
	SoundDevice::Format _format;
	bool _bRecording;

	HANDLE _playSemaphore;
	HANDLE _recSemaphore;
	void postNewRecordBuffer( );

	Mutex _lock;
};

struct WaveBuffer
{
	WAVEHDR hdr;
	unsigned char* data;
};

WinSoundDevice* gInstance = NULL;

bool
SoundDevice::Initialize( unsigned sampleRate, unsigned channels )
{
    shutdown();
    SamplesPerSec = sampleRate;
    Channels = channels;
    return true;
}

/**
  * Obtain the instance of the singleton SoundDevice class.
  */
SoundDevice*
SoundDevice::instance()
{
	if ( gInstance == 0 ) {
		gInstance = new WinSoundDevice();
		gInstance->open();
	}

	return gInstance;
}

/**
  * Shuts down the sound system.
  */
void
SoundDevice::shutdown()
{
    delete gInstance;
    gInstance = 0;
}

WinSoundDevice::WinSoundDevice() :
	_sampleRate( SamplesPerSec ),
	_format( SoundDevice::Signed_16_LE ),
	_hDevice(NULL),
	_bRecording(false),
	_hRecDevice(0),
	_audioSink(0)
{
	_playSemaphore = CreateSemaphore( NULL, NumPrePostedBuffers, NumPrePostedBuffers, NULL );
	_recSemaphore = CreateSemaphore( NULL, NumPrePostedBuffers, NumPrePostedBuffers, NULL );
}

WinSoundDevice::~WinSoundDevice()
{
	CloseHandle(_playSemaphore);
	CloseHandle(_recSemaphore);
}

#pragma warning( disable:4312 )

void CALLBACK waveOutProc(
  HWAVEOUT hwo,      
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2     
)
{
	if ( uMsg != WOM_DONE ) {
		return;
	}

	LPWAVEHDR hdr = (LPWAVEHDR)(dwParam1);

	if ( gInstance ) {
		waveOutUnprepareHeader( gInstance->_hDevice, hdr, sizeof WAVEHDR );
		gInstance->releaseBlock( (unsigned char*)hdr->lpData );
		ReleaseSemaphore( gInstance->_playSemaphore, 1, NULL );
	}
}

void CALLBACK waveInProc(
  HWAVEIN hwi,       
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2     
)
{
	if ( uMsg != WIM_DATA ) {
		return;
	}

	LPWAVEHDR hdr = (LPWAVEHDR)(dwParam1);

	if ( gInstance ) {
        Mutex::Scope lock(gInstance->_lock);

		waveInUnprepareHeader( gInstance->_hRecDevice, hdr, sizeof WAVEHDR );
		ReleaseSemaphore( gInstance->_recSemaphore, 1, NULL );

		if ( gInstance->_bRecording && gInstance->_audioSink ) {
			gInstance->_audioSink->handleAudioData( (unsigned char*)hdr->lpData, 
				hdr->dwBytesRecorded );

			gInstance->postNewRecordBuffer();
		} else {
			gInstance->releaseBlock( (unsigned char*)hdr->lpData );
		}
	}
}

/**
  * Returns true if recording in progress.
  */
bool
WinSoundDevice::isRecording()
{
	return _bRecording;
}

/**
  * Begin a playing session.
  */
void
WinSoundDevice::beginPlay()
{
	assert( _hDevice == 0 );

	MMRESULT result;

	WAVEFORMATEX formatEx;
	ZeroMemory( &formatEx, sizeof formatEx );

	formatEx.wFormatTag =  WAVE_FORMAT_PCM;
	formatEx.nChannels = numChannels();
	formatEx.nSamplesPerSec = SamplesPerSec;
	formatEx.nBlockAlign = numChannels() * 16 / 8;
	formatEx.nAvgBytesPerSec = SamplesPerSec * formatEx.nBlockAlign;
	formatEx.wBitsPerSample = 16;
	formatEx.cbSize = 0;

	result = waveOutOpen( &_hDevice,
		WAVE_MAPPER,
		&formatEx,
		(DWORD)waveOutProc,
		NULL,
		CALLBACK_FUNCTION
		);

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}
}

/**
  * Ends a play session.
  */
void WinSoundDevice::endPlay()
{
	for( unsigned i = 0; i < NumPrePostedBuffers; ++i ) 
	{
		WaitForSingleObject( _playSemaphore, INFINITE );
	}

	MMRESULT result = waveOutPause( _hDevice );

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}
		
	result = waveOutReset( _hDevice );

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}

	result = waveOutClose( _hDevice );

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}

	_hDevice = 0;

	ReleaseSemaphore( _playSemaphore, NumPrePostedBuffers, 0);
}

void WinSoundDevice::open()
{
}

void
WinSoundDevice::winClose()
{

}

/**
  * Write a buffer to the sound output. 
  *
  * \param pdata Buffer previously obtained from allocateBlock()
  */
void 
WinSoundDevice::write( unsigned char* pdata )
{
	if ( WAIT_OBJECT_0 != WaitForSingleObject( _playSemaphore, INFINITE ) ) {
        assert(0);
	}

	WaveBuffer* buff = (WaveBuffer*)(pdata - sizeof WaveBuffer);

	WAVEHDR* hdr = &buff->hdr;
	MMRESULT result;
	
	result = waveOutPrepareHeader( _hDevice, hdr, sizeof WAVEHDR);

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}

	result = waveOutWrite( _hDevice, hdr, sizeof WAVEHDR );

	if ( result != MMSYSERR_NOERROR ) {
		waveOutUnprepareHeader( _hDevice, hdr, sizeof WAVEHDR );
		releaseBlock( pdata );
		delete hdr;
		fail( result );
	}
}

/**
  * Recording is not currently supported.
  */
void 
WinSoundDevice::read(unsigned char* data)
{

}

void 
SoundDevice::close()
{
	if ( gInstance ) {
		gInstance->winClose();
	}
}

unsigned int 
WinSoundDevice::sampleRate()
{
	return _sampleRate;
}

/** 
  * Returns the number of samples per block.
  */
unsigned int
WinSoundDevice::samplesPerBlock()
{
	return _sampleRate / 10;
}

unsigned int
WinSoundDevice::numChannels()
{
    return Channels;
}   

SoundDevice::Format 
WinSoundDevice::format()
{
	return _format;
}

void 
WinSoundDevice::fail(MMRESULT result)
{
	_TCHAR buffer[1000];
	if( ! mciGetErrorString( result, buffer, sizeof buffer ) ) {
		//throw Exception(_T("Failed to open sound device: unknown error."));
	} else {
		//throw Exception( buffer );	
	}
    assert(0);
}

unsigned char* 
WinSoundDevice::allocateBlock()
{
	WaveBuffer* buff = (WaveBuffer*)malloc( 
        samplesPerBlock() * 2 * numChannels() + sizeof WaveBuffer );

	if ( buff == 0 ) {
		//throw Exception(_T("Could not allocate buffer."));
        assert(0);
	}

	ZeroMemory( &buff->hdr, sizeof WAVEHDR );

	buff->hdr.lpData = (LPSTR)( (unsigned char*)buff + sizeof WaveBuffer );
	buff->hdr.dwBufferLength = samplesPerBlock() * 2 * numChannels();
	buff->hdr.dwLoops = 1;
	buff->data = (unsigned char*)buff->hdr.lpData;

	return buff->data;
}

void WinSoundDevice::releaseBlock( unsigned char* block )
{
	WaveBuffer* buff = (WaveBuffer*)(block - sizeof WaveBuffer);

	free(buff);
}

void WinSoundDevice::beginRecord( AudioSink* pAudioSink )
{
	if ( _audioSink != NULL ) {
		//throw Exception(_T("WinSoundDevice: beginRecord called while recording!"));
        assert(0);
	}

	MMRESULT result;

	if ( _hRecDevice == NULL ) {
		WAVEFORMATEX formatEx;
		ZeroMemory( &formatEx, sizeof formatEx );

		formatEx.wFormatTag =  WAVE_FORMAT_PCM;
		formatEx.nChannels = numChannels();
		formatEx.nSamplesPerSec = SamplesPerSec;
		formatEx.nBlockAlign = numChannels() * 16 / 8;
		formatEx.nAvgBytesPerSec = SamplesPerSec * formatEx.nBlockAlign;
		formatEx.wBitsPerSample = 16;
		formatEx.cbSize = 0;

		result = waveInOpen( &_hRecDevice, 
			WAVE_MAPPER,
			&formatEx, 
			(DWORD)waveInProc,
			NULL,
			CALLBACK_FUNCTION
			);

		if ( result != MMSYSERR_NOERROR ) {
			fail(result);	
		}
	}

	for ( unsigned int i = 0; i < NumPrePostedBuffers; ++i ) {
		postNewRecordBuffer();
	}

	_bRecording = true;
	_audioSink = pAudioSink;
	result = waveInStart( _hRecDevice );

	if ( result != MMSYSERR_NOERROR ) {
		fail(result);
		_bRecording = false;
		_audioSink = NULL;
	}
}

void WinSoundDevice::endRecord( )
{
	//AutoLock lock(_lock);
	MMRESULT result;
	
	_bRecording = false;
	_audioSink = NULL;

	for( unsigned i = 0; i < NumPrePostedBuffers; ++i ) 
	{
		WaitForSingleObject( _recSemaphore, INFINITE );
	}

	result = waveInStop( _hRecDevice );

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}

	result = waveInReset( _hRecDevice );

	if ( result != MMSYSERR_NOERROR ) {
		fail( result );
	}

	ReleaseSemaphore( _recSemaphore, NumPrePostedBuffers, 0);
}

void WinSoundDevice::postNewRecordBuffer()
{
	WaitForSingleObject( _recSemaphore, INFINITE );
	unsigned char* data = allocateBlock();
	WaveBuffer* buff = (WaveBuffer*)(data - sizeof WaveBuffer);

	MMRESULT result;
	
	result = waveInPrepareHeader( _hRecDevice, &buff->hdr, sizeof WAVEHDR );

	if ( result != MMSYSERR_NOERROR ) {
		fail(result);
	}

	result = waveInAddBuffer( _hRecDevice, &buff->hdr, sizeof WAVEHDR );

	if ( result != MMSYSERR_NOERROR ) {
		waveInUnprepareHeader( _hRecDevice, &buff->hdr, sizeof WAVEHDR );
		fail(result);
	}
}

