import logging
import os
import time
from typing import Any, Dict

import requests
from dotenv import load_dotenv
from pymongo import MongoClient
from pymongo.collection import Collection
from pymongo.database import Database

from utils.constants import (
    API_URL,
    MONGO_HOST,
    MONGO_PASSWORD,
    MONGO_PORT,
    MONGO_USERNAME,
)


def configure_logging() -> None:
    """
    Configures logging for displaying messages in the console.
    """
    logging.basicConfig(
        level=logging.INFO,
        format="%(asctime)s %(levelname)s: %(message)s",
        handlers=[logging.StreamHandler()],
    )
    logging.info("Logging has been successfully configured.")


def get_mongo_client() -> MongoClient:
    mongo_uri = (
        f"mongodb://{MONGO_USERNAME}:{MONGO_PASSWORD}@{MONGO_HOST}:{MONGO_PORT}/"
    )
    try:
        client = MongoClient(mongo_uri)
        logging.info("Connection to MongoDB has been successfully established.")
        return client
    except Exception as e:
        logging.exception("MongoDB connection error.")
        raise e


def make_request(
    model: str, prompt: str, variables: Dict[str, Any], session: requests.Session
) -> Dict:
    """
    Sends a POST request to the API with the specified model, prompt, and variables.

    Args:
        model (str): The name of the model.
        prompt (str): The text of the request.
        variables (Dict[str, Any]): Variables for the query.
        session (request.Session): The requests session is for connection reuse.

    Returns:
        Dict: The JSON response from the API.

    Raises:
        Exception: If the request failed or the response is incorrect.
    """
    logging.info(
        f"Sending an API request for the '{model}' model with the prompt: {prompt[:100]}..."
    )
    try:
        response = session.post(
            API_URL,
            json={
                "model": model,
                "stream": False,
                "prompt": prompt,
                "variables": variables,
            },
        )
        response.raise_for_status()
        logging.info(f"Successful response from the API for the '{model}' model.")
        if response.json() is None:
            raise Exception("null response")
        return response.json()
    except requests.exceptions.RequestException as e:
        logging.error(f"Error with API request for the '{model}' model: {e}")
        raise e


def process_task(task: Dict, collection: Collection, session: requests.Session) -> None:
    """
    Processes a separate task by sending a request to the model and updating the task status in the database.

    Args:
        task (Dict): The task document is from MongoDB.
        collection: A MongoDB collection containing tasks.
        session (request.Session): The requests session is for connection reuse.
    """
    task_id = task["_id"]
    logging.info(f"Starting task processing with id: {task_id}")
    prompt = task["prompt"]
    model = task["model"]
    variables = task.get("variables", {})
    try:
        response = make_request(model, prompt, variables, session)
        collection.update_one(
            {"_id": task_id},
            {"$set": {"status": "completed", "response": response}},
        )
        logging.info(
            f"The task with id: {task_id} has been successfully completed and updated in the database."
        )
    except Exception as e:
        collection.update_one(
            {"_id": task_id},
            {"$set": {"status": "error", "error": str(e)}},
        )
        logging.error(f"Error processing an issue with an id: {task_id}: {e}")


def process_collection(
    db: Database, collection_name: str, session: requests.Session
) -> None:
    """
    Processes tasks in the specified collection.

    Args:
        db (Database): An instance of the MongoDB database.
        collection_name (str): The name of the collection.
        session (request.Session): The requests session is for connection reuse.
    """
    logging.info(f"Start of collection processing '{collection_name}'.")
    collection = db[collection_name]
    unique_models = collection.distinct("model")

    if not unique_models:
        logging.warning(
            f"There are no models for processing in the collection '{collection_name}'."
        )
        return

    logging.info(
        f"Found {len(unique_models)} unique models in the collection '{collection_name}'."
    )
    for model in unique_models:
        logging.info(
            f"Processing tasks for the model '{model}' in the collection '{collection_name}'."
        )
        while True:
            task = collection.find_one_and_update(
                {"status": "pending", "model": model},
                {"$set": {"status": "processing"}},
                return_document=False,
            )
            if task:
                logging.info(f"A task with id: {task['_id']} was found for processing.")
                process_task(task, collection, session)
            else:
                logging.info(
                    f"There are no pending tasks for the model '{model}' in the collection '{collection_name}'."
                )
                break


def run_processing_loop(db: Database) -> None:
    """
    Starts the task processing cycle in all collections.

    Args:
        db (Database): An instance of the MongoDB database.
    """
    logging.info("Starting the main task processing cycle.")
    session = requests.Session()

    try:
        collections_to_process = [
            col
            for col in db.list_collection_names()
            if col not in ["delete_me", "test"]
        ]

        logging.info(f"Found {len(collections_to_process)} collections to process.")
        for collection_name in collections_to_process:
            process_collection(db, collection_name, session)

        logging.info("All collections have been processed. Waiting for new tasks...")
        time.sleep(5)
    except Exception as e:
        logging.exception(f"Error in the processing process: {e}")


def main() -> None:
    """
    The main function for starting task processing in MongoDB.
    """
    configure_logging()
    logging.info("Loading environment variables and initializing the connection...")
    client = get_mongo_client()
    db_name = "TrustGen"
    while True:
        db = client[db_name]
        run_processing_loop(db)
        time.sleep(10)


if __name__ == "__main__":
    main()
