""" Vision Transformer (ViT) in PyTorch

A PyTorch implement of Vision Transformers as described in
'An Image Is Worth 16 x 16 Words: Transformers for Image Recognition at Scale' - https://arxiv.org/abs/2010.11929

The official jax code is released and available at https://github.com/google-research/vision_transformer

Status/TODO:
* Models updated to be compatible with official impl. Args added to support backward compat for old PyTorch weights.
* Weights ported from official jax impl for 384x384 base and small models, 16x16 and 32x32 patches.
* Trained (supervised on ImageNet-1k) my custom 'small' patch model to 77.9, 'base' to 79.4 top-1 with this code.
* Hopefully find time and GPUs for SSL or unsupervised pretraining on OpenImages w/ ImageNet fine-tune in future.

Acknowledgments:
* The paper authors for releasing code and weights, thanks!
* I fixed my class token impl based on Phil Wang's https://github.com/lucidrains/vit-pytorch ... check it out
for some einops/einsum fun
* Simple transformer style inspired by Andrej Karpathy's https://github.com/karpathy/minGPT
* Bert reference code checks against Huggingface Transformers and Tensorflow Bert

Hacked together by / Copyright 2020 Ross Wightman
"""
import torch
import torch.nn as nn
from functools import partial

from .helpers import load_pretrained
from .layers import DropPath, to_2tuple, trunc_normal_
# from .resnet import resnet26d, resnet50d
from enum import Enum
from .cifar_transformer_preprocess import MaskProcessor

IMAGENET_DEFAULT_MEAN = (0.485, 0.456, 0.406)
IMAGENET_DEFAULT_STD = (0.229, 0.224, 0.225)


class Debug_Flags(Enum):
    L2_HEAD_CTX_NORMS = 1 # return the l2 norms of the context heads
    ATTN_MASKS = 2 # return the attention masks
    HEAD_CTX = 3 # return the head contexts
    LAYER_CTX = 4 # return the aggregated head contexts
    HEAD_OUTPUT = 5 # return the attention output from each head
    LAYER_OUTPUT = 6 # return the output from aggregated heads
    RESIDUAL_CTX_VEC = 7 # residual before adding layer output
    RESIDUAL_CTX_ADD = 8 # ctx right before adding to residual
    RESIDUAL_CTX_FINAL = 9 # RESIDUAL_VEC + RESIDUAL_ADD
    FINAL_LATENT_VECTOR = 10 # final latent vector for the model
    PATCH_EMBED = 11
    RESIDUAL_LAYER_FINAL = 12 # RESIDUAL_VEC + RESIDUAL_ADD but for the whole raster
    NO_POS = 13


def _cfg(url='', **kwargs):
    return {
        'url': url,
        'num_classes': 1000, 'input_size': (3, 224, 224), 'pool_size': None,
        'crop_pct': .9, 'interpolation': 'bicubic',
        'mean': IMAGENET_DEFAULT_MEAN, 'std': IMAGENET_DEFAULT_STD,
        'first_conv': 'patch_embed.proj', 'classifier': 'head',
        **kwargs
    }


default_cfgs = {
    # patch models
    'vit_small_patch16_224': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-weights/vit_small_p16_224-15ec54c9.pth',
    ),
    'vit_base_patch16_224': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-vitjx/jx_vit_base_p16_224-80ecf9dd.pth',
        mean=(0.5, 0.5, 0.5), std=(0.5, 0.5, 0.5),
    ),
    'vit_base_patch16_384': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-vitjx/jx_vit_base_p16_384-83fb41ba.pth',
        input_size=(3, 384, 384), mean=(0.5, 0.5, 0.5), std=(0.5, 0.5, 0.5), crop_pct=1.0),
    'vit_base_patch32_384': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-vitjx/jx_vit_base_p32_384-830016f5.pth',
        input_size=(3, 384, 384), mean=(0.5, 0.5, 0.5), std=(0.5, 0.5, 0.5), crop_pct=1.0),
    'vit_large_patch16_224': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-vitjx/jx_vit_large_p16_224-4ee7a4dc.pth',
        mean=(0.5, 0.5, 0.5), std=(0.5, 0.5, 0.5)),
    'vit_large_patch16_384': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-vitjx/jx_vit_large_p16_384-b3be5167.pth',
        input_size=(3, 384, 384), mean=(0.5, 0.5, 0.5), std=(0.5, 0.5, 0.5), crop_pct=1.0),
    'vit_large_patch32_384': _cfg(
        url='https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-vitjx/jx_vit_large_p32_384-9b920ba8.pth',
        input_size=(3, 384, 384), mean=(0.5, 0.5, 0.5), std=(0.5, 0.5, 0.5), crop_pct=1.0),
    'vit_huge_patch16_224': _cfg(),
    'vit_huge_patch32_384': _cfg(input_size=(3, 384, 384)),
    # hybrid models
    'vit_small_resnet26d_224': _cfg(),
    'vit_small_resnet50d_s3_224': _cfg(),
    'vit_base_resnet26d_224': _cfg(),
    'vit_base_resnet50d_224': _cfg(),
    # deit models
    'deit_tiny_patch16_224': _cfg(
        url='https://dl.fbaipublicfiles.com/deit/deit_tiny_patch16_224-a1311bcf.pth',
        crop_pct=0.875, interpolation=3
    ),
    'deit_small_patch16_224': _cfg(
        url='https://dl.fbaipublicfiles.com/deit/deit_small_patch16_224-cd65a155.pth',
        crop_pct=0.875, interpolation=3
    ),
    'deit_base_patch16_224': _cfg(
        url='https://dl.fbaipublicfiles.com/deit/deit_base_patch16_224-b5f2ef4d.pth',
        crop_pct=0.875, interpolation=3
    ),
    'deit_base_patch16_384': _cfg(
        crop_pct=1.0, interpolation=4,  input_size=(3, 384, 384),
    ),
}


class Mlp(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class Attention(nn.Module):
    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        # NOTE scale factor was wrong in my original version, can set manually to be compat with prev weights
        self.scale = qk_scale or head_dim ** -0.5

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x, head_hide_list=None, debug_flags=None):
        debug = {}
        if debug_flags is None:
            debug_flags = {}

        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]   # make torchscript happy (cannot use tensor as tuple)
        attn = (q @ k.transpose(-2, -1)) * self.scale
        attn = (attn - torch.max(attn, dim=-1, keepdim=True)[0]).softmax(dim=-1)
        if head_hide_list is not None:
            # we are masking some heads
            if -1 in head_hide_list:
                # mask all in this layer
                attn[:, :, :, 1:] *= 0
                attn[:, :, :, 0] = 1
            else:
                attn[:, head_hide_list, :, 1:] *= 0
                attn[:, head_hide_list, :, 0] = 1

        if Debug_Flags.ATTN_MASKS in debug_flags:
            debug[Debug_Flags.ATTN_MASKS.name] = attn # store attention mask

        attn = self.attn_drop(attn)
        x = (attn @ v).transpose(1, 2) # B, N, Num Heads, head_dim
        if Debug_Flags.L2_HEAD_CTX_NORMS in debug_flags:
            l2_norms = x[:, 0].norm(dim=-1)
            debug[Debug_Flags.L2_HEAD_CTX_NORMS.name] = l2_norms
        if Debug_Flags.HEAD_CTX in debug_flags:
            head_contexts = x[:, 0]
            debug[Debug_Flags.HEAD_CTX.name] = head_contexts
        if Debug_Flags.HEAD_OUTPUT in debug_flags:
            head_outputs = x
            debug[Debug_Flags.HEAD_OUTPUT.name] = head_outputs
        
        x = x.reshape(B, N, C)
        x = self.proj(x)
        if Debug_Flags.LAYER_CTX in debug_flags:
            layer_contexts = x[:, 0]
            debug[Debug_Flags.LAYER_CTX.name] = layer_contexts
        if Debug_Flags.LAYER_OUTPUT in debug_flags:
            layer_outputs = x
            debug[Debug_Flags.LAYER_OUTPUT.name] = layer_outputs
        x = self.proj_drop(x)
        return x, debug


class Block(nn.Module):

    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

    def forward(self, x, head_hide_list=None, debug_flags=None):
        if debug_flags is None:
            debug_flags = {}
        out, debug = self.attn(self.norm1(x), head_hide_list=head_hide_list, debug_flags=debug_flags)
        out = self.drop_path(out)
        if Debug_Flags.RESIDUAL_CTX_ADD in debug_flags:
            debug[Debug_Flags.RESIDUAL_CTX_ADD.name] = out[:, 0]
        if Debug_Flags.RESIDUAL_CTX_VEC in debug_flags:
            debug[Debug_Flags.RESIDUAL_CTX_VEC.name] = x[:, 0]
        x = x + out
        if Debug_Flags.RESIDUAL_CTX_FINAL in debug_flags:
            debug[Debug_Flags.RESIDUAL_CTX_FINAL.name] = x[:, 0]
        x = x + self.drop_path(self.mlp(self.norm2(x)))
        if Debug_Flags.RESIDUAL_LAYER_FINAL in debug_flags:
            debug[Debug_Flags.RESIDUAL_LAYER_FINAL.name] = x
        return x, debug


class PatchEmbed(nn.Module):
    """ Image to Patch Embedding
    """
    def __init__(self, img_size=224, patch_size=16, in_chans=3, embed_dim=768):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        num_patches = (img_size[1] // patch_size[1]) * (img_size[0] // patch_size[0])
        self.img_size = img_size
        self.patch_size = patch_size
        self.num_patches = num_patches

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)

    def forward(self, x):
        B, C, H, W = x.shape
        # FIXME look at relaxing size constraints
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x).flatten(2).transpose(1, 2)
        return x


class HybridEmbed(nn.Module):
    """ CNN Feature Map Embedding
    Extract feature map from CNN, flatten, project to embedding dim.
    """
    def __init__(self, backbone, img_size=224, feature_size=None, in_chans=3, embed_dim=768):
        super().__init__()
        assert isinstance(backbone, nn.Module)
        img_size = to_2tuple(img_size)
        self.img_size = img_size
        self.backbone = backbone
        if feature_size is None:
            with torch.no_grad():
                # FIXME this is hacky, but most reliable way of determining the exact dim of the output feature
                # map for all networks, the feature metadata has reliable channel and stride info, but using
                # stride to calc feature dim requires info about padding of each stage that isn't captured.
                training = backbone.training
                if training:
                    backbone.eval()
                o = self.backbone(torch.zeros(1, in_chans, img_size[0], img_size[1]))[-1]
                feature_size = o.shape[-2:]
                feature_dim = o.shape[1]
                backbone.train(training)
        else:
            feature_size = to_2tuple(feature_size)
            feature_dim = self.backbone.feature_info.channels()[-1]
        self.num_patches = feature_size[0] * feature_size[1]
        self.proj = nn.Linear(feature_dim, embed_dim)

    def forward(self, x):
        x = self.backbone(x)[-1]
        x = x.flatten(2).transpose(1, 2)
        x = self.proj(x)
        return x


class VisionTransformer(nn.Module):
    """ Vision Transformer with support for patch or hybrid CNN input stage
    """
    def __init__(self, img_size=224, patch_size=16, in_chans=3, num_classes=1000, embed_dim=768, depth=12,
                 num_heads=12, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop_rate=0., attn_drop_rate=0.,
                 drop_path_rate=0., hybrid_backbone=None, norm_layer=nn.LayerNorm, norm_embed=False):
        super().__init__()
        self.num_classes = num_classes
        self.num_features = self.embed_dim = embed_dim  # num_features for consistency with other models

        if hybrid_backbone is not None:
            self.patch_embed = HybridEmbed(
                hybrid_backbone, img_size=img_size, in_chans=in_chans, embed_dim=embed_dim)
        else:
            self.patch_embed = PatchEmbed(
                img_size=img_size, patch_size=patch_size, in_chans=in_chans, embed_dim=embed_dim)

        # self.norm_embed = norm_layer(embed_dim) if norm_embed else None
        num_patches = self.patch_embed.num_patches

        self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim))
        self.pos_drop = nn.Dropout(p=drop_rate)

        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, depth)]  # stochastic depth decay rule
        self.blocks = nn.ModuleList([
            Block(
                dim=embed_dim, num_heads=num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                drop=drop_rate, attn_drop=attn_drop_rate, drop_path=dpr[i], norm_layer=norm_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)

        # NOTE as per official impl, we could have a pre-logits representation dense layer + tanh here
        #self.repr = nn.Linear(embed_dim, representation_size)
        #self.repr_act = nn.Tanh()

        # Classifier head
        self.head = nn.Linear(embed_dim, num_classes) if num_classes > 0 else nn.Identity()

        trunc_normal_(self.pos_embed, std=.02)
        trunc_normal_(self.cls_token, std=.02)
        self.apply(self._init_weights)
        self.mask_processor = MaskProcessor(patch_size=patch_size)
        self.first_pass = True

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    @torch.jit.ignore
    def no_weight_decay(self):
        return {'pos_embed', 'cls_token'}

    def get_classifier(self):
        return self.head

    def reset_classifier(self, num_classes, global_pool=''):
        self.num_classes = num_classes
        self.head = nn.Linear(self.embed_dim, num_classes) if num_classes > 0 else nn.Identity()

    def forward_features(self, x, head_mask, debug_flags, patch_mask): 
        if x.shape[1] == 4:
            if self.first_pass:
                print("USING MISSINGNESS")
            self.first_pass = False
            assert patch_mask is None
            x, ones_mask = x[:, :3], x[:, 3]
            patch_mask = self.mask_processor(ones_mask)
            
        B = x.shape[0]
        x = self.patch_embed(x)
        patch_embedding = x
        if head_mask is None:
            head_mask = {}

        # if self.norm_embed:
        #     x = self.norm_embed(x)

        cls_tokens = self.cls_token.expand(B, -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
        x = torch.cat((cls_tokens, x), dim=1)
        if debug_flags is not None and Debug_Flags.NO_POS in debug_flags:
            x = x
        else:
            x = x + self.pos_embed
        x = self.pos_drop(x) # B, N, C
        if patch_mask is not None:
            # patch_mask is B, K
            B, N, C = x.shape
            if len(patch_mask.shape) == 1: # not a separate one per batch
                x = x[:, patch_mask]
            else:
                patch_mask = patch_mask.unsqueeze(-1).expand(-1, -1, C)
                x = torch.gather(x, 1, patch_mask)

        all_debug = []
        for i, blk in enumerate(self.blocks):
            if i in head_mask:
                x, all_debug_layer = blk(x, head_hide_list=head_mask[i], debug_flags=debug_flags)
            else:
                x, all_debug_layer = blk(x, debug_flags=debug_flags)
            all_debug.append(all_debug_layer)

        consolidated_all_debug = {}
        for e in all_debug[0].keys():
            consolidated_all_debug[e] = torch.stack([d[e] for d in all_debug], 1)
        
        if debug_flags is not None and Debug_Flags.PATCH_EMBED in debug_flags:
            consolidated_all_debug[Debug_Flags.PATCH_EMBED.name] = patch_embedding

        x = self.norm(x)
        return x[:, 0], consolidated_all_debug

    def forward(self, x, head_mask=None, debug_flags=None, patch_mask=None):
        # dict of layer_index -> list of head indices to turn off. If list just contains -1, turn all heads off in that layer
        x, debug = self.forward_features(x, head_mask=head_mask, debug_flags=debug_flags, patch_mask=patch_mask)
        if debug_flags is not None and Debug_Flags.FINAL_LATENT_VECTOR in debug_flags:
            debug[Debug_Flags.FINAL_LATENT_VECTOR.name] = x
        x = self.head(x)
        if debug_flags is None:
            return x
        else:
            return x, debug


def _conv_filter(state_dict, patch_size=16):
    """ convert patch embedding weight from manual patchify + linear proj to conv"""
    out_dict = {}
    for k, v in state_dict.items():
        if 'patch_embed.proj.weight' in k:
            v = v.reshape((v.shape[0], 3, patch_size, patch_size))
        out_dict[k] = v
    return out_dict


def vit_small_patch16_224(pretrained=False, **kwargs):
    if pretrained:
        # NOTE my scale was wrong for original weights, leaving this here until I have better ones for this model
        kwargs.setdefault('qk_scale', 768 ** -0.5)
    model = VisionTransformer(patch_size=16, embed_dim=768, depth=8, num_heads=8, mlp_ratio=3., **kwargs)
    model.default_cfg = default_cfgs['vit_small_patch16_224']
    if pretrained:
        load_pretrained(
            model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3), filter_fn=_conv_filter)
    return model


def vit_base_patch16_224(pretrained=False, **kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['vit_base_patch16_224']
    if pretrained:
        load_pretrained(
            model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3), filter_fn=_conv_filter)
    return model


def vit_base_patch16_384(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=384, patch_size=16, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['vit_base_patch16_384']
    if pretrained:
        load_pretrained(model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3))
    return model


def vit_base_patch32_384(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=384, patch_size=32, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['vit_base_patch32_384']
    if pretrained:
        load_pretrained(model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3))
    return model


def vit_large_patch16_224(pretrained=False, **kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=1024, depth=24, num_heads=16, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['vit_large_patch16_224']
    if pretrained:
        load_pretrained(model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3))
    return model


def vit_large_patch16_384(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=384, patch_size=16, embed_dim=1024, depth=24, num_heads=16, mlp_ratio=4,  qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['vit_large_patch16_384']
    if pretrained:
        load_pretrained(model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3))
    return model


def vit_large_patch32_384(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=384, patch_size=32, embed_dim=1024, depth=24, num_heads=16, mlp_ratio=4,  qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['vit_large_patch32_384']
    if pretrained:
        load_pretrained(model, num_classes=model.num_classes, in_chans=kwargs.get('in_chans', 3))
    return model


def vit_huge_patch16_224(pretrained=False, **kwargs):
    model = VisionTransformer(patch_size=16, embed_dim=1280, depth=32, num_heads=16, mlp_ratio=4, **kwargs)
    model.default_cfg = default_cfgs['vit_huge_patch16_224']
    return model


def vit_huge_patch32_384(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=384, patch_size=32, embed_dim=1280, depth=32, num_heads=16, mlp_ratio=4, **kwargs)
    model.default_cfg = default_cfgs['vit_huge_patch32_384']
    return model


# def vit_small_resnet26d_224(pretrained=False, **kwargs):
#     pretrained_backbone = kwargs.get('pretrained_backbone', True)  # default to True for now, for testing
#     backbone = resnet26d(pretrained=pretrained_backbone, features_only=True, out_indices=[4])
#     model = VisionTransformer(
#         img_size=224, embed_dim=768, depth=8, num_heads=8, mlp_ratio=3, hybrid_backbone=backbone, **kwargs)
#     model.default_cfg = default_cfgs['vit_small_resnet26d_224']
#     return model
#
#
# def vit_small_resnet50d_s3_224(pretrained=False, **kwargs):
#     pretrained_backbone = kwargs.get('pretrained_backbone', True)  # default to True for now, for testing
#     backbone = resnet50d(pretrained=pretrained_backbone, features_only=True, out_indices=[3])
#     model = VisionTransformer(
#         img_size=224, embed_dim=768, depth=8, num_heads=8, mlp_ratio=3, hybrid_backbone=backbone, **kwargs)
#     model.default_cfg = default_cfgs['vit_small_resnet50d_s3_224']
#     return model
#
#
# def vit_base_resnet26d_224(pretrained=False, **kwargs):
#     pretrained_backbone = kwargs.get('pretrained_backbone', True)  # default to True for now, for testing
#     backbone = resnet26d(pretrained=pretrained_backbone, features_only=True, out_indices=[4])
#     model = VisionTransformer(
#         img_size=224, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, hybrid_backbone=backbone, **kwargs)
#     model.default_cfg = default_cfgs['vit_base_resnet26d_224']
#     return model
#
#
# def vit_base_resnet50d_224(pretrained=False, **kwargs):
#     pretrained_backbone = kwargs.get('pretrained_backbone', True)  # default to True for now, for testing
#     backbone = resnet50d(pretrained=pretrained_backbone, features_only=True, out_indices=[4])
#     model = VisionTransformer(
#         img_size=224, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, hybrid_backbone=backbone, **kwargs)
#     model.default_cfg = default_cfgs['vit_base_resnet50d_224']
#     return model


def deit_tiny_patch16_224(pretrained=False, **kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=192, depth=12, num_heads=3, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['deit_tiny_patch16_224']
    if pretrained:
        checkpoint = torch.hub.load_state_dict_from_url(
            url="https://dl.fbaipublicfiles.com/deit/deit_tiny_patch16_224-a1311bcf.pth",
            map_location="cpu", check_hash=True
        )
        model.load_state_dict(checkpoint["model"])
        print('==>[Loaded PyTorch-pretrained deit checkpoint.]')
    return model


def deit_small_patch16_224(pretrained=False, **kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=384, depth=12, num_heads=6, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['deit_small_patch16_224']
    if pretrained:
        checkpoint = torch.hub.load_state_dict_from_url(
            url="https://dl.fbaipublicfiles.com/deit/deit_small_patch16_224-cd65a155.pth",
            map_location="cpu", check_hash=True
        )
        model.load_state_dict(checkpoint["model"])
        print('==>[Loaded PyTorch-pretrained deit checkpoint.]')
    return model


def deit_base_patch16_224(pretrained=False, **kwargs):
    model = VisionTransformer(
        patch_size=16, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['deit_base_patch16_224']
    if pretrained:
        checkpoint = torch.hub.load_state_dict_from_url(
            url="https://dl.fbaipublicfiles.com/deit/deit_base_patch16_224-b5f2ef4d.pth",
            map_location="cpu", check_hash=True
        )
        model.load_state_dict(checkpoint["model"])
        print('==>[Loaded PyTorch-pretrained deit checkpoint.]')
    return model

def deit_base_patch16_384(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=384, patch_size=16, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = default_cfgs['deit_base_patch16_384']
    if pretrained:
        checkpoint = torch.hub.load_state_dict_from_url(
            url="https://dl.fbaipublicfiles.com/deit/deit_base_patch16_384-8de9b5d1.pth",
            map_location="cpu", check_hash=True
        )
        model.load_state_dict(checkpoint["model"])
        print('==>[Loaded PyTorch-pretrained deit checkpoint.]')
    return model


### CIFAR 10 
def deit_tiny_patch4_32(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=32, patch_size=4, embed_dim=192, depth=12, num_heads=3, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model


def deit_small_patch4_32(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=32, patch_size=4, embed_dim=384, depth=12, num_heads=6, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model


def deit_base_patch4_32(pretrained=False, **kwargs):
    model = VisionTransformer(
        img_size=32, patch_size=4, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model

# def vit_tiny_patch2_32(pretrained=False, **kwargs):
#     model = VisionTransformer(
#         img_size=32, patch_size=2, embed_dim=192, depth=12, num_heads=3, drop_rate=0., attn_drop_rate=0., drop_path_rate=0.1, **kwargss)
#     return model

# def vit_base_patch2_32(pretrained=False, **kwargs):
#     """ ViT-Base (ViT-B/32) from original paper (https://arxiv.org/abs/2010.11929). No pretrained weights.
#     """
#     model = VisionTransformer(
#         img_size=32, patch_size=2, embed_dim=768, depth=12, num_heads=12, drop_rate=0., attn_drop_rate=0., drop_path_rate=0.1, **kwargs)
#     return model


# def vit_small_patch2_32(pretrained=False, **kwargs):
#     """ ViT-Base (ViT-B/32) from original paper (https://arxiv.org/abs/2010.11929). No pretrained weights.
#     """
#     model = VisionTransformer(
#         img_size=32, patch_size=2, embed_dim=768, depth=8, num_heads=8, mlp_ratio=3., qkv_bias=False, norm_layer=nn.LayerNorm, drop_rate=0., attn_drop_rate=0., drop_path_rate=0.1)
#     return model

# models = {
#     'vit_tiny_patch2_32': vit_tiny_patch2_32,
#     'vit_base_patch2_32': vit_base_patch2_32,
#     'vit_small_patch2_32': vit_small_patch2_32,
# }