import json
import os
import pandas as pd
from typing import Any


def load_standard_data(input_path: str) -> list[dict]:
    if ".jsonl" in input_path:
        with open(input_path, "r") as f:
            data = [json.loads(x) for x in f.readlines()]
    elif ".json" in input_path:
        with open(input_path, "r") as f:
            data = json.load(f)
            for id, item in enumerate(data):
                if "id" not in item:
                    item["id"] = id
    elif ".tsv" in input_path:
        data = pd.read_csv(input_path, sep="\t").to_dict(orient="records")
        for id, item in enumerate(data):
            if "prompt" not in item and "attack" in item:
                item["prompt"] = item["attack"]
                if "id" not in item:
                    item["id"] = id
    else:
        raise ValueError(f"Unrecognized file type: {input_path}")

    return data


def write_standard_data(data: list[dict], output_path: str, makedirs: bool = True):
    if makedirs:
        os.makedirs(os.path.dirname(os.path.abspath(output_path)), exist_ok=True)

    with open(output_path, "w") as f:
        for item in data:
            f.write(json.dumps(item) + "\n")


def export_standard_to_csv(data: list[dict], output_path: str):
    df = pd.DataFrame(data)
    df.to_csv(output_path, index=False)


def import_csv_to_standard(csv_path: str) -> list[dict]:
    df = pd.read_csv(csv_path)
    data = []
    for i, row in df.iterrows():
        data.append(dict(row))
    return data


def transpose_standard_data(data: list[dict[str, Any]] | dict[str, list[Any]]) -> dict[str, list[Any]] | list[dict[str, Any]]:
    """
    Transposes a list of dictionaries or a dictionary of lists into the opposite structure.

    If `data` is a list of dictionaries, the result is a dictionary where the keys are the keys from the input dictionaries, and the values are lists containing the corresponding values from each input dictionary.

    If `data` is a dictionary of lists, the result is a list of dictionaries where each dictionary has the keys from the input dictionary, and the values are the elements at the corresponding index from the input lists.
    """
    if isinstance(data, list):
        all_keys = set().union(*map(dict.keys, data))
        result = {key: [] for key in all_keys}
        for d in data:
            for k in all_keys:
                if k in d:
                    result[k].append(d[k])
                else:
                    result[k].append(None)
        return result
    else:
        result = []
        n_items = len(next(iter(data.values())))
        for i in range(n_items):
            result.append({k: v[i] for k, v in data.items() if v[i] is not None})
        return result
