import os
import shutil

def remove_folders_by_name(root_dir, folder_name_pattern):
    """
    Remove folders that match a specific name pattern from all subdirectories
    
    Args:
        root_dir: Root directory to start search
        folder_name_pattern: Start of folder name to match
    """
    removed_count = 0
    total_size_freed = 0
    print('root_dir:', root_dir)
    
    for dirpath, dirnames, filenames in os.walk(root_dir, topdown=False):
        # Find matching directories
        matching_dirs = [d for d in dirnames if d.startswith(folder_name_pattern)]
   
        for dirname in matching_dirs:
            dir_path = os.path.join(dirpath, dirname)
            # Calculate directory size before removing
            dir_size = 0
            for root, dirs, files in os.walk(dir_path):
                for file in files:
                    file_path = os.path.join(root, file)
                    try:
                        dir_size += os.path.getsize(file_path)
                    except (OSError, FileNotFoundError):
                        pass
            
            total_size_freed += dir_size
            # Remove directory
            shutil.rmtree(dir_path)
            removed_count += 1
            print(f"Removed directory: {dir_path}")
    
    # Convert bytes to MB
    size_mb = total_size_freed / (1024 * 1024)
    print(f"\nSummary:")
    print(f"Total directories removed: {removed_count}")
    print(f"Total space freed: {size_mb:.2f} MB")

if __name__ == "__main__":
    import argparse
    parser = argparse.ArgumentParser()
    parser.add_argument("root_dir", help="Root directory to search")
    parser.add_argument("folder_pattern", help="Start of folder name to match")
    args = parser.parse_args()
    
    # Confirm with user
    print(f"This will recursively remove all directories that start with '{args.folder_pattern}'")
    print(f"in '{args.root_dir}' and its subdirectories.")
    response = input("Continue? This cannot be undone! [y/N]: ")
    
    if response.lower() == 'y':
        remove_folders_by_name(args.root_dir, args.folder_pattern)
    else:
        print("Operation cancelled")