#!/bin/bash
#Set job requirements

#SBATCH -n 1
#SBATCH -t 00:20:00
#SBATCH --job-name=run_projections
#SBATCH --partition=gpu_a100
#SBATCH --gpus=1
#SBATCH --gpus-per-node=1
#SBATCH --mem=30G
#SBATCH --mail-type=BEGIN,END
#SBATCH --mail-user=f.g.holstege@uva.nl

# Set directory for code
cd $HOME/opt-separation

# Loading modules and activating environment
module load 2023
source ~/JHS_installations/venvs/env_opt_separation/bin/activate

# Define arrays of models and their corresponding types
declare -a models=(
  "meta-llama/Llama-3.1-8B-Instruct"
)

declare -a model_types=(
  "llama"
)

# Define datasets to process
declare -a datasets=(
  "winobias"
)

# Configuration parameters
BATCH_SIZE=64
MAX_LENGTH=40  # Set to appropriate value or leave as null with --max_length=null
DEVICE="cuda"  # Change to "cuda" for NVIDIA GPUs or "auto" for automatic selection
TORCH_DTYPE="float16"  # Options: float16, float32, bfloat16
OUTPUT_DIR_PRED="$TMPDIR"  # Using TMPDIR for temporary storage
PERMANENT_OUTPUT_FOLDER="$HOME/opt-separation/data/result_data"

# Echo TMPDIR for debugging
echo "TMPDIR is set to: $TMPDIR"
echo "Current working directory: $(pwd)"

# Check if TMPDIR is set
if [ -z "$TMPDIR" ]; then
    echo "Error: TMPDIR is not set"
    exit 1
fi

# Create the output directories
mkdir -p "$OUTPUT_DIR_PRED"
mkdir -p "$PERMANENT_OUTPUT_FOLDER"

# Test write permissions
if ! touch "$OUTPUT_DIR_PRED/test.txt" 2>/dev/null; then
    echo "Error: Cannot write to $OUTPUT_DIR_PRED"
    exit 1
fi
rm -f "$OUTPUT_DIR_PRED/test.txt"

# Log file
LOG_FILE="${PERMANENT_OUTPUT_FOLDER}/prediction_runs_$(date +%Y%m%d_%H%M%S).log"
echo "Starting prediction runs at $(date)" > "$LOG_FILE"

# Loop through each model, model type, and dataset combination
for i in "${!models[@]}"; do
  MODEL=${models[$i]}
  MODEL_TYPE=${model_types[$i]}
  
  for DATASET in "${datasets[@]}"; do
    echo "====================================================" | tee -a "$LOG_FILE"
    echo "Running prediction for:" | tee -a "$LOG_FILE"
    echo "Model: $MODEL" | tee -a "$LOG_FILE"
    echo "Model Type: $MODEL_TYPE" | tee -a "$LOG_FILE"
    echo "Dataset: $DATASET" | tee -a "$LOG_FILE"
    echo "Started at: $(date)" | tee -a "$LOG_FILE"
    
    # Run the prediction
    python calc_pred_LLM.py \
      --dataset="$DATASET" \
      --model_name="$MODEL" \
      --model_type="$MODEL_TYPE" \
      --batch_size="$BATCH_SIZE" \
      --max_length="$MAX_LENGTH" \
      --apply_projection="False" \
      --torch_dtype="$TORCH_DTYPE" \
      --device="$DEVICE" \
      --output_folder="$OUTPUT_DIR_PRED" 2>&1 | tee -a "$LOG_FILE"
    
    RUN_STATUS=$?
    if [ $RUN_STATUS -eq 0 ]; then
      echo "Run completed successfully at $(date)" | tee -a "$LOG_FILE"
      
      # Copy results from temporary to permanent location
      MODEL_NAME_SHORT=$(python -c "from config import map_model_name; print(map_model_name('$MODEL'))")
      SOURCE_FILE="$OUTPUT_DIR_PRED/${DATASET}_professions_${MODEL_NAME_SHORT}.csv"
      DEST_FILE="$PERMANENT_OUTPUT_FOLDER/${DATASET}_professions_${MODEL_NAME_SHORT}.csv"
      
      if [ ! -f "$SOURCE_FILE" ]; then
        echo "Warning: Source file not found: $SOURCE_FILE" | tee -a "$LOG_FILE"
        echo "Checking directory contents:" | tee -a "$LOG_FILE"
        ls -l "$OUTPUT_DIR_PRED" | tee -a "$LOG_FILE"
      else
        cp "$SOURCE_FILE" "$DEST_FILE"
        echo "Copied results to $DEST_FILE" | tee -a "$LOG_FILE"
      fi
    else
      echo "Run FAILED with exit code $RUN_STATUS at $(date)" | tee -a "$LOG_FILE"
    fi
    
    echo "Finished at: $(date)" | tee -a "$LOG_FILE"
    echo "" | tee -a "$LOG_FILE"
  done
done

# Copy all results to permanent storage as a backup
if [ -d "$OUTPUT_DIR_PRED" ]; then
    cp -r "$OUTPUT_DIR_PRED"/* "$PERMANENT_OUTPUT_FOLDER/"
    echo "Copied all results to $PERMANENT_OUTPUT_FOLDER/" | tee -a "$LOG_FILE"
else
    echo "Warning: Output directory not found: $OUTPUT_DIR_PRED" | tee -a "$LOG_FILE"
fi

echo "All prediction runs completed at $(date)" | tee -a "$LOG_FILE"
