# Tiny TinyResnet for CIFAR (32x32)

from __future__ import absolute_import

'''TinyResnet for cifar dataset.
https://github.com/HobbitLong/RepDistiller/blob/master/models/TinyResnet.py
Ported form
https://github.com/facebook/fb.TinyResnet.torch
and
https://github.com/pytorch/vision/blob/master/torchvision/models/TinyResnet.py
(c) YANG, Wei
'''
import torch.nn as nn
import torch.nn.functional as F
import math

from bypass.core.activation import ActivationForBypass

__all__ = ['TinyResnet']

class Sum(nn.Module):
    def forward(self,inp1,inp2,*args):
        return inp1+inp2
def conv3x3(in_planes, out_planes, stride=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu1=nn.ReLU()
        self.relu2=nn.ReLU()
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.downsample = downsample
        self.stride = stride

        self.sum=Sum()
    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu1(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)
        out=self.sum(out,residual)
        # out += residual
        out = self.relu2(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, planes * 4, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes * 4)
        self.relu1=nn.ReLU()
        self.relu2=nn.ReLU()
        self.relu3=nn.ReLU()
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu1(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu2(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu3(out)
        return out

class BypassBasicBlock(BasicBlock):
    def __init__(self, inplanes, planes, stride=1, downsample=None):

        super(BypassBasicBlock,self).__init__(inplanes, planes, stride=stride, downsample=downsample)
        self.relu1=ActivationForBypass(planes,nn.ReLU())
        # self.relu2=ActivationForBypass(planes,nn.ReLU())
    
class NaiveBypassBasicBlock(BasicBlock):
    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(NaiveBypassBasicBlock,self).__init__(inplanes, planes, stride=stride, downsample=downsample)
        self.bn_relu1=ActivationForBypass(planes,nn.Sequential(self.bn1,self.relu1))
        # self.bn_relu1.weight = self.bn1.weight
        # self.bn_relu1.bias = self.bn1.bias
        del(self.bn1)
        del(self.relu1)

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        # out = self.bn1(out)
        # out = self.relu1(out)
        out=self.bn_relu1(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)
        out=self.sum(out,residual)
        # out += residual
        out = self.relu2(out)
        return out
class BypassBNBasicblock(BasicBlock):
    def __init__(self, inplanes, planes, stride=1, downsample=None):
        if downsample:
            downsample = nn.Sequential(
                nn.Conv2d(inplanes, planes * self.expansion,
                          kernel_size=1, stride=stride, bias=False),
                ActivationForBypass(planes * self.expansion,nn.BatchNorm2d(planes * self.expansion)),
            )
        super(BypassBNBasicblock,self).__init__(inplanes, planes, stride=stride, downsample=downsample)
        self.bn1=ActivationForBypass(planes,self.bn1)
        self.bn2=ActivationForBypass(planes,self.bn2)
        if downsample:
            self.bn2.delta =  downsample[1].delta

class BypassBottleneck(Bottleneck):
    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BypassBottleneck,self).__init__(inplanes, planes, stride=stride, downsample=downsample)
        self.relu1=ActivationForBypass(planes,nn.ReLU())
        self.relu2=ActivationForBypass(planes,nn.ReLU())
        #self.relu3=ActivationForBypass(planes * 4,nn.ReLU())
class BypassBNBottleneck(Bottleneck):
    def __init__(self, inplanes, planes, stride=1, downsample=None):
        if downsample:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * self.expansion,
                          kernel_size=1, stride=stride, bias=False),
                ActivationForBypass(planes * self.expansion,nn.BatchNorm2d(planes * self.expansion)),
            )
        super(BypassBNBottleneck,self).__init__(inplanes, planes, stride=stride, downsample=downsample)
        self.bn1=ActivationForBypass(planes,nn.Sequential(self.bn1,nn.ReLU))
        self.bn2=ActivationForBypass(planes,nn.Sequential(self.bn2,nn.ReLU))
        self.bn3=ActivationForBypass(planes*4,self.bn3)
BLOCK_DICT={
    'basicblock':BasicBlock,
    'bypass_basicblock':BypassBasicBlock, # 이 경우, BN pruning 수행
    'bypassbn_basicblock':BypassBNBasicblock, # 이 경우, 각 conv의 W-pruning 진행,
    'naive_bypass_basicblock':NaiveBypassBasicBlock,
    'bottleneck':Bottleneck,
    'bypass_botteleneck':BypassBottleneck,
    'bypassbn_bottleneck':BypassBNBottleneck
}

class TinyResnet(nn.Module):

    def __init__(self, depth, num_filters, block_name='BasicBlock', num_classes=10): # 56, [16, 16, 32, 64]
        super(TinyResnet, self).__init__()
        if block_name.split('_')[-1].lower() == 'basicblock':
            assert (depth - 2) % 6 == 0, 'When use basicblock, depth should be 6n+2, e.g. 20, 32, 44, 56, 110, 1202'
            n = (depth - 2) // 6
            block = BLOCK_DICT[block_name.lower()]
        elif block_name.split('_')[-1].lower() == 'bottleneck':
            assert (depth - 2) % 9 == 0, 'When use bottleneck, depth should be 9n+2, e.g. 20, 29, 47, 56, 110, 1199'
            n = (depth - 2) // 9
            block = BLOCK_DICT[block_name.lower()]
        else:
            raise ValueError('block_name shoule be Basicblock or Bottleneck')

        self.inplanes = num_filters[0]
        self.conv1 = nn.Conv2d(3, num_filters[0], kernel_size=3, padding=1,
                               bias=False)
        self.bn1 = nn.BatchNorm2d(num_filters[0])
        self.relu = nn.ReLU()
        self.layer1 = self._make_layer(block, num_filters[1], n)
        self.layer2 = self._make_layer(block, num_filters[2], n, stride=2)
        self.layer3 = self._make_layer(block, num_filters[3], n, stride=2)
        self.avgpool = nn.AvgPool2d(8)
        self.fc = nn.Linear(num_filters[3] * block.expansion, num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes * block.expansion),
            )

        layers = list([])
        layers.append(block(self.inplanes, planes, stride, downsample))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes))

        return nn.Sequential(*layers)

    def forward(self, x, return_features=False):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)  # 32x32
        x = self.layer1(x)  # 32x32
        x = self.layer2(x)  # 16x16
        x = self.layer3(x)  # 8x8
        x = self.avgpool(x)
        features = x.view(x.size(0), -1)
        x = self.fc(features)

        if return_features:
            return x, features
        return x
class BypassTinyResnet(TinyResnet):
    def __init__(self, depth, num_filters, block_name='bypass_basicblock', num_classes=10):
        super(BypassTinyResnet,self).__init__(depth, num_filters, block_name=block_name, num_classes=num_classes)
        # self.relu=ActivationForBypass(num_filters[0],self.relu)
class BypassBNTinyResnet(TinyResnet):
    def __init__(self, depth, num_filters, block_name='BasicBlock', num_classes=10):
        super(BypassBNTinyResnet,self).__init__(depth, num_filters, block_name=block_name, num_classes=num_classes)
        self.bn1=ActivationForBypass(num_filters[0],self.bn1)
def TinyResnet8(num_classes):
    return TinyResnet(8, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet14(num_classes):
    return TinyResnet(14, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet20(num_classes):
    return TinyResnet(20, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet32(num_classes):
    return TinyResnet(32, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet44(num_classes):
    return TinyResnet(44, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet56(num_classes):
    return TinyResnet(56, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet110(num_classes):
    return TinyResnet(110, [16, 16, 32, 64], 'basicblock', num_classes=num_classes)


def TinyResnet8x4(num_classes):
    return TinyResnet(8, [32, 64, 128, 256], 'basicblock', num_classes=num_classes)


def TinyResnet32x4(num_classes):
    return TinyResnet(32, [32, 64, 128, 256], 'basicblock', num_classes=num_classes)

class cifar10_Bypassresnet56(BypassTinyResnet):
    def __init__(self,**kwargs):
        super(cifar10_Bypassresnet56,self).__init__(
            56, 
            [16, 16, 32, 64], 
            'bypass_basicblock', 
            num_classes=10,
            **kwargs
        )
class cifar10_BypassBNresnet56(BypassBNTinyResnet):
    def __init__(self,**kwargs):
        super(cifar10_BypassBNresnet56,self).__init__(
            56, 
            [16, 16, 32, 64], 
            'bypassbn_basicblock', 
            num_classes=10,
            **kwargs
        )
class cifar10_NaiveBypassBNresnet56(TinyResnet):
    def __init__(self,**kwargs):
        super(cifar10_NaiveBypassBNresnet56,self).__init__(
            56, 
            [16, 16, 32, 64], 
            'naive_bypass_basicblock', 
            num_classes=10,
            **kwargs
        )

if __name__ == '__main__':
    import torch
    a=torch.rand(1,3,32,32)
    model1=BypassTinyResnet(56, [16, 16, 32, 64], 'bypass_basicblock', num_classes=10)
    model2=cifar10_NaiveBypassBNresnet56()
    model3=TinyResnet(56, [16, 16, 32, 64], 'basicblock', num_classes=10)

    loaded_state_dict=torch.load('/workspace/jaeheun_MildPruning/save/mild_pruning_W/benchmark_cifar10_resnet56/baseline/save/cifar10_resnet56.pth')

    model1.load_state_dict(loaded_state_dict,strict=False)
    aa=model2.load_state_dict(loaded_state_dict,strict=False)
    model3.load_state_dict(loaded_state_dict,strict=True)
    print(torch.all(model1(a)==model2(a)))
    print(torch.all(model3(a)==model2(a)))
    print(torch.all(model1(a)==model3(a)))
    print(1)