#ifndef BOOSTGRAPH_H
#define BOOSTGRAPH_H

#include "MazeGraph.h"
#include <iostream>
#include <vector>
#include <string>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/graph_traits.hpp>
#include <boost/graph/graphviz.hpp>
#include <boost/graph/graph_utility.hpp>
#include <cstdlib>
#include <stdexcept>
#include <chrono>

//#include <mutex>



struct VertexProperties
{
    std::string node;           // Name or identifier of the node
    double credit;              // Credit associated with the node, can represent rewards or other measures
    int node_id;                // Unique ID for the node
    double eligibilityTrace;    // Eligibility trace used in reinforcement learning algorithms
};

struct EdgeProperties
{
    double probability; // Log probability associated with the edge
};

class BoostGraph
{
public:
    // Define a directed graph using Boost's adjacency_list with custom vertex and edge properties
    typedef boost::adjacency_list<
        boost::vecS,      // Store edges in a standard vector structure
        boost::vecS,      // Store vertices in a standard vector structure
        boost::directedS, // Define the graph as directed
        VertexProperties, // Attach VertexProperties to each vertex
        EdgeProperties    // Attach EdgeProperties to each edge
        >
        Graph;

    // Typedefs for easy access to graph-related types
    typedef boost::graph_traits<Graph>::vertex_descriptor Vertex;
    typedef boost::graph_traits<Graph>::edge_descriptor Edge;
    typedef boost::graph_traits<Graph>::edge_iterator edge_iterator;
    typedef boost::graph_traits<Graph>::vertex_iterator vertex_iterator;

    // Default constructor
    BoostGraph() {
        //std::lock_guard<std::mutex> lck (mutex_); // Optional mutex lock for thread safety (commented out)
    }

    // Constructor that initializes the graph from a MazeGraph object based on the state and optimality
    BoostGraph(const MazeGraph& turnModel, int state, bool optimal)
    {
        std::vector<std::string> rcppNodeList; // List of node names
        std::vector<MazeEdge> rcppEdgeList;    // List of edges
        std::unordered_map<std::string, std::vector<std::string>> turnNodes; // Map of turn nodes

        // Initialize nodes and paths based on the state and whether the graph is optimal
        if (state == 0)
        {
            rcppNodeList = turnModel.getNodeListS0(); // Get nodes for state 0
            rcppEdgeList = turnModel.getEdgeListS0(); // Get edges for state 0
            turnNodes = turnModel.getTurnNodeMapS0(); // Get turn nodes for state 0
            // Initialize paths for state 0
            Path0 = turnModel.getPath0S0();
            Path1 = turnModel.getPath1S0();
            Path2 = turnModel.getPath2S0();
            Path3 = turnModel.getPath3S0();
            Path4 = turnModel.getPath4S0();
            Path5 = turnModel.getPath5S0();
            if(!optimal)
            {
                // If not optimal, initialize additional paths for suboptimality in state 0
                Path0S1 = turnModel.getPath6S0();
                Path1S1 = turnModel.getPath7S0();
                Path2S1 = turnModel.getPath8S0();
                Path3S1 = turnModel.getPath9S0();
                Path4S1 = turnModel.getPath10S0();
                Path5S1 = turnModel.getPath11S0();
            }
        }
        else
        {
            if (optimal) {
                // For optimal state 1, initialize nodes, edges, and paths
                rcppNodeList = turnModel.getNodeListS1();
                rcppEdgeList = turnModel.getEdgeListS1();
                turnNodes = turnModel.getTurnNodeMapS1();
                Path0 = turnModel.getPath0S1();
                Path1 = turnModel.getPath1S1();
                Path2 = turnModel.getPath2S1();
                Path3 = turnModel.getPath3S1();
                Path4 = turnModel.getPath4S1();
                Path5 = turnModel.getPath5S1();
            }else{
                // Exit early if the model is not optimal and state is not 0
                return;
            }
        }

        // Add all nodes to the graph
        for (int i = 0; i < rcppNodeList.size(); i++)
        {
            addNode(rcppNodeList[i], i);
        }
        
        // Add all edges to the graph, converting the probabilities to log scale
        for (const MazeEdge& edge : rcppEdgeList) 
        {
            addEdge(edge.src, edge.dest, std::log(edge.prob));
        }
    }

    // Method to add a node to the graph with an optional initial credit value
    void addNode(const std::string &nodeName, int node_id, double initialCredit = 0.0)
    {
        Vertex v = boost::add_vertex(graph);  // Add a vertex to the graph
        graph[v].node = nodeName;             // Set the node name
        graph[v].credit = initialCredit;      // Set the initial credit value
        graph[v].node_id = node_id;           // Set the node ID
        graph[v].eligibilityTrace = 0;        // Initialize the eligibility trace to 0
    }

    // Method to add an edge between two nodes with a given probability
    void addEdge(const std::string &srcNodeName, const std::string &destNodeName, double probability)
    {
        //std::cout << "Adding edge between src:" <<srcNodeName << " and dest:" << destNodeName <<std::endl; 
        Vertex src = findNode(srcNodeName);    // Find the source node
        Vertex dest = findNode(destNodeName);  // Find the destination node
        if (src != boost::graph_traits<Graph>::null_vertex() &&
            dest != boost::graph_traits<Graph>::null_vertex())
        {
            // Add the edge if both nodes are valid
            boost::add_edge(src, dest, EdgeProperties{probability}, graph);
        }
    }

    // Method to find a node by its name
    Vertex findNode(const std::string &nodeName)
    {
        vertex_iterator v, vend;
        for (boost::tie(v, vend) = boost::vertices(graph); v != vend; ++v)
        {
            if (graph[*v].node == nodeName)
            {
                return *v;  // Return the vertex if the name matches
            }
        }
        return boost::graph_traits<Graph>::null_vertex(); // Return null_vertex if not found
    }

    // Method to find a node by its ID
    Vertex findNodeById(const int& id)
    {
        vertex_iterator v, vend;
        for (boost::tie(v, vend) = boost::vertices(graph); v != vend; ++v)
        {
            if (graph[*v].node_id == id)
            {
                return *v;  // Return the vertex if the ID matches
            }
        }
        return boost::graph_traits<Graph>::null_vertex(); // Return null_vertex if not found
    }

    // Method to find an edge between two vertices
    Edge findEdge(const Vertex &src , const Vertex &dest)
    {
        Edge edge;  
        if (src != boost::graph_traits<Graph>::null_vertex() && dest != boost::graph_traits<Graph>::null_vertex())
        {
            edge_iterator e, eend;
            for (boost::tie(e, eend) = boost::edges(graph); e != eend; ++e)
            {
                if (boost::source(*e, graph) == src && boost::target(*e, graph) == dest)
                {
                    edge = *e;  // Return the edge if source and target match
                    break;
                }
            }
        }
        return edge; // Return the found edge or a default edge if not found
    }

    // Method to get the probability of a given edge
    double getEdgeProbability(const Edge &edge)
    {
        return graph[edge].probability;
    }

    // Method to get the credit associated with a given node
    double getNodeCredits(const Vertex &node)
    {
        return graph[node].credit;
    }

    // Method to get the ID of a given node
    int getNodeId(const Vertex &node)
    {
        return graph[node].node_id;
    }

    // Method to get the name of a given node
    std::string getNodeName(const Vertex &node)
    {
        return graph[node].node;
    }

    // Method to set the credit for a given node
    void setNodeCredits(const Vertex &node, double credits)
    {
        graph[node].credit = credits;
    }

    // Method to get the eligibility trace of a given node
    double getEligibilityTrace(const Vertex &node)
    {
        return graph[node].eligibilityTrace;
    }

    // Method to set the eligibility trace for a given node
    void setEligibilityTrace(const Vertex &node, double trace)
    {
        //std::cout << "Setting eligibility trace of " << graph[node].node << " to " << trace << "\n";
        graph[node].eligibilityTrace = trace;
    }

    // Method to update all eligibility traces by multiplying them with a given factor
    void updateAllEligibilityTraces(double factor)
    {
        //std::cout << "Updating all eligibility trace by factor = " << factor << "\n";
        for (auto vi = boost::vertices(graph); vi.first != vi.second; ++vi.first) {
            graph[*vi.first].eligibilityTrace *= factor;

            if (std::isnan(graph[*vi.first].eligibilityTrace)) {
                // Throw an error if the eligibility trace is NaN
                throw std::runtime_error("Error nan eligibility trace");
            }
            if (std::isinf(graph[*vi.first].eligibilityTrace)) {
                // Throw an error if the eligibility trace is infinite
                throw std::runtime_error("Error inf eligibility trace");
            }
        }
    }
};

    void tdUpdateAllVertexCredits(double alpha, double td_err)
{
    // Iterate through all vertices in the graph
    for (auto vi = boost::vertices(graph); vi.first != vi.second; ++vi.first) {
        // Update the credit of each vertex based on the TD error and eligibility trace
        graph[*vi.first].credit += (alpha * td_err * graph[*vi.first].eligibilityTrace);
        
        // Check if the updated credit is infinite
        if (std::isinf(graph[*vi.first].credit)) {
            // std::cout << "Node: " << graph[*vi.first].node << " credit is infinity. Check" << std::endl;
            throw std::runtime_error("Error infinite credit val");
        }
        // Check if the updated credit is NaN
        if (std::isnan(graph[*vi.first].credit)) {
            throw std::runtime_error("Error nan node credits");
        }

        // Optionally check for excessively high credit values
        // if (graph[*vi.first].credit > 1e3) {
        //         std::cout << "Node: " << graph[*vi.first].node << " td_err=" << td_err << ", etrace=" << graph[*vi.first].eligibilityTrace << std::endl;
        //         throw std::runtime_error("Very node credits. Check");
        // }
    }
}

void setEdgeProbability(const Edge &edge, double probability)
{
    // Set the probability of the given edge
    graph[edge].probability = probability;
}

void setVertexCredits(const std::vector<double> &creditVector)
{
    // Iterate through the vertices
    for (auto vd = boost::vertices(graph).first; vd != boost::vertices(graph).second; ++vd) {
        // Access the vertex descriptor and update its credit using the provided credit vector
        Vertex vertex = *vd;
        graph[vertex].credit = creditVector[graph[vertex].node_id];         
    }
}

std::vector<double> getVertexCredits()
{
    // Create a vector to store the credits of all vertices
    std::vector<double> vertexCredits(boost::num_vertices(graph), 0.0);
    // Iterate through the vertices
    for (auto vd = boost::vertices(graph).first; vd != boost::vertices(graph).second; ++vd) {
        // Access the vertex descriptor and properties
        Vertex vertex = *vd;
        int nodeId = graph[vertex].node_id;
        double nodeCredits = graph[vertex].credit;
        vertexCredits[nodeId] = nodeCredits; // Store the credit value in the vector
    }
    return vertexCredits;
}

Vertex findParent(const Vertex &node)
{
    // Initialize variables for parent and target vertices
    Vertex parent = boost::graph_traits<Graph>::null_vertex();
    Vertex target = boost::graph_traits<Graph>::null_vertex();
    Graph::out_edge_iterator outIt, outEnd;
    vertex_iterator v, vend;

    // Iterate through all vertices
    for (boost::tie(v, vend) = boost::vertices(graph); v != vend; ++v)
    {
        // Iterate through the outgoing edges of each vertex
        for (boost::tie(outIt, outEnd) = boost::out_edges(*v, graph); outIt != outEnd; ++outIt)
        {
            target = boost::target(*outIt, graph); // Get the target vertex of the edge
            if(target == node)
            {
                parent = boost::source(*outIt, graph); // Set the parent vertex
                break; // Assuming each node has only one parent
            }
        }
    }
    return parent;
}

Vertex findMaxCreditSibling(const Vertex &node)
{
    double maxCredits = -std::numeric_limits<double>::infinity(); // Initialize with negative infinity
    Vertex maxCreditSibling = boost::graph_traits<Graph>::null_vertex();

    // Get the parent node of the given node
    Vertex parent = findParent(node);

    if (parent != boost::graph_traits<Graph>::null_vertex())
    {
        // Iterate through the adjacent nodes (siblings) of the parent
        boost::graph_traits<Graph>::adjacency_iterator adjIt, adjEnd;
        for (boost::tie(adjIt, adjEnd) = boost::adjacent_vertices(parent, graph); adjIt != adjEnd; ++adjIt)
        {
            double credits = getNodeCredits(*adjIt); // Get the credit of the sibling node
            if (credits > maxCredits)
            {
                maxCredits = credits; // Update the maximum credit and sibling vertex
                maxCreditSibling = *adjIt;
            }
        }
    }
    return maxCreditSibling;
}

bool isTerminalVertex(const std::string& vertexName) {
    Vertex v = findNode(vertexName); // Get the vertex descriptor for the given vertex name

    // Get the out-degree of the vertex
    int outDegree = out_degree(v, graph);

    // A vertex is terminal if its out-degree is 0
    return outDegree == 0;
}

std::vector<Vertex> getAllVertices() const {
    std::vector<Vertex> vertices;
    // Iterate through all vertices and add them to the vector
    for (auto vi = boost::vertices(graph); vi.first != vi.second; ++vi.first) {
        vertices.push_back(*vi.first);
    }
    return vertices;
}

Vertex getChildWithMaxCredit(const std::string& parentVertexName) {
    Vertex parentVertex = findNode(parentVertexName); // Get the vertex descriptor of the parent node

    // Initialize variables to keep track of the child with max credit
    Vertex maxCreditChild = boost::graph_traits<Graph>::null_vertex();
    double maxCredit = -std::numeric_limits<double>::infinity(); // Initialize with negative infinity

    // Iterate through the outgoing edges of the parent vertex
    Graph::out_edge_iterator ei, ei_end;
    for (boost::tie(ei, ei_end) = out_edges(parentVertex, graph); ei != ei_end; ++ei) {
        Vertex childVertex = boost::target(*ei, graph); // Get the target vertex of the outgoing edge

        // Get the credit of the child node
        double childCredit = getNodeCredits(childVertex);

        // Update max credit and maxCreditChild if the child has higher credit
        if (childCredit > maxCredit) {
            maxCredit = childCredit;
            maxCreditChild = childVertex;
        }
    }
    return maxCreditChild;
}

// Function to print the credits of all nodes in the graph
void printNodeCredits() const {
    vertex_iterator vi, vi_end;
    for (boost::tie(vi, vi_end) = boost::vertices(graph); vi != vi_end; ++vi) {
        Vertex v = *vi;
        // Output the node identifier and its associated credit value
        std::cout << graph[v].node << ": " << graph[v].credit << ",";
    }
    std::cout << std::endl; // End the line after printing all nodes
}

// Function to print the probabilities of all outgoing edges for each node
void printNodeProbabilities() {
    for (vertex_iterator it = vertices(graph).first; it != vertices(graph).second; ++it) {
        Vertex node = *it;
        std::vector<Vertex> children; // Stores child nodes
        std::vector<double> values;    // Stores credit values of child nodes

        // Iterate over outgoing edges from the current node
        Graph::out_edge_iterator ei, ei_end;
        for (tie(ei, ei_end) = out_edges(node, graph); ei != ei_end; ++ei) {
            Vertex childVertex = target(*ei, graph); // Get the target vertex of the outgoing edge                
            children.push_back(childVertex);
            values.push_back(graph[childVertex].credit); // Collect child node credits
        }

        // Calculate and print the probabilities for each edge
        for (Vertex child : children) {
            double logsumexp_value = logsumexp(values); // Compute logsumexp for numerical stability
            double prob = exp(graph[child].credit - logsumexp_value); // Compute the probability
            std::cout << graph[node].node << "-->" << graph[child].node << ", prob=" << prob << "; ";
        }
    }
    std::cout << std::endl; // End the line after printing all probabilities
}

// Function to print the eligibility traces of all nodes in the graph
void printNodeEligibilityTraces() const {
    std::cout << "Node eligibility trace:\n";
    vertex_iterator vi, vi_end;
    for (boost::tie(vi, vi_end) = boost::vertices(graph); vi != vi_end; ++vi) {
        Vertex v = *vi;
        // Output the node identifier and its associated eligibility trace value
        std::cout << "Vertex " << graph[v].node << ": " << graph[v].eligibilityTrace << ",";
    }
    std::cout << std::endl; // End the line after printing all nodes
}

// Function to decay the credit values of all nodes by a factor of gamma
void decayCredits(double gamma) {
    for (vertex_iterator it = vertices(graph).first; it != vertices(graph).second; ++it) {
        Vertex node = *it;
        graph[node].credit *= gamma; // Scale the credit value
    }
}

// Function to find siblings of a given node (i.e., nodes with the same parent)
std::vector<Vertex> findSiblings(Vertex node) {
    std::vector<Vertex> siblings;
    Vertex parent = findParent(node); // Find the parent of the node

    // Iterate over outgoing edges from the parent node to get siblings
    Graph::out_edge_iterator it, end;
    for (boost::tie(it, end) = out_edges(parent, graph); it != end; ++it) {
        Vertex sibling = boost::target(*it, graph);
        siblings.push_back(sibling);
    }

    return siblings; // Return the list of sibling nodes
}

// Function to get all outgoing edges from a given node
std::vector<Edge> getOutGoingEdges(Vertex node) {
    std::vector<Edge> outgoingEdges;
    Graph::out_edge_iterator edgeIt, edgeEnd;
    for (tie(edgeIt, edgeEnd) = out_edges(node, graph); edgeIt != edgeEnd; ++edgeIt) {
        outgoingEdges.push_back(*edgeIt); // Collect outgoing edges
    }
    return outgoingEdges; // Return the list of outgoing edges
}

// Function to compute softmax probabilities for a vector of values
std::vector<double> softmax(const std::vector<double>& values) {
    std::vector<double> probabilities;
    double sumExp = 0.0;

    // Compute the sum of exponentials of all values
    for (double value : values) {
        sumExp += std::exp(value);
    }

    // Compute the softmax probability for each value
    for (double value : values) {
        probabilities.push_back(std::exp(value) / sumExp);
    }

    return probabilities; // Return the computed probabilities
}

// Function to check if a vertex has outgoing edges
bool has_outgoing_edges(Vertex v) {
    std::vector<Vertex> children;
    Graph::out_edge_iterator edgeIt, edgeEnd;
    for (tie(edgeIt, edgeEnd) = out_edges(v, graph); edgeIt != edgeEnd; ++edgeIt) {
        Vertex child = target(*edgeIt, graph);
        children.push_back(child); // Collect children
    }
    return children.size() != 0; // Return true if there are outgoing edges
}

// Function to sample a child vertex based on softmax probabilities of outgoing edges
Vertex sampleChild(Vertex parent) {
    std::vector<double> edgeCredits;
    std::vector<Vertex> children;

    // Collect outgoing edges and their associated credits
    Graph::out_edge_iterator edgeIt, edgeEnd;
    for (tie(edgeIt, edgeEnd) = out_edges(parent, graph); edgeIt != edgeEnd; ++edgeIt) {
        Vertex child = target(*edgeIt, graph);
        children.push_back(child);
        edgeCredits.push_back(graph[child].credit);
    }

    // Compute the softmax probabilities for the edge credits
    std::vector<double> probs = softmax(edgeCredits);

    // Sample one child vertex based on the softmax probabilities
    unsigned int seed = std::chrono::system_clock::now().time_since_epoch().count();
    std::mt19937 gen(seed);
    std::discrete_distribution<> dist(probs.begin(), probs.end());
    Vertex sampled = children[dist(gen)];
    return sampled; // Return the sampled child vertex
}

// Function to update edge probabilities using softmax of node credits
void updateEdgeProbabilitiesSoftmax() {
    for (vertex_iterator it = vertices(graph).first; it != vertices(graph).second; ++it) {
        Vertex node = *it;
        std::vector<Vertex> children;
        std::vector<double> values;

        // Collect child nodes and their credits
        Graph::out_edge_iterator ei, ei_end;
        for (tie(ei, ei_end) = out_edges(node, graph); ei != ei_end; ++ei) {
            Vertex childVertex = target(*ei, graph);
            children.push_back(childVertex);
            values.push_back(graph[childVertex].credit);
        }

        // Update the probability for each edge based on the softmax of child credits
        for (Vertex child : children) {
            Edge edge = findEdge(node, child);
            double logsumexp_value = logsumexp(values); // Compute logsumexp for numerical stability
            graph[edge].probability = graph[child].credit - logsumexp_value;

            // Check for invalid probabilities
            if (std::isnan(graph[edge].probability)) {
                std::cout << "Node credits: ";
                for (const double& p : values) {
                    std::cout << p << " ";
                }
                std::cout << std::endl;
                std::cout << "Edge src: " << graph[node].node << " dest: " << graph[child].node << " logprob is nan. Check" << std::endl;
                throw std::runtime_error("Error nan probability");
            }
            if (std::isinf(graph[edge].probability)) {
                throw std::runtime_error("Error infinite probability");
            }
        }
    }
}

// Function to compute logsumexp for numerical stability
double logsumexp(const std::vector<double>& values) {
    if (values.empty()) {
        return -std::numeric_limits<double>::infinity();
    }

    double max_val = values[0];
    for (double val : values) {
        if (val > max_val) {
            max_val = val;
        }
    }

    double sum_exp = 0.0;
    for (double val : values) {
        sum_exp += std::exp(val - max_val);
    }

    return max_val + std::log(sum_exp);
}

// Function to get the turns corresponding to a specific path and state
std::vector<std::string> getTurnsFromPaths(int path, int state, bool optimal) {
    std::vector<std::string> turns;
    if (!optimal && state == 1) {
        // Select turns based on the path and state for non-optimal cases
        switch (path) {
            case 0: turns = Path0S1; break;
            case 1: turns = Path1S1; break;
            case 2: turns = Path2S1; break;
            case 3: turns = Path3S1; break;
            case 4: turns = Path4S1; break;
            case 5: turns = Path5S1; break;
        }
    } else {
        // Select turns based on the path for optimal cases
        switch (path) {
            case 0: turns = Path0; break;
            case 1: turns = Path1; break;
            case 2: turns = Path2; break;
            case 3: turns = Path3; break;
            case 4: turns = Path4; break;
            case 5: turns = Path5; break;
        }
    }

    return turns; // Return the list of turns
}

// Function to get the path number from a list of turns and a root node
int getPathFromTurns(std::vector<std::string> turns, Vertex rootNode, bool optimal) {
    int path = -1;
    if (!optimal && getNodeName(rootNode) == "I") {
        // Match turns to a path number for non-optimal cases
        if (turns == Path0S1) path = 0;
        else if (turns == Path1S1) path = 1;
        else if (turns == Path2S1) path = 2;
        else if (turns == Path3S1) path = 3;
        else if (turns == Path4S1) path = 4;
        else if (turns == Path5S1) path = 5;
    } else {
        // Match turns to a path number for optimal cases
        if (turns == Path0) path = 0;
        else if (turns == Path1) path = 1;
        else if (turns == Path2) path = 2;
        else if (turns == Path3) path = 3;
        else if (turns == Path4) path = 4;
        else if (turns == Path5) path = 5;
    }

    return path; // Return the path number
}

// Function to get a vector of node names for a specific node
std::vector<std::string> getTurnNodes(std::string nodeName) {
    std::vector<std::string> turnNodesVec;
    auto it = turnNodes.find(nodeName);

    if (it != turnNodes.end() && !it->second.empty()) {
        // Return the vector associated with the nodeName
        turnNodesVec = it->second;
    }

    return turnNodesVec; // Return the list of turn nodes
}


    // void printGraph()
    // {
    //     vertex_iterator v, vend;
    //     edge_iterator e, eend;
    //     for (boost::tie(v, vend) = boost::vertices(graph); v != vend; ++v)
    //     {
    //         std::cout << "Node: " << graph[*v].node << " (Credit: " << graph[*v].credit << ")\n";
    //         for (boost::tie(e, eend) = boost::out_edges(*v, graph); e != eend; ++e)
    //         {
    //             Vertex src = boost::source(*e, graph);
    //             Vertex dest = boost::target(*e, graph);
    //             std::cout << "  Edge to: " << graph[dest].node << " (Probability: " << graph[*e].probability << ")\n";
    //         }
    //     }
    // }

    void printGraph()
    {
        boost::dynamic_properties dp;
        dp.property("node_id", boost::get(&VertexProperties::node_id, graph));
        dp.property("label", boost::get(&VertexProperties::node, graph));
        dp.property("weight", boost::get(&EdgeProperties::probability, graph));
        std::ofstream graph_file_out("./out.gv");
        boost::write_graphviz_dp(graph_file_out, graph, dp);


    }

    void resetNodeCredits() {

        // Iterate over all vertices and reset credits
        vertex_iterator v, vend;
        for (boost::tie(v, vend) = boost::vertices(graph); v != vend; ++v)
        {
            graph[*v].credit = 0.0; // Resetting the credits (modify based on your actual property)
        }
    }

    


private:
    Graph graph;
    std::vector<std::string> Path0;
    std::vector<std::string> Path1;
    std::vector<std::string> Path2;
    std::vector<std::string> Path3;
    std::vector<std::string> Path4;
    std::vector<std::string> Path5;
    std::vector<std::string> Path0S1;
    std::vector<std::string> Path1S1;
    std::vector<std::string> Path2S1;
    std::vector<std::string> Path3S1;
    std::vector<std::string> Path4S1;
    std::vector<std::string> Path5S1;
    std::unordered_map<std::string, std::vector<std::string>> turnNodes;
    //std::mutex mutex_;



};



#endif