import numpy as np
import matplotlib.pyplot as plt
import argparse

# Argument parsing
parser = argparse.ArgumentParser()
parser.add_argument('--dataset', type=str, default='ucfcrime', help='Dataset name')
parser.add_argument('--normalize', action='store_true', help='Apply per-class max normalization')
args = parser.parse_args()

# Dataset dictionary
data = {
    'ucfcrime': {
        'Abuse':        [68.02, 70.09, 70.74],
        'Arrest':       [72.21, 47.09, 75.05],
        'Arson':        [65.49, 66.75, 72.68],
        'Assault':      [56.44, 72.03, 72.58],
        'Burglary':     [68.02, 65.88, 74.99],
        'Explosion':    [56.33, 57.64, 63.46],
        'Fighting':     [58.14, 79.27, 62.81],
        'RoadAccident': [57.41, 59.11, 66.32],
        'Robbery':      [76.03, 62.39, 76.29],
        'Stealing':     [74.91, 61.51, 75.43],
        'Shooting':     [60.95, 38.42, 62.26],
        'Shoplifting':  [64.27, 73.29, 85.72],
        'Vandalism':    [66.89, 63.05, 69.23],
    },
    'shang': {
        'Car':             [70.07, 74.76, 74.83],
        'Chasing':         [94.49, 84.36, 91.33],
        'Fall':            [72.96, 65.30, 82.17],
        'Fighting':        [76.91, 63.48, 83.64],
        'Monocycle':       [67.23, 75.32, 75.46],
        'Robbery':         [76.74, 87.26, 90.43],
        'Running':         [37.95, 60.95, 60.78],
        'Skateboard':      [76.04, 78.29, 82.38],
        'Throwing_object': [89.63, 83.14, 91.95],
        'Vehicle':         [79.39, 67.38, 79.87],
        'Vaudeville':      [44.04, 53.66, 53.61],
    },
    'msad': {
        'Assault':          [54.78, 66.03, 58.73],
        'Explosion':        [50.86, 66.25, 57.20],
        'Fighting':         [71.74, 79.75, 81.14],
        'Fire':             [71.97, 49.44, 71.23],
        'Object_falling':   [90.52, 75.92, 90.92],
        'People_falling':   [60.64, 42.50, 56.93],
        'Robbery':          [68.10, 66.90, 70.95],
        'Shooting':         [71.20, 86.87, 77.88],
        'Traffic_accident': [62.23, 70.08, 70.93],
        'Vandalism':        [83.40, 75.82, 87.05],
        'Water_incident':   [97.95, 88.93, 98.75],
    },
    'xd': {
        'Fighting':      [79.59, 67.81, 84.76],
        'Shooting':      [54.59, 42.94, 61.99],
        'Riot':          [97.62, 86.07, 97.67],
        'Abuse':         [59.42, 54.49, 64.96],
        'Car Accident':  [50.83, 32.53, 51.70],
        'Explosion':     [64.32, 39.22, 68.54],
    }
}
data = data[args.dataset]
classes = list(data.keys())
num_vars = len(classes)

# Normalize values per class if requested
rgb_vals, event_vals, fusion_vals = [], [], []

for cls in classes:
    rgb, event, fusion = data[cls]
    if args.normalize:
        max_val = max(rgb, event, fusion)
        rgb_vals.append(rgb / max_val * 100)
        event_vals.append(event / max_val * 100)
        fusion_vals.append(fusion / max_val * 100)
    else:
        rgb_vals.append(rgb)
        event_vals.append(event)
        fusion_vals.append(fusion)

# Close the polygon
rgb_vals += rgb_vals[:1]
event_vals += event_vals[:1]
fusion_vals += fusion_vals[:1]

# Radar chart setup
angles = np.linspace(0, 2 * np.pi, num_vars, endpoint=False).tolist()
angles += angles[:1]

fig, ax = plt.subplots(figsize=(8, 8), subplot_kw=dict(polar=True))

ax.plot(angles, rgb_vals, label="Image", marker='o')
ax.fill(angles, rgb_vals, alpha=0.25)
ax.plot(angles, event_vals, label="Event", marker='o')
ax.fill(angles, event_vals, alpha=0.25)
ax.plot(angles, fusion_vals, label="Fusion", marker='o')
ax.fill(angles, fusion_vals, alpha=0.25)

ax.set_xticks(angles[:-1])
ax.set_xticklabels(classes, fontsize=25)
ax.tick_params(axis='y', labelsize=12)

ax.grid(color='#606060', linewidth=1.0, linestyle='--')
# ax.legend(loc='upper right', bbox_to_anchor=(1.1, 1.1), fontsize=12)

plt.savefig('figure2.png', dpi=300, bbox_inches='tight')
