import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid
import base64
import io
import sys
import random
from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, process_images, get_model_name_from_path
import datasets as hf_datasets

from PIL import Image
import math


def split_list(lst, n):
    """Split a list into n (roughly) equal-sized chunks"""
    chunk_size = math.ceil(len(lst) / n)  # integer division
    return [lst[i:i+chunk_size] for i in range(0, len(lst), chunk_size)]


def get_chunk(lst, n, k):
    chunks = split_list(lst, n)
    return chunks[k]


def bytes_to_PIL_image(img_buffer):
    img_io = io.BytesIO(img_buffer)
    img_io.seek(0)
    image = Image.open(img_io).convert('RGB')
    return image


def input_data(path):
    try:
        parquet_list = [file for file in os.listdir(path) if file.endswith('.parquet')]
        if len(parquet_list) > 0:
            hf_data = hf_datasets.load_dataset(path)['train'].cast_column("image", hf_datasets.Image(decode=False))
            data = []
            for line in hf_data:
                line['image'] = bytes_to_PIL_image(line['image']['bytes'])
                data.append(line)
            return data
    except:
        try:
            with open(path, 'r') as f:
                data = json.load(f)
        except:
            data = []
            for line in open(path, 'r'):
                data.append(json.loads(line))
        if type(data) == dict:
            data = [data]
    return data


def eval_model(args):
    # Model
    disable_torch_init()
    model_path = os.path.expanduser(args.model_path)
    
    # import ipdb; ipdb.set_trace()
    if args.model_base is None or len(args.model_base) < 5:
        model_base = None
        model_name = 'llava-v1.5-7b'
    else:
        model_base = args.model_base
        model_name = 'llava-v1.5-7b-lora'
    tokenizer, model, image_processor, context_len = load_pretrained_model(model_path, model_base, model_name, device_map={"": 'cuda'})
    
    questions = input_data(os.path.expanduser(args.question_file))
    questions = get_chunk(questions, args.num_chunks, args.chunk_idx)
    answers_file = os.path.expanduser(args.answers_file)
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    
    ans_data = []

    question_idx=0
    random.shuffle(questions)
    for line in tqdm(questions):
        qs = line['question']
        cur_prompt = qs

        if line['visual_input'] == 1:
            if model.config.mm_use_im_start_end:
                qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
            else:
                qs = DEFAULT_IMAGE_TOKEN + '\n' + qs

            image_path = os.path.join(args.image_path, line['filename'][1:])
            image = Image.open(image_path).convert('RGB')
            image_tensor = process_images([image], image_processor, model.config)[0]

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        input_ids = tokenizer_image_token(prompt, tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()
    

        with torch.inference_mode():
            output_ids = model.generate(
                input_ids,
                images=image_tensor.unsqueeze(0).half().cuda() if line['visual_input'] == 1 else None,
                image_sizes=[image.size] if line['visual_input'] == 1 else None,
                do_sample=True if args.temperature > 0 else False,
                temperature=args.temperature,
                top_p=args.top_p,
                num_beams=args.num_beams,
                # no_repeat_ngram_size=3,
                max_new_tokens=1024,
                use_cache=True)

        outputs = tokenizer.batch_decode(output_ids, skip_special_tokens=True)[0].strip()
        line['model_prediction'] = outputs

        print("-"*20)
        print(outputs)
        print("-"*20)

        ans_id = shortuuid.uuid()
        ans_data.append(line)
        question_idx += 1

    with open(answers_file, "w") as f:
        json.dump(ans_data, f)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model-path", type=str, default="facebook/opt-350m")
    parser.add_argument("--model-base", type=str, default=None)
    parser.add_argument("--image-folder", type=str, default="")
    parser.add_argument("--question-file", type=str, default="tables/question.jsonl")
    parser.add_argument("--image-path", type=str, default="")
    parser.add_argument("--answers-file", type=str, default="answer.json")
    parser.add_argument("--conv-mode", type=str, default="llava_v1")
    parser.add_argument("--num-chunks", type=int, default=1)
    parser.add_argument("--chunk-idx", type=int, default=0)
    parser.add_argument("--temperature", type=float, default=0.2)
    parser.add_argument("--top_p", type=float, default=None)
    parser.add_argument("--num_beams", type=int, default=3)
    args = parser.parse_args()

    print(args.conv_mode)

    eval_model(args)
