# LinearizeLLM Framework

**LinearizeLLM** is an agent-based framework that automates the reformulation of nonlinear optimization problems (NLPs) into linear optimization problems (LPs)/mixed integer linear optimization problems (MILPs) using specialized LLM agents. Each nonlinear pattern (e.g., bilinear products) is assigned to a reformulation agent that derives exact linear reformulations.

## 📋 Prerequisites

Before using the LinearizeLLM framework, ensure you have the following:

### Required Software
- **Python 3.8+** - Core programming language
- **Gurobi Optimization Solver** - Commercial license required for optimization

### API Keys
**⚠️ IMPORTANT**: You must provide your own API keys for LLM providers:
- **OpenAI API Key** - For GPT-4o, o3, and other OpenAI models
- **Google API Key** - For Gemini models (Gemini-2.5-Flash, Gemini-2.5-Pro)

**No default API keys are provided** - you must obtain your own keys from the respective providers.

### Setting Up API Keys

You can set your API keys in several ways:

1. **Environment Variables** (Recommended):
   ```bash
   # For OpenAI
   export OPENAI_API_KEY="your-openai-api-key"
   
   # For Google
   export GOOGLE_API_KEY="your-google-api-key"
   ```

2. **Interactive Setup**: The system will prompt you to enter API keys when needed

3. **Config File**: Keys are automatically saved to `~/.linearizellm_api_keys.json` for future use

### Prompt Templates
The framework uses prompt templates located in:
```
configs/prompts/
├── absolute_value_pattern_prompt.txt
├── bilinear_pattern_prompt.txt
├── code_conversion_prompt.txt
├── code_validation_prompt.txt
├── max_pattern_prompt.txt
├── min_pattern_prompt.txt
├── monotone_transformation_prompt.txt
├── pattern_detection_prompt.txt
└── quotient_pattern_prompt.txt
```

These templates guide the LLM in:
- **Pattern Detection** - Identifying nonlinear patterns in particulat NLPs
- **Reformulation** - Linearize nonlinear patterns
- **Code Generation** - Creating executable Gurobi Python code
- **Validation** - Ensuring generated code matches the original model

## 🚀 Quick Start

### Installation Steps

### Installation

1. **Clone the repository:**
   ```bash
   git clone <repository-url>
   cd LinearizeLLM
   ```

2. **Create and activate a virtual environment (recommended):**
   ```bash
   # Using conda
   conda create -n linearizellm python=3.9
   conda activate linearizellm
   
   # Or using venv
   python -m venv linearizellm_env
   source linearizellm_env/bin/activate  # On Windows: linearizellm_env\Scripts\activate
   ```

3. **Install dependencies:**
   ```bash
   pip install -r requirements.txt
   ```

4. **Install Gurobi:**
   ```bash
   # Download and install Gurobi from https://www.gurobi.com/downloads/
   # Then install the Python package:
   pip install gurobipy
   ```

5. **Set up API keys:**
   ```bash
   # For OpenAI
   export OPENAI_API_KEY="your-openai-api-key"
   
   # For Google Gemini
   export GOOGLE_API_KEY="your-google-api-key"
   ```

### Quick Test

```bash
# Run a quick test with the interactive menu
python run_experiments.py --quick-test

# Or run a specific problem
python run_linearizellm.py --file media_selection_nonlinear_bincon
```

## 📖 Documentation

- **[Workflow Documentation](docs/WORKFLOW_README.md)** - Detailed explanation of the LinearizeLLM framework's architecture and workflow steps
- **[LLM Model Manager](docs/LLM_MODEL_MANAGER_README.md)** - Guide for using different LLM providers and configurations

## 🏗️ Architecture

The LinearizeLLM framework follows a modular architecture:

```
src/
├── core/           # Core workflow components
├── agents/         # Specialized reformulation agents
├── utils/          # Utilities and helpers
└── scripts/        # Execution scripts
```

## 🔄 Workflow Overview

The LinearizeLLM framework processes optimization problems through 6 main steps:

1. **LaTeX Model Loading** - Extract mathematical model and parameters
2. **Pattern Detection** - Identify nonlinear patterns using LLMs
3. **Linearization** - Apply appropriate linearization techniques
4. **Code Generation** - Generate executable Gurobi Python code
5. **Validation** - Validate generated code against original model
6. **Execution** - Run optimization and capture results

## 🧪 Context Experiments

The framework includes experimental capabilities to test how different information levels affect LLM performance:

### Experiment Scenarios

- **`no_context`**: Only the LaTeX mathematical model is provided
  - Tests baseline performance with minimal information
  - Simulates real-world scenarios where only equations are available

- **`partial_info`**: LaTeX model + decision variable context
  - Provides information about which symbols are decision variables
  - Tests impact of knowing variable types without concrete values

### Running Context Experiments

```bash
# Interactive menu for context experiments
python run_experiments.py --interactive

# Quick test with both scenarios
python run_experiments.py --quick-test

# Full experimental suite (all problems, all scenarios, all seeds)
python run_experiments.py --interactive
# Then select option 4: Full experimental suite
```

### Results Structure

Context experiment results are saved in:
```
data/context_experiment_results/
├── {llm_model}/
│   ├── no_context/
│   │   └── {problem_name}/
│   │       └── seed_{N}/
│   └── partial_info/
│       └── {problem_name}/
│           └── seed_{N}/
```

## 🎯 Supported Nonlinear Patterns

- **Bilinear Terms** (`x * y`) → Linearization techniques
- **Min/Max Functions** (`min(x,y)`, `max(x,y)`) → Linearization techniques
- **Absolute Value** (`|x|`) → Linearization techniques
- **Quotient Expressions** (`x/y`) → Linearization techniques
- **Monotone Transformations** (e.g., `log(x)`, `exp(x)`) → Linearization techniques

## 🤖 LLM Integration

### Supported LLM Providers

- **OpenAI**: GPT-4o, GPT-4o-mini, o3, GPT-4, GPT-3.5-turbo
- **Google**: Gemini-2.5-Pro, Gemini-2.5-Flash

### Available Problems

The framework includes 20 optimization problems for testing:

1. `aircraft_monotone_max`
2. `blend_problem_abs`
3. `blend_problem_frac`
4. `diet_problem_min_abs`
5. `diet_problem_monotone`
6. `diet_problem_nonlinear_frac`
7. `knapsack_problem_nonlinear_min_1`
8. `knapsack_problem_nonlinear_min_2`
9. `media_selection_nonlinear_binbin`
10. `media_selection_nonlinear_bincon`
11. `multi_nonlinear_abs`
12. `netasgn_nonlinear_abs`
13. `netasgn_nonlinear_max`
14. `netmcol_nonlinear_bincon`
15. `netmcol_nonlinear_frac`
16. `nltrans_nonlinear_bincon`
17. `nltrans_nonlinear_max`
18. `prod_nonlinear_bincon`
19. `prod_nonlinear_max`
20. `revenue_maximization_nonlinear_bincon`

### Model Selection

You can now easily select which model to use at the beginning of execution:

```bash
# Interactive model selection
python src/scripts/run_linearizellm_data.py --file problem.tex

# Use specific model directly
python src/scripts/run_linearizellm_data.py --file problem.tex --model gemini-2.5-pro

# Use custom configuration file
python src/scripts/run_linearizellm_data.py --file problem.tex --model-config configs/gemini_25_pro_config.json

# Standalone model selector
python src/utils/model_selector.py
```

## 📁 Project Structure

```
LinearizeLLM/
├── src/                    # Source code
│   ├── core/              # Core workflow components
│   ├── agents/            # Specialized reformulation agents
│   ├── utils/             # Utilities and helpers
│   └── scripts/           # Execution scripts
├── data/                  # Data and results
│   ├── LinearizeLLM_data/     # Input problems (20 optimization problems)
│   │   └── instances_linearizellm/
│   ├── context_experiment_results/  # Context experiment outputs
│   └── results/           # Standard results
├── docs/                  # Documentation
├── configs/               # Configuration files
│   └── prompts/           # LLM prompt templates
└── requirements.txt       # Python dependencies
```

## 🔧 Troubleshooting

### Common Issues

1. **Gurobi Installation Issues:**
   ```bash
   # Ensure Gurobi is properly installed
   python -c "import gurobipy; print('Gurobi OK')"
   
   # If not working, try:
   pip install gurobipy --upgrade
   ```

2. **API Key Issues:**
   ```bash
   # Check if API keys are set
   echo $OPENAI_API_KEY
   echo $GOOGLE_API_KEY
   
   # If not set, the system will prompt you to enter them
   # Or set them manually:
   export OPENAI_API_KEY="your-openai-key"
   export GOOGLE_API_KEY="your-google-key"
   ```

3. **LangChain Import Errors:**
   ```bash
   # Reinstall LangChain packages
   pip install langchain-openai langchain-google-genai --upgrade
   ```

4. **Permission Issues:**
   ```bash
   # On Windows, run PowerShell as Administrator
   # On Linux/Mac, check file permissions
   chmod +x run_experiments.py
   ```

### Getting Help

- Check the [Workflow Documentation](docs/WORKFLOW_README.md) for detailed explanations
- Review [LLM Model Manager](docs/LLM_MODEL_MANAGER_README.md) for API configuration
- Examine [Experimental Setup](EXPERIMENTAL_SETUP_README.md) for context experiments

## 🛠️ Usage

### Running Experiments

The framework provides multiple ways to run experiments:

#### 1. Interactive Menu (Recommended)

```bash
python run_experiments.py --interactive
```

This launches an interactive menu with the following options:

- **Single LinearizeLLM run** - Process individual problems
- **Single context experiment** - Test information level effects
- **Multiple context experiments** - Run across multiple problems
- **Full experimental suite** - Complete evaluation
- **Quick test** - Fast validation

#### 2. Context Experiments

Context experiments test how different information levels affect LLM performance:

- **`no_context`**: Only LaTeX model (minimal information)
- **`partial_info`**: LaTeX + decision variable types (medium information)

```bash
# Quick test with both scenarios
python run_experiments.py --quick-test

# Quick test with specific LLM model
python run_experiments.py --quick-test --llm-model o3
```

#### 3. Direct Problem Processing

```bash
# Process specific problem
python run_linearizellm.py --file problem_name

# Use specific LLM model
python run_linearizellm.py --file problem_name --model gemini-2.5-flash
```

#### 4. Interactive Menu Features

The interactive menu (`run_experiments.py`) provides:

**Main Menu Options:**
- **Option 1**: Single LinearizeLLM run (like `run_linearizellm.py`)
- **Option 2**: Single context experiment (test information level effects)
- **Option 3**: Multiple context experiments (across multiple problems)
- **Option 4**: Full experimental suite (all problems, scenarios, seeds)
- **Option 5**: Quick test (single instance, both scenarios, seed 1)

**Context Scenario Selection:**
- **Option 1**: `no_context` only (minimal information)
- **Option 2**: `partial_info` only (medium information)
- **Option 3**: Both scenarios (compare performance)

**Seed Configuration:**
- **Option 1**: Single seed (seed 1)
- **Option 2**: Multiple seeds (1, 2, 3)
- **Option 3**: All seeds (1, 2, 3, 4, 5)
- **Option 4**: Custom seeds

**LLM Model Selection:**
- **Option 1**: gemini-2.5-flash (recommended)
- **Option 2**: o3 (OpenAI)
python run_linearizellm.py --file problem_name.tex

# Use different LLM model
python run_linearizellm.py --file problem_name.tex --model gpt-4o

# Process multiple problems
python run_linearizellm.py --files "problem1.tex,problem2.tex"

# Process first N problems
python run_linearizellm.py --first 5
```

### Programmatic

```python
from src.core.agent_pipeline import LinearizeLLMWorkflow

workflow = LinearizeLLMWorkflow(
    tex_path="problem.tex",
    problem_id="my_problem",
    llm_model="gpt-4o"
)

results = workflow.run(verbose=True)
```

## 🧪 Running Experiments

The LinearizeLLM framework includes an enhanced experimental system for comprehensive evaluation with multiple scenarios, seeds, and problem instances.

### 🎯 Interactive Experiment Menu

The easiest way to run experiments is through the interactive menu:

```bash
# Launch interactive experiment menu
python run_experiments.py
```

This will show you a comprehensive menu with the following options:

#### **Main Menu Options:**
1. **Single LinearizeLLM run** - Process one problem (like the original `run_linearizellm.py`)
2. **Single context experiment** - One problem with context scenarios
3. **Multiple context experiments** - Multiple problems with context scenarios  
4. **Full experimental suite** - All problems, all scenarios, all seeds
5. **Quick test** - Single instance, both scenarios, seed 1

#### **Problem Selection Options:**
1. **Process all problems** - All 20 problems in the dataset
2. **Process specific problem** - Choose one problem by name
3. **Process first N problems** - Select first N problems
4. **Process range of problems** - Select problems by index range
5. **Process multiple specific problems** - Choose specific problems by name
6. **Process random sample** - Random selection of problems

### 🎲 Context Scenarios

The experimental system supports two context scenarios:

1. **`no_context`** - Standard LinearizeLLM processing without additional context
2. **`partial_info`** - Processing with partial problem information for robustness testing

### 🌱 Seed Configuration

Experiments can be run with different seed configurations:

- **Single seed** - Run with one specific seed (e.g., seed 1)
- **Multiple seeds** - Run with selected seeds (e.g., seeds 1, 3, 5)
- **All seeds** - Run with all available seeds (1, 2, 3, 4, 5)
- **Custom seeds** - Specify your own seed values

### 🤖 LLM Model Selection

Choose between available models:
- **`gemini-2.5-flash`** - Google's Gemini 2.5 Flash model
- **`o3`** - OpenAI's O3 model

### 📊 Command Line Experiment Options

For non-interactive usage, you can also run experiments directly:

```bash
# Quick test (single instance, both scenarios, seed 1)
python run_experiments.py --quick-test

# Single context experiment
python run_experiments.py --scenario no_context --instance media_selection_nonlinear_bincon --seed 1

# Multiple context experiments with specific seeds
python run_experiments.py --scenarios no_context partial_info --instances media_selection_nonlinear_bincon blend_problem_abs --seeds 1 2 3

# Full experimental suite
python run_experiments.py --full-suite
```

### 🔬 Advanced Experimental Setup

For more advanced experimental configurations, use the dedicated experimental setup:

```bash
# Run single experiment with specific parameters
python experimental_setup.py --single-experiment --instance media_selection_nonlinear_bincon --scenario no_context --seed 1

# Run multiple experiments with different seeds
python experimental_setup.py --seeds 1 3 5

# Run experiments with specific scenarios
python experimental_setup.py --scenarios no_context partial_info

# Run experiments on specific instances
python experimental_setup.py --instances media_selection_nonlinear_bincon blend_problem_abs
```

### 📈 Results Structure

Experimental results are saved with the following structure:

```
data/context_experiment_results/
├── gemini-2.5-flash/
│   ├── no_context/
│   │   └── problem_name/
│   │       └── seed_1/
│   │           └── problem_problem_name/
│   │               └── run_timestamp/
│   │                   ├── models/
│   │                   ├── code/
│   │                   ├── steps/
│   │                   └── optimization_results.json
│   └── partial_info/
│       └── ...
└── o3/
    └── ...
```

### 🎯 Example Experiment Workflows

#### **Quick Validation:**
```bash
python run_experiments.py --quick-test
```
Runs a single problem with both scenarios using seed 1 for quick validation.

#### **Comprehensive Evaluation:**
```bash
python run_experiments.py
# Select option 4: Full experimental suite
```
Runs all 20 problems with both scenarios and all 5 seeds for comprehensive evaluation.

#### **Focused Analysis:**
```bash
python run_experiments.py
# Select option 3: Multiple context experiments
# Choose specific problems and seeds
```
Runs selected problems with specific seeds for focused analysis.

#### **Model Comparison:**
```bash
# Test with Gemini
python run_experiments.py --quick-test --llm-model gemini-2.5-flash

# Test with O3
python run_experiments.py --quick-test --llm-model o3
```
Compare performance between different LLM models.

## 📊 Results

Results are automatically saved to `data/results/` with the following structure:

```
data/results/problem_problem_name/run_timestamp/
├── models/                # LaTeX models
├── code/                  # Generated code
├── steps/                 # Intermediate results
└── optimization_results.json
```

## 🔧 Configuration

### Environment Variables

```bash
# OpenAI
export OPENAI_API_KEY="your-key"

# Google
export GOOGLE_API_KEY="your-key"
```

### Customizing Prompts

The framework uses prompt templates to guide the LLM. You can customize these prompts by editing the files in `configs/prompts/`:

- **`pattern_detection_prompt.txt`** - Instructions for identifying nonlinear patterns
- **`bilinear_pattern_prompt.txt`** - Specific guidance for bilinear term reformulation
- **`code_conversion_prompt.txt`** - Instructions for generating Gurobi Python code
- **`code_validation_prompt.txt`** - Guidelines for validating generated code

Example of customizing a prompt:
```bash
# Edit pattern detection prompt
nano configs/prompts/pattern_detection_prompt.txt

# Or use your preferred editor
code configs/prompts/pattern_detection_prompt.txt
```

**Important**: When modifying prompts, ensure they maintain the expected input/output format to avoid breaking the workflow.

### Custom Model Configuration

Create a JSON configuration file:

```json
{
  "provider": "openai",
  "model_name": "gpt-4o",
  "temperature": 0.0,
  "max_tokens": 4000
}
```

## 📄 License

This project is licensed under the MIT License - see the LICENSE file for details.

## 🙏 Acknowledgments

- Built with LangChain for LLM integration
- Uses Gurobi for optimization
- Inspired by the need for automated mathematical optimization and linearization